<?php

/**
  * SquirrelMail User Information Plugin
  * Copyright (c) 2008-2009 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage user_info
  *
  */



/**
  * Initialize this plugin (load config values)
  *
  * @return boolean FALSE if no configuration file could be loaded, TRUE otherwise
  *
  */
function user_info_init()
{

   if (!@include_once(SM_PATH . 'config/config_user_info.php'))
      if (!@include_once(SM_PATH . 'plugins/user_info/config.php'))
         if (!@include_once(SM_PATH . 'plugins/user_info/config_default.php'))
            return FALSE;

   return TRUE;

}



/**
  * Gather user data
  *
  * @param boolean $force_load When TRUE, the caller is asking
  *                            that we make sure that the user
  *                            info is loaded.  Otherwise, we
  *                            will only attempt to load it if
  *                            the configuration for preloading
  *                            is turned on (OPTIONAL; default = TRUE).
  *
  * @return array The array of user information for the
  *               current user.  The array keys and data
  *               contained therein depend on what modules
  *               are being used by this plugin, but should
  *               always at least contain "ip_address",
  *               "browser_type" and "browser_version"
  *               The array will be empty if $force_load is
  *               FALSE and pro-active loading is turned off,
  *               but could also be empty under other circumstances.
  *
  */
function get_user_info($force_load=TRUE)
{

   global $preload_user_info, $username, $user_info_modules;
   user_info_init();

   
   // testing only - make sure this is commented out!
   //
   //sqsession_unregister('user_info');


   // don't need to do anything if we are not pro-actively
   // loading the user information, or if the user info is
   // already in the session (make sure it belongs to the
   // current user by also verifying the IP address and
   // username)
   //
   $user_info = array();
   if ((!$force_load && !$preload_user_info)
    || (!sqGetGlobalVar('REMOTE_ADDR', $ip_address, SQ_SERVER)
     || (sqGetGlobalVar('user_info', $user_info, SQ_SESSION)
      && (/*!empty($username) &&*/ $user_info['username'] == $username)
      && $user_info['ip_address'] == $ip_address)))
      return $user_info;


   // get user data and put into session
   //
   $user_info = array_merge(
      array('username' => $username,
            'ip_address' => $ip_address),
      get_browser_details()
   );


   // load module data
   //
   foreach ($user_info_modules as $module)
   {
      if (!empty($module['file']) && file_exists($module['file']))
      {
         include_once($module['file']);
         if (function_exists($module['function']))
            $user_info = array_merge($user_info, $module['function']($user_info));
      }
   }


   // store in session
   //
   sqsession_register($user_info, 'user_info');
   return $user_info;

}



/**
  * Inspect client browser details
  *
  * @return array A list of key-value browser attributes,
  *               currently "browser_name" and "browser_version".
  *
  * Example User Agent strings:
  *
  * MSIE 6 in Avant shell
  * mozilla/4.0 (compatible; msie 6.0; windows nt 5.1; avant browser [avantbrowser.com]; .net clr 1.1.4322)
  *
  * Netscape 7
  * mozilla/5.0 (windows; u; windows nt 5.1; en-us; rv:1.0.2) gecko/20030208 netscape/7.02
  *
  * Mozilla 1.1 (htmlarea doesn't work)
  * mozilla/5.0 (windows; u; windows nt 5.1; en-us; rv:1.1) gecko/20020826
  *
  * Mozilla 1.4 (htmlarea does work!)
  * Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.4) Gecko/20030624             (on linux)
  * Mozilla/5.0 (Windows; U; Windows NT 5.0; en-US; rv:1.4) Gecko/20030624     (on win2k)
  * mozilla/5.0 (windows; u; windows nt 5.1; en-us; rv:1.4) gecko/20030624     (on winxp)
  *
  */
function get_browser_details()
{

   sqGetGlobalVar('HTTP_USER_AGENT', $user_agent, SQ_SERVER);
   $user_agent = strtolower($user_agent);

   if (preg_match('/opera[\s\/](\d+\.\d+)/', $user_agent, $version))
      $browser_name = 'Opera';
   else if (preg_match('/msie (\d+\.\d+)/', $user_agent, $version))
      $browser_name = 'Explorer';
   else if (preg_match('/gecko\/(\d+)/', $user_agent, $version))
      $browser_name = 'Gecko';

// Mozilla should be identified as Gecko above, and never get to this
// part... that should be OK for our purposes, but if this causes
// problems, can push this else if above the Gecko lines above
   else if (preg_match('/mozilla\/(\d+\.\d+)/', $user_agent, $version))
      $browser_name = 'Mozilla';

//echo "$user_agent<hr><hr>$browser_name<br>" . $version[1];
   return array('browser_name' => $browser_name,
                'browser_version' => $version[1]);


}



/**
  * Validate that this plugin is configured correctly
  *
  * @return boolean Whether or not there was a
  *                 configuration error for this plugin.
  *
  */
function user_info_check_configuration_do()
{

   global $user_info_modules;


   // make sure plugin is correctly configured
   //
   if (!user_info_init())
   {
      do_err('User Information plugin is not configured correctly', FALSE);
      return TRUE;
   }


   // check modules
   //
   foreach ($user_info_modules as $module)
   {
      if (!empty($module['file']) && !file_exists($module['file']))
      {
         do_err('User Information plugin has a missing module: ' . $module['file'], FALSE);
         return TRUE;
      }
      include_once($module['file']);
      if (!function_exists($module['function']))
      {
         do_err('User Information plugin has an incorrectly configured module: function ' . $module['function'] . ' not found in ' . $module['file'], FALSE);
         return TRUE;
      }
      $user_info = $module['function'](array('username' => 'test@example.org',
                                             'ip_address' => '10.10.10.10'));
      if (!is_array($user_info))
      {
         do_err('User Information plugin has an bad module: function ' . $module['function'] . ' in ' . $module['file'] . ' does not return an array', FALSE);
         return TRUE;
      }
   }

   return FALSE;

}



