<?php
/* unsafe_image_rules -- Version 0.2
 * By Robin Rainton <robin@rainton.com>
 * http://rainton.com/fun/freeware
 *
 * Always show unsafe images in HTML mail. Saves the user
 * having to always click that pesky link at the bottom of the
 * message each time.
 */

/*
 * Adminstrator options.
 * Set unsafe_image_rules_disable_all to non-zero to prevent this option
 * from being displayed to the users, and of course, to disable this
 * for any users that might have previously set it.
 */

global $unsafe_image_rules_disable_all;
$unsafe_image_rules_disable_all = 0;

/******************************************************************************
 * You should not need to edit anything below here
 */

function squirrelmail_plugin_init_unsafe_image_rules() {
  global $squirrelmail_plugin_hooks;
#  $squirrelmail_plugin_hooks['read_body_header']['unsafe_image_rules'] =
  $squirrelmail_plugin_hooks['message_body']['unsafe_image_rules'] =
    'unsafe_image_rules_main';
  $squirrelmail_plugin_hooks['optpage_register_block']['unsafe_image_rules'] =
    'unsafe_image_rules_optpage_register_block';

}

function unsafe_image_rules_main()
{
  global $has_unsafe_images, $view_unsafe_images, $data_dir, $username, $message;
  global $unsafe_image_rules_disable_all;

/*
 * Don't bother if no unsafe images in this body.
 */

  if (!$has_unsafe_images && $view_unsafe_images) return;

/*
 * Handle the case where we always show unsafe images.
 * Only if the adminstrator option is set.
 */

  if (!(isset($unsafe_image_rules_disable_all) && $unsafe_image_rules_disable_all))
  {
    $unsafe_image_rules_all = getPref($data_dir, $username, 'unsafe_image_rules_all');
    if (isset($unsafe_image_rules_all) && $unsafe_image_rules_all == 1)
    {
      $view_unsafe_images = TRUE;
      return;
    }
  }

/*
 * Look through rules for a match with this message if required.
 */

  $unsafe_image_rules_trusted = getPref($data_dir, $username,
                                        'unsafe_image_rules_trusted');
  if (isset($unsafe_image_rules_trusted) && $unsafe_image_rules_trusted &&
      is_in_unsafe_image_rules_list($message))
  {
    $view_unsafe_images = TRUE;
    return;
  }

 /*
  * Look for a match with someone in address book if that option is given.
  */

  $unsafe_image_rules_addr = getPref($data_dir, $username,
                                        'unsafe_image_rules_addr');
  if (isset($unsafe_image_rules_addr) && $unsafe_image_rules_addr &&
      is_from_sender_in_address_book($message))
  {
    $view_unsafe_images = TRUE;
    return;
  }

 /*
  * Look for a match with one of our identities if that option is given.
  */

  $unsafe_image_rules_ids = getPref($data_dir, $username,
                                    'unsafe_image_rules_ids');
  if (isset($unsafe_image_rules_ids) && $unsafe_image_rules_ids &&
      is_from_our_id($message))
  {
    $view_unsafe_images = TRUE;
    return;
  }
}

/*
 * Figure out if the message we're looking matches any details in our list.
 */

function is_in_unsafe_image_rules_list($message)
{
  $trusted = load_unsafe_image_rules();

/*
 * Loop through options seeing if this messages details match one.
 */

  for ($lp = 0; $lp < count($trusted); $lp++) {
    if ($trusted[$lp]['where'] == "To or Cc") {
      if (unsafe_image_rules_match($message, "To", $trusted[$lp]['how'], $trusted[$lp]['what'])) return TRUE;
      if (unsafe_image_rules_match($message, "Cc", $trusted[$lp]['how'], $trusted[$lp]['what'])) return TRUE;
    } else {
      if (unsafe_image_rules_match($message, $trusted[$lp]['where'],
                                   $trusted[$lp]['how'], $trusted[$lp]['what'])) return TRUE;
    }
  }
  return FALSE;
}

function unsafe_image_rules_match($message, $where, $how, $what)
{
  switch ($where) {
/*
 * To and Cc fields are arrays, so we have to join then all together
 */
    case "To" : $search = join(",", $message->header->to); break;
    case "Cc" : $search = join(",", $message->header->cc); break;
/*
 * From and Subject are standard text fields
 */
    case "From" : $search = $message->header->from; break;
    case "Subject" : $search = $message->header->subject; break;
/*
 * If we don't recognise the $where then how can it match?
 */
    default: return FALSE;
  }

  if ($how == "regexp")
  {
    if (!is_good_regexp($what)) return FALSE;
    return preg_match($what, $search);
  }
  else
  {
    return ((stristr($search, $what) == FALSE) ? FALSE : TRUE);
  }
}

/*
 * Function to figure out if sender is in address book.
 */

function is_from_sender_in_address_book($message)
{
/*
 *Initialize and addressbook
 */

  $abook = addressbook_init();
  $backend = $abook->localbackend;
  $res = $abook->list_addr($backend);

/*
 * Add addresses to list and see if any match using separate function.
 */

  $address_list = array();
  while (list($undef, $row) = each($res))
  {
    $address_list[] = $row['email'];
  }
  return is_from_address_in_list($message, $address_list);
}

/*
 * Function to figure out if sender is one of our IDs.
 */

function is_from_our_id($message)
{
  global $data_dir, $username;
  $idents = getPref($data_dir, $username, 'identities', 0);

  $address_list = array();
  for ($lp = 0; $lp < $idents; $lp++)
  {
    $address_list[] = getPref($data_dir, $username, 'email_address' . ($lp > 0 ? "$lp" : ""));
  }
  return is_from_address_in_list($message, $address_list);
}

/*
 * Given the message and a list of addresses, see if it's from
 * one of those addresses.
 */

function is_from_address_in_list($message, $address_list)
{
/*
 * Don't bother if null list is given.
 */

  if (array_count_values($address_list) < 1) return FALSE;

/*
 * Parse important parts out of from and replyto
 * We do this to remove the name (stuff outside <>) and anything but
 * the actual domain name from the address.
 */

  $regex = '/[<|\s]*([\w|\d|\-|\.]+@)([a-z]+[\w|0-9|\-]*\.)*([a-z]{2,7})[>|\s]*$/';

  $num_matches = preg_match_all($regex, $message->header->from, $matches);
  if ($num_matches == 1) $from = $matches[1][0] . $matches[2][0] . $matches[3][0];
  $num_matches = preg_match_all($regex, $message->header->replyto, $matches);
  if ($num_matches == 1) $replyto = $matches[1][0] . $matches[2][0] . $matches[3][0];

/*
 * Check all addresses - parsing them too on the way.
 */

  reset($address_list);
  $address = current($address_list);
  do
  {
    $num_matches = preg_match_all($regex, $address, $matches);
    if ($num_matches == 1) $address = $matches[1][0] . $matches[2][0] . $matches[3][0];
    if (isset($from) && $address == $from) return TRUE;
    if (isset($replyto) && $address == $replyto) return TRUE;
  } while ($address = next($address_list));
  return FALSE;
}

/*
 * Register our options section.
 */

function unsafe_image_rules_optpage_register_block() {
  global $optpage_blocks;

  $optpage_blocks[] = array(
      'name' => _("Unsafe Image Rules"),
      'url'  => '../plugins/unsafe_image_rules/options.php',
      'desc' => _("Set up rules about how unsafe images in HTML messages are handled."),
      'js'   => false
  );
}

/*
 * Function to load trusted message details
 */

function load_unsafe_image_rules()
{
  global $data_dir, $username;

  $out = array();
  for ($lp=0; $opt = getPref($data_dir, $username, 'unsafe_image_rules' . $lp); $lp++) {
    $ary = explode(',', $opt);
    $out[$lp]['where'] = array_shift($ary);
/*
 * If $where begins with R means this is a regexp
 * Done this way for backward compatibility
 */

    if (substr($out[$lp]['where'], 0, 1) == "R")
    {
      $out[$lp]['how'] = "regexp";
      $out[$lp]['where'] = substr($out[$lp]['where'], 1);
    }
    else
    {
      $out[$lp]['how'] = "contains";
    }
/*
 * Whatever is left is the string. If it contained commas then it
 * get's split, so put it back together.
 */
    $out[$lp]['what'] = join(",", $ary);
  }
  return $out;
}

/*
 * Function to figure out if the regular expression passed is good or not.
 * This is a bit simple - I'm open to suggestions as to how to better it.
 */

function set_bad_unsafe_image_regexp($errno, $errstr, $errfile, $errline) {
  global $bad_unsafe_image_regexp;
  $bad_unsafe_image_regexp = 1;
}

function is_good_regexp($regexp) {
  global $bad_unsafe_image_regexp;
  $bad_unsafe_image_regexp = 0;

  $old_handler = set_error_handler("set_bad_unsafe_image_regexp");
  preg_match($regexp, "");
  set_error_handler($old_handler);

  return ($bad_unsafe_image_regexp == 1) ? FALSE : TRUE;
}

?>
