<?php

/**
  * SquirrelMail Shared Calendar Plugin
  * Copyright (C) 2004-2005 Paul Lesneiwski <pdontthink@angrynerds.com>
  * This program is licensed under GPL. See COPYING for details
  *
  */


/**
  * Calendar class
  *
  */
class Calendar
{

   var $id;
   var $sequence;
   var $dom;
   var $prodID;
   var $type;
   var $name;
   var $owners;
   var $readable_users;
   var $writeable_users;
   var $createdOn;
   var $lastUpdatedOn;
   var $createdBy;
   var $lastUpdatedBy;
   var $holidays = array();
   var $events;
   var $allEvents;
   var $unknownAttributes = array();

   var $isExternalSource = FALSE;

   var $currentMonthEventCache = 0;



   /**
     * Calendar constructor
     *
     * @param mixed $id The ID of this calendar (optional; ID is auto-
     *                  generated if not given - NOTE that this is never
     *                  a correct ID for (default) private type calendars!).
     *                  May be specified as a string or a Property object.
     * @param mixed $sequence The edit sequence ID for this calendar
     *                        May be specified as an int or a Property object.
     * @param mixed $dom The domain this calendar belongs under
     *                   May be specified as a string or a Property object.
     * @param mixed $prodID The product ID of the creating calendar system
     *                       May be specified as a string or a Property object.
     * @param mixed $type The type of this calendar, which should
     *                    correspond to the calendar type constants
     *                    defined in {@link constants.php}
     *                    May be specified as a string or a Property object.
     * @param mixed $name The name of this calendar
     *                    May be specified as a string or a Property object.
     * @param mixed $createdBy The name of the user who created this calendar
     *                         May be specified as a string or a Property object.
     * @param mixed $createdOn The date/time this calendar was created (optional;
     *                         defaults to today's date)
     *                         May be specified as a UTC-formatted timestamp 
     *                         string or a Property object.
     * @param mixed $lastUpdatedBy The name of the user who last updated this calendar
     *                             May be specified as a string or a Property object.
     * @param mixed $lastUpdatedOn The date/time this calendar was last updated
     *                             May be specified as a UTC-formatted timestamp 
     *                             string or a Property object.
     * @param mixed $owners The users who share ownership of this calendar
     *                      May be specified as an array or a Property object.
     * @param mixed $readable_users The users who have read access to this calendar
     *                              May be specified as an array or a Property object.
     * @param mixed $writeable_users The users who have write access to this calendar
     *                               May be specified as an array or a Property object.
     * @param array $unknownAttributes Extra unknown attributes in an 
     *                                 array keyed by attribute name, although
     *                                 the value MUST be the full iCal line
     *                                 describing the property, INCLUDING its
     *                                 name.  These properties are often
     *                                 derived from custom attributes from an
     *                                 imported iCal file
     * @param string $fallbackName An additional name to be used if 
     *                             none is given above or in any of the
     *                             calendar's extra attributes (optional)
     * @param string $uploadFilename A flag that indicates the source of this
     *                               calendar; if given, it represents the 
     *                               filename from which it was uploaded,
     *                               in which case the ID (if not
     *                               given) will be constructed differently,
     *                               so that it is not create-time based.
     *                               (optional; default = empty string)
     *
     * Note that by default, if no owner, readable or writeable user is 
     * specified, the current user is given read permission only.
     *
     * Note that by default, if no calendar type is specified, it is assumed
     * to be a PUBLIC calendar!  
     *
     */
   function Calendar($id='', $sequence=0, $dom='', $prodID='', $type='', $name='', 
                     $createdBy='', $createdOn='', $lastUpdatedBy='', $lastUpdatedOn='', 
                     $owners=array(), $readable_users=array(), $writeable_users=array(),
                     $unknownAttributes=array(), $fallbackName='', $uploadFilename='')
   {

      if (is_object($id) && strtolower(get_class($id)) == 'property')
         $this->id                = $id;
      else
         $this->id                = new Property('X-SQ-CALID', $id);

      if (is_object($sequence) && strtolower(get_class($sequence)) == 'property')
         $this->sequence          = $sequence;
      else
         $this->sequence          = new Property('X-SQ-CALSEQUENCE', $sequence);

      if (is_object($dom) && strtolower(get_class($dom)) == 'property')
         $this->dom               = $dom;
      else
         $this->dom               = new Property('X-SQ-CALDOMAIN', 
                                    strtr($dom, '@|_-.:/ \\', '________'));

      if (is_object($prodID) && strtolower(get_class($prodID)) == 'property')
         $this->prodID            = $prodID;
      else
         $this->prodID            = new Property('PRODID', $prodID);

      if (is_object($type) && strtolower(get_class($type)) == 'property')
         $this->type              = $type;
      else
         $this->type              = new Property('X-SQ-CALTYPE', $type);

      if (is_object($name) && strtolower(get_class($name)) == 'property')
         $this->name              = $name;
      else
         $this->name              = new Property('X-SQ-CALNAME', $name);

      if (is_object($createdBy) && strtolower(get_class($createdBy)) == 'property')
         $this->createdBy         = $createdBy;
      else
         $this->createdBy         = new Property('X-SQ-CALCREATOR', $createdBy);

      if (is_object($createdOn) && strtolower(get_class($createdOn)) == 'property')
         $this->createdOn         = $createdOn;
      else
         $this->createdOn         = new Property('X-SQ-CALCREATED', 
                                 (empty($createdOn) ? gmdate('Ymd\THis\Z') : $createdOn),
                                 array(), SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_UTC);

      if (is_object($lastUpdatedBy) && strtolower(get_class($lastUpdatedBy)) == 'property')
         $this->lastUpdatedBy     = $lastUpdatedBy;
      else
         $this->lastUpdatedBy     = new Property('X-SQ-CALLASTUPDATOR', 
                                 (empty($lastUpdatedBy) ? $this->createdBy() : $lastUpdatedBy));

      if (is_object($lastUpdatedOn) && strtolower(get_class($lastUpdatedOn)) == 'property')
         $this->lastUpdatedOn     = $lastUpdatedOn;
      else
         $this->lastUpdatedOn     = new Property('X-SQ-CALLAST-MODIFIED', 
                    (empty($lastUpdatedOn) ? $this->createdOn->getRawValue() : $lastUpdatedOn),
                    array(), SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_UTC);

      if (is_object($owners) && strtolower(get_class($owners)) == 'property')
         $this->owners            = $owners;
      else
         $this->owners            = new Property('X-SQ-CALOWNERS', $owners);

      if (is_object($readable_users) && strtolower(get_class($readable_users)) == 'property')
         $this->readable_users    = $readable_users;
      else
         $this->readable_users    = new Property('X-SQ-CALREADABLEUSERS', $readable_users);

      if (is_object($writeable_users) && strtolower(get_class($writeable_users)) == 'property')
         $this->writeable_users   = $writeable_users;
      else
         $this->writeable_users   = new Property('X-SQ-CALWRITEABLEUSERS', $writeable_users);

      $this->unknownAttributes = $unknownAttributes;



      $sm_cal_prodid = str_replace('###VERSION###', calendar_version(), SM_CAL_PRODID);
      global $domain, $username, $useDomainInCalID;



      // insert default values if not given above
      //
      // note that user is only given read access by default;
      // caller must explicitly give any more than that
      //
      $o = $this->getOwners();
      $w = $this->getWriteableUsers();
      $r = $this->getReadableUsers();
      if (empty($o) && empty($w) && empty($r))
         $this->readable_users->setValue(array($username));

      $c = $this->createdBy();
      if (empty($c))
      $this->setCreator($username);


      $p = $this->getProductID();
      if (empty($p))
         $this->prodID->setValue($sm_cal_prodid);

      $d = $this->dom->getValue();
      if (empty($d))
         $this->dom->setValue(strtr($domain, '@|_-.:/ \\', '________'));

      $i = $this->getID();
      if (empty($i))
      {
         //Note: Apple uses this field for cal ID: X-WR-RELCALID
         if (!empty($unknownAttributes['X-WR-RELCALID']))
         {
            $this->id = Property::extractICalProperty($unknownAttributes['X-WR-RELCALID']);
            unset($unknownAttributes['X-WR-RELCALID']);
         }


         // uploaded calendars get an ID that is constructed w/out
         // the default of having the create time in it
         //
         else if ($uploadFilename)
         {
            // if calendar itself already has createdOn date, 
            // use that
            //
            if (is_object($createdOn) && strtolower(get_class($createdOn)) == 'property')
               $this->id->setValue('sm_cal_' . gmdate('Ymd\THis\Z', $this->createdOn())
                                 . ($useDomainInCalID ? '_' . $this->getDomain() : ''));


            // otherwise, fudge it
            //
            else 
            {
               // try to chop off extension/incrememtal file numbers
               //
               if (strlen($uploadFilename) > 6)
                  $uploadFilename = substr($uploadFilename, 0, strlen($uploadFilename) - 5);
               $this->id->setValue(strtr('sm_uploaded_cal_' . $uploadFilename . '__' . $username 
                                 . ($useDomainInCalID ? '__' . $domain : ''), 
                                   '@|_-.:/\ ', '________'));
            }
         }


         else
//Note: this is not a correct ID for (default) private cals!  caller should not
//      rely on auto-ID generation for private calendars!
            $this->id->setValue('sm_cal_' . gmdate('Ymd\THis\Z') 
                              . ($useDomainInCalID ? '_' . $this->getDomain() : ''));
      }
      
      $n = $this->getName();
      if (empty($n))
      {
         //Note: Apple uses this field for cal name: X-WR-CALNAME
         if (!empty($unknownAttributes['X-WR-CALNAME']))
         {
            $this->name = Property::extractICalProperty($unknownAttributes['X-WR-CALNAME']);
            unset($unknownAttributes['X-WR-CALNAME']);
         }
         else if (!empty($fallbackName))
            $this->name->setValue($fallbackName);
         else
//TODO: there has to be a better way to resolve a calendar name...!
            $this->name->setValue($this->getProductID());
      }
      
//TODO: yikes, is this too permissive?  should we go the other way, assume private?
//      although a user only has ONE private calendar, so that might be a bad idea
      $t = $this->getCalendarType();
      if (empty($t))
         $this->type->setValue(SM_CAL_TYPE_PUBLIC);
      
//sm_print_r($this);
   }



// ---------- PRIVATE ----------



   /**
     * Gathers calendar event data for the given month.
     *
     * NOTE that any events already in this object with
     * the same event IDs as those being retrieved will
     * be overwritten - the caller must save them first 
     * to avoid loss.
     *
     * @param int $year The year of the month to be retrieved
     * @param int $month The month to be retrieved
     * @param string $user The user for which events are being retrieved
     *
     * @access private
     *
     */
   function retrieveEventsForMonth($year, $month, $user)
   {

      // if source is external, we already have everything
      //
      if ($this->isExternal()) return;


      if (!is_array($this->events))
         $this->events = array();


      if (!isset($this->events['onetime']) 
       || !is_array($this->events['onetime']))
         $this->events['onetime'] = array();


      // get events from calendar backend
      // and merge them with any other events
      // that we might already have
      // 
      $this->events['onetime'] 
         = array_merge($this->events['onetime'], 
                       get_events_for_month($this->getID(), $year, $month, $user));

   }



   /**
     * Gathers all calendar event, holiday and other data for 
     * all time periods.
     *
     * Most useful for exporting all events, as normal operation
     * will be faster if only the needed events are in memory.
     *
     * In fact, events pulled in this function are stored in a 
     * separate array from what is normally used; the array
     * is not categorized between one-time, recurring, holiday
     * events, etc.
     *
     * NOTE that any events already in this object with
     * the same event IDs as those being retrieved will
     * be overwritten - the caller must save them first 
     * to avoid loss.
     *
     * @param string $user The user for which events are being retrieved
     *
     * @access private
     *
     */
   function retrieveAllEvents($user)
   {

      // if source is external, we already have everything
      //
      if ($this->isExternal()) return;


      if (!is_array($this->allEvents))
         $this->allEvents = array();


      // get events from calendar backend
      // and merge them with any other events
      // that we might already have
      // 
      $this->allEvents = array_merge($this->allEvents, get_all_events($this->getID(), $user));

   }



   /**
     * Gathers recurring calendar event data 
     *
     * NOTE that any recurring events already in this object 
     * will be removed first - the caller must save them 
     * first to avoid loss.
     *
     * @param string $user The user for which events are being retrieved
     *
     * @access private
     *
     */
   function retrieveRecurringEvents($user)
   {

      // if source is external, we already have everything
      //
      if ($this->isExternal()) return;


      if (!is_array($this->events))
         $this->events = array();


      if (!isset($this->events['recurring']) 
       || !is_array($this->events['recurring']))
         $this->events['recurring'] = array();


      // get events from calendar backend
      //
      $this->events['recurring'] = get_recurring_events($this->getID(), $user);

   }



   /**
     * Gathers calendar holiday data 
     *
     * NOTE that any holidays already in this object 
     * will be removed first - the caller must save them 
     * first to avoid loss.
     *
     * @param string $user The user for which holidays are being retrieved
     *
     * @access private
     *
     */
   function retrieveHolidays($user)
   {

      // if source is external, we already have everything
      //
      if ($this->isExternal()) return;


      if (!is_array($this->holidays))
         $this->holidays = array();


      // get holidays from calendar backend
      //
      $this->holidays = get_calendar_holidays($this->getID(), $user);

   }



// ---------- PUBLIC ----------



   /**
     * Determines if the source of this calendar is externally located
     *
     * @return boolean TRUE if calendar source is an external URI, FALSE otherwise
     *
     * @access public
     *
     */
   function isExternal()
   {
      return $this->isExternalSource;
   }



   /**
     * Set As External
     * 
     * Tells us if this calendar's source was from the outside world
     *
     * @param boolean $external TRUE if the source is external, FALSE otherwise
     *
     * @access public
     *
     */
   function setExternal($external)
   {
      $this->isExternalSource = $external;
   }



   /**
     * Get Calendar ID
     *
     * @return string This calendar's internal ID
     *
     * @access public
     *
     */
   function getID()
   {

      // external calendars always have this prepended to the ID...
      //
      if ($this->isExternal())
         return 'SM_EXTERNAL' . $this->id->getValue();
      else
         return $this->id->getValue();
   }



   /**
     * Set Calendar ID
     *
     * @param string $id The new ID to be assigned to this calendar
     *
     * @access public
     *
     */
   function setID($id)
   {

      // if already prefixed with external indicator, strip that off
      //
      if (strpos($id, 'SM_EXTERNAL') === 0)
         $id = substr($id, 11);

      $this->id->setValue($id);
   }



   /**
     * Increment Sequence Number
     *
     * @access public
     *
     */
   function incrementSequence()
   {
      $this->sequence->setValue($this->sequence->getValue() + 1);
   }



   /**
     * Set Calendar Type
     *
     * @param string $type The new type to assign to this calendar,
     *                     which should correspond to the calendar type
     *                     constants defined in {@link constants.php}
     *
     * @access public
     *
     */
   function setType($type)
   {
      $this->type->setValue($type);
   }



   /**
     * Get Calendar Type
     *
     * @return string This calendar's type, which will
     *                correspond to the calendar type
     *                constants defined in {@link constants.php}
     *
     * @access public
     *
     */
   function getCalendarType()
   {
      return $this->type->getValue();
   }



   /**
     * Get Calendar Name
     *
     * @return string This calendar's name
     *
     * @access public
     *
     */
   function getName()
   {

      // need to translate (default) personal calendar name 
      // here since we can't always count on the correct
      // language having been used when it was created
      //
      global $username, $domain;
      if ($this->getCalendarType() == SM_CAL_TYPE_PERSONAL 
       && get_personal_cal_id($username, $domain, TRUE) == $this->getID())
      {
         global $username;
         return sprintf(_("Personal Calendar for %s"), $username);
      }

      return $this->name->getValue();
   }



   /**
     * Set Calendar Name
     *
     * @param string $name The new name to be assigned to this calendar
     *
     * @access public
     *
     */
   function setName($name)
   {
      $this->name->setValue($name);
   }



   /**
     * Get Calendar Domain
     *
     * @return string This calendar's domain
     *
     * @access public
     *
     */
   function getDomain()
   {
      return $this->dom->getValue();
   }



   /**
     * Get Product ID
     *
     * @return string The product ID for the product that created this calendar
     *
     * @access public
     *
     */
   function getProductID()
   {
      return $this->prodID->getValue();
   }



   /**
     * Get Creator Name
     *
     * @return string This event's creator
     *
     * @access public
     *
     */
   function createdBy()
   {
      return $this->createdBy->getValue();
   }



   /**
     * Set Username of User Who Created This Calendar
     *
     * @param string $user The name of the user who created this calendar
     *
     * @access public
     *
     */
   function setCreator($user)
   {
      return $this->createdBy->setValue($user);
   }



   /**
     * Get Creation Date
     *
     * @return timestamp This calendar's creation date
     *
     * @access public
     *
     */
   function createdOn()
   {
      return $this->createdOn->getValue();
   }



   /**
     * Get User That Last Updated Calendar
     *
     * @return string This calendar's last editor
     *
     * @access public
     *
     */
   function lastUpdatedBy()
   {
      return $this->lastUpdatedBy->getValue();
   }



   /**
     * Set Username of User Who Last Updated This Calendar
     *
     * @param string $user The name of the user updating this calendar
     *
     * @access public
     *
     */
   function setLastUpdator($user)
   {
      return $this->lastUpdatedBy->setValue($user);
   }



   /**
     * Get Last Update Date
     *
     * @return timestamp The date of this calendar's last update
     *
     * @access public
     *
     */
   function lastUpdatedOn()
   {
      return $this->lastUpdatedOn->getValue();
   }



   /**
     * Set Last Update Date
     *
     * @param string $timestamp The date this calendar was last updated
     *                          (should be a UTC-formatted date/time string)
     *
     * @access public
     *
     */
   function setLastUpdateDate($timestamp)
   {
      return $this->lastUpdatedOn->setValue($timestamp);
   }



   /**
     * Get Calendar Owners
     *
     * @return array An array listing all calendar owners
     *
     * @access public
     *
     */
   function getOwners()
   {
      $o = $this->owners->getValue();
      if (empty($o))
         return array();
      else if (is_string($o))
         return array($o);
      else
         return $o;
   }



   /**
     * Get Readable Users
     *
     * @return array An array listing all users
     *               who have read access to this calendar
     *
     * @access public
     *
     */
   function getReadableUsers()
   {
      $r = $this->readable_users->getValue();
      if (empty($r))
         return array();
      else if (is_string($r))
         return array($r);
      else
         return $r;
   }



   /**
     * Get Writeable Users
     *
     * @return array An array listing all users
     *               who have write access to this calendar
     *
     * @access public
     *
     */
   function getWriteableUsers()
   {
      $w = $this->writeable_users->getValue();
      if (empty($w))
         return array();
      else if (is_string($w))
         return array($w);
      else
         return $w;
   }



   /**
     * Get Unknown Attributes
     *
     * @return array An array of all unknown attributes
     *
     * @access public
     *
     */
   function getUnknownAttributes()
   {
      return $this->unknownAttributes;
   }



   /**
     * Add a series of new events to this calendar.  Events
     * are merged with those already in the calendar.
     *
     * Note: can only accomodate up to 1,000,000 events in one 
     * calendar when $forceAddAll is TRUE.
     *
     * @param array $newEvents An array of event arrays, keyed 
     *                         by string sindicating event 
     *                         type ("onetime", "recurring", etc)
     * @param string $oldParentCalID The ID of the calendar that
     *                               has owned the events up to now
     * @param boolean $holdNewEvents If TRUE, will NOT save new events
     *                               (updated events are ALWAYS saved)
     *                               to the backend, and will instead 
     *                               just keep the events in memory 
     *                               inside of the calendar object for 
     *                               saving at a later time. (optional; 
     *                               default = FALSE)
     * @param boolean $forceAddAll If TRUE, will add all events,
     *                             even when event IDs conflict
     *                             (when events would otherwise be
     *                             synched) (conflicts are avoided 
     *                             by adding random numbers to event 
     *                             IDs) (optional; default = FALSE)
     *
     * @access public
     *
     */
   function addEvents($newEvents, $oldParentCalID, 
                      $holdNewEvents=FALSE, $forceAddAll=FALSE)
   {

      if (empty($newEvents) || !is_array($newEvents)) return;


      if (!isset($this->events) 
       || !is_array($this->events))
         $this->events = array();
      if (!isset($this->events['onetime']) 
       || !is_array($this->events['onetime']))
         $this->events['onetime'] = array();
      if (!isset($this->events['recurring']) 
       || !is_array($this->events['recurring']))
         $this->events['recurring'] = array();


      $allEvents = array();
//TODO: what about other event types???  todos? holidays?
      if (isset($newEvents['onetime']) && is_array($newEvents['onetime']))
         $allEvents = array_merge($allEvents, $newEvents['onetime']);

      if (isset($newEvents['recurring']) && is_array($newEvents['recurring']))
         $allEvents = array_merge($allEvents, $newEvents['recurring']);


      // 
      //
      foreach ($allEvents as $event)
      {
         
         // attempt to get same event from backend
         //
         $event2 = get_event($this->getID(), $event->getID());
         if ($event2 === FALSE) $event2 = $this->getEvent($event->getID());


         // if we have a duplicate event ID but are forced
         // to add it anyway, create unique ID and then add it
         //
         if ($forceAddAll) 
         {
            sq_mt_randomize();
            $id = $event->getID();
            while ($event2 !== FALSE)
            {
               $id = $event->getID() . mt_rand(1, 1000000);
               $event2 = get_event($this->getID(), $id);
               if ($event2 === FALSE) $event2 = $this->getEvent($id);
            }
            $event->setID($id);

         }


         // add event to this cal if not found
         //
         if ($event2 === FALSE)
         {

            $event->removeParentCalendar($oldParentCalID);
            $event->addParentCalendar($this->getID());
//TODO: see notes below about permissions not getting preserved
            $event->resetPermissionsToParent(array($this));
            $event->removeParentCalendar($oldParentCalID);


            // save event in correct place in memory
            // if we are not saving to the backend
            //
            if ($holdNewEvents)
            {
//TODO: what about other event types???  todos? holidays?
               if ($event->isRecurring())
                  $this->events['recurring'][] = $event;
               else if ($event->isOneTime())
                  $this->events['onetime'][] = $event;
            }
            else
               create_event($this->getID(), $event);

         }


         // synch: if date of new event is newer,
         // replace event, otherwise, skip it
         //
         else if ($event->lastUpdatedOn() > $event2->lastUpdatedOn())
         {
            $event->removeParentCalendar($oldParentCalID);
            $event->addParentCalendar($this->getID());
//TODO: uploaded events may at some point have different permissions
//      that we want to obey, but for now, there should not be a
//      difference between cal and evt perms.  and I am too lazy to
//      check if an upload will somehow munge any perms we may have
//      had (because it gets uploaded to a temporary new calendar 
//      that is different from this one)
            $event->resetPermissionsToParent(array($this));
            update_event($this->getID(), $event);
         }

      }

   }



   /**
     * Dump all events currently cached in this object.  Note
     * that this may not in all cases correspond to ALL the 
     * events on en entire calendar (although it will in the
     * case of an uploaded/external calendar or after calling
     * $this->retrieveAllEvents()).
     *
     * @return array An array of event arrays, keyed by strings
     *               indicating event type ("onetime", 
     *               "recurring", etc)
     *
     * @access public
     *
     */
   function getEvents()
   {
      return $this->events;
   }



   /**
     * Get Event
     *
     * Looks for (and returns) an event amongst whatever events
     * are currently loaded into this calendar.  FALSE is returned
     * when the event is not found -- note that sometimes this may 
     * not mean that there is no such event in this calendar!
     *
     * @param string $eventID The ID of the event to search for
     *
     * @return object An event object if the event was found
     *                in memory in the current object; FALSE otherwise
     *
     * @access public
     *
     */
   function getEvent($eventID)
   {
//TODO: what about other event types???  todos? holidays?

      // look amongst one time events 
      //
      foreach ($this->events['onetime'] as $event)
         if ($event->getID() == $eventID)
            return $event;


      // look amongst recurring events 
      //
      foreach ($this->events['recurring'] as $event)
         if ($event->getID() == $eventID)
            return $event;


      return FALSE;
   }



   /**
     * Determines if this calendar comes before
     * or after the given calendar, for use with
     * sorting calendars.
     *
     * @param object $otherCalendar The calendar to 
     *                              compare to this
     *                              one.
     *
     * @return int -1 if this calendar comes first,
     *             1 if this calendar comes second,
     *             or 0 if the calendars should be
     *             considered to be "equal".
     *
     * @access public
     *
     */
   function compareTo($otherCalendar)
   {

      return strcasecmp($this->getName(), $otherCalendar->getName());

   }



   /**
     * Returns any events that occur on the given day
     *
     * @param int $year The year of the day whose events are being returned
     * @param int $month The month of the day whose events are being returned
     * @param int $day The day whose events are being returned
     * @param string $user The user for which events are being returned
     *
     * @return array A list of all the events that occur today, sorted
     *               chronologically (array of Event objects)
     *
     * @access public
     *
     */
   function getEventsForDay($year, $month, $day, $user)
   {

      // we only cache one month's worth of events at once
      //
      if (!$this->isExternal() && $this->currentMonthEventCache != $month)
         unset($this->events['onetime']);


      // get events if needed
      //
      if (!isset($this->events['onetime']) 
       || !is_array($this->events['onetime']))
         $this->retrieveEventsForMonth($year, $month, $user);
      if (!isset($this->events['recurring']) 
       || !is_array($this->events['recurring'])) 
      {
         $this->retrieveRecurringEvents($user);

         // force events to cache all recurrences, which speeds
         // responsiveness tremendously
         //
         // Cache thru the 7th of January, two years ahead,
         // because the most extreme case is year view where
         // we start by getting the last few days of the previous
         // year.  For month and day view, this is quite a bit
         // of overkill; I don't think it is much of a hit, but
         // it *could* be, especially with many complicated RRULEs.
         // So if needed, it might be possible to make a
         // differentiation for that here and only cache as 
         // much as is really needed.
         //
         $cacheDate = mktime(12, 0, 0, 1, 7, $year + 2);
         foreach ($this->events['recurring'] as $index => $event)
            $this->events['recurring'][$index]->forceEventOccurrenceCache($cacheDate);
      }




      // loop through all events, looking for ones for this day
      //
      $returnArray = array();
      foreach ($this->events['onetime'] as $event)
         if ($event->occursOnDay($year, $month, $day))
            $returnArray[] = $event;
      foreach ($this->events['recurring'] as $event)
         if ($event->occursOnDay($year, $month, $day))
            $returnArray[] = $event;


      // resort return array by starting event time
      //
      usort($returnArray, 'sortEventsByTime');


      $this->currentMonthEventCache = $month;

      return $returnArray;

   }



   /**
     * Displays the requested month 
     *
     * @param int $year The year of the month to be shown
     * @param int $month The month to be shown
     * @param int $startDay The day of the week for first display column
     * @param string $user The user for which events are being displayed
     * @param string $viewType The type of month view to be shown, which
     *                         should correspond to the MONTH view mode
     *                         constants defined in {@link constants.php}
     *                         (optional; default is normal month view mode)
     *
     * @access public
     *
     */
   function showMonth($year, $month, $startDay, $user, $viewType=SM_CAL_VIEW_MODE_MONTH)
   {

      // display events
      //
      if ($viewType == SM_CAL_VIEW_MODE_MONTH || $viewType == SM_CAL_VIEW_MODE_ALL_MONTHS
       || $viewType == SM_CAL_VIEW_MODE_MONTH_MINIATURE)
      {
         include_once(SM_PATH . 'plugins/calendar/interface/month.php');
         showCalendarMonth($this, $year, $month, $startDay, $user, $viewType);
      }
      else
      {
         global $color;
         plain_error_message('ERROR IN CALENDAR CLASS (showMonth): Invalid view mode', $color);
         exit;
      }

   }



   /**
     * Displays the requested year 
     *
     * @param int $year The year to be shown
     * @param string $user The user for which events are being displayed
     *
     * @access public
     *
     */
   function showYear($year, $user)
   {

//LEFT OFF HERE
//LEFT OFF HERE

   }



   /**
     * Displays all months for the requested year 
     *
     * @param int $year The year to be shown
     * @param string $user The user for which events are being displayed
     * @param int $startDay The day of the week for first display column
     *
     * @access public
     *
     */
   function showAllMonths($year, $user, $startDay)
   {

      for ($i = 1; $i < 13; $i++)
      {
         $this->showMonth($year, $i, $startDay, $user, SM_CAL_VIEW_MODE_ALL_MONTHS);
      }

   }



   /**
     * Displays the requested day
     *
     * @param int $year The year of the day to be shown
     * @param int $month The month of the day to be shown
     * @param int $day The day to be shown
     * @param string $user The user for which events are being displayed
     * @param boolean $allDay Indicates if the whole day should be
     *                        displayed, from midnight to midnight.  If
     *                        FALSE, a limited number of hours are
     *                        displayed per the configuration file.
     *
     * @access public
     *
     */
   function showDay($year, $month, $day, $user, $allDay=FALSE)
   {

      // display events
      //
      include_once(SM_PATH . 'plugins/calendar/interface/day.php');
      showCalendarDay($this, $year, $month, $day, $user, $allDay);

   }



   /**
     * Returns any holidays that occur on the given day
     *
     * @param int $year The year of the day whose holidays are being returned
     * @param int $month The month of the day whose holidays are being returned
     * @param int $day The day whose holidays are being returned
     * @param string $user The user for which holidays are being returned
     *
     * @return array A list of all the holidays that occur today, sorted
     *               chronologically (array of Event objects)
     *
     * @access public
     *
     */
   function getHolidaysForDay($year, $month, $day, $user)
   {

      // do we already have holidays in memory?
      // then we don't need to load them from the backend
      //
      if (empty($this->holidays) || !is_array($this->holidays) || sizeof($this->holidays) == 0)
      {
         $this->retrieveHolidays($user);

         // force events to cache all recurrences, which speeds
         // responsiveness tremendously
         //
         // Cache thru the 7th of January, two years ahead,
         // because the most extreme case is year view where
         // we start by getting the last few days of the previous
         // year.  For month and day view, this is quite a bit
         // of overkill; I don't think it is much of a hit, but
         // it *could* be, especially with many complicated RRULEs.
         // So if needed, it might be possible to make a
         // differentiation for that here and only cache as 
         // much as is really needed.
         //
         $cacheDate = mktime(12, 0, 0, 1, 7, $year + 2);
         foreach ($this->holidays as $index => $holiday)
            $this->holidays[$index]->forceEventOccurrenceCache($cacheDate);

      }


      // loop through all holidays, looking for ones for this day
      //
      $returnArray = array();
      foreach ($this->holidays as $holiday)
         if ($holiday->occursOnDay($year, $month, $day))
            $returnArray[] = $holiday;


      return $returnArray;

   }



   /**
     * Removes a user from this calendar, from the list
     * of owners, readable users, and writeable users
     *
     * @param string $user The user to be removed
     *
     * @access public
     *
     */
   function remove_user($user)
   {

      $this->owners->setValue(array_diff($this->getOwners(), array($user)));
      $this->readable_users->setValue(array_diff($this->getReadableUsers(), array($user)));
      $this->writeable_users->setValue(array_diff($this->getWriteableUsers(), array($user)));

   }



   /** 
     * Adds a new user to this calendar
     *
     * @param string $user The user name being added
     * @param string $accessLevel The access level being
     *                            granted to the new user,
     *                            which should correspond
     *                            to the calendar access constants
     *                            defined in {@link constants.php}
     * @access public
     *
     */
   function add_user($user, $accessLevel)
   {

      if ($accessLevel == SM_CAL_ACCESS_LEVEL_OWNER)
         $this->owners->setValue(array_merge($this->getOwners(), array($user)));

      else if ($accessLevel == SM_CAL_ACCESS_LEVEL_READ)
         $this->readable_users->setValue(array_merge($this->getReadableUsers(), array($user)));

      else if ($accessLevel == SM_CAL_ACCESS_LEVEL_WRITE)
         $this->writeable_users->setValue(array_merge($this->getWriteableUsers(), array($user)));

   }



   /**
     * Determines if the given user is an owner of this calendar
     *
     * @param string $user The user to inspect for ownership
     *
     * @return boolean TRUE if the user is an owner of this calendar, FALSE otherwise
     *
     * @access public
     *
     */
   function isOwner($user)
   {

      // can't just test to see if user is in owner array, since
      // we allow for wildcards in owner names
      //
      foreach ($this->getOwners() as $owner)
         if (preg_match('/^' . str_replace(array('?', '*'), array('\w{1}', '.*?'),
                    strtoupper($owner)) . '$/', strtoupper($user)))
            return TRUE;


      return FALSE;

   }



   /**
     * Determines if the given user has read access to this calendar
     *
     * @param string $user The user to inspect for read permission
     *
     * @return boolean TRUE if the user has read access to this calendar, FALSE otherwise
     *
     * @access public
     *
     */
   function canRead($user)
   {

      // public calendars are always readable to anyone
      // (TODO: in the future, we may want to limit this
      // per domain)
      //
      if ($this->getCalendarType() == SM_CAL_TYPE_PUBLIC)
         return TRUE;


      // can't just test to see if user is in readable_users array, since
      // we allow for wildcards in user names
      //
      foreach ($this->getReadableUsers() as $readUser)
         if (preg_match('/^' . str_replace(array('?', '*'), array('\w{1}', '.*?'),
                    strtoupper($readUser)) . '$/', strtoupper($user)))
            return TRUE;


      return FALSE;

   }



   /**
     * Determines if the given user has write access to this calendar
     *
     * @param string $user The user to inspect for write permission
     *
     * @return boolean TRUE if the user has write access to this calendar, FALSE otherwise
     *
     * @access public
     *
     */
   function canWrite($user)
   {

      // can't just test to see if user is in writeable_users array, since
      // we allow for wildcards in user names
      //
      foreach ($this->getWriteableUsers() as $writeUser)
         if (preg_match('/^' . str_replace(array('?', '*'), array('\w{1}', '.*?'),
                    strtoupper($writeUser)) . '$/', strtoupper($user)))
            return TRUE;


      return FALSE;

   }



   /**
     * Save Events To Backend 
     *
     * Saves all currently loaded events into the backend (NOTE
     * that this may not always represent ALL events for the
     * calendar!).  This should only be called once; events are
     * CREATED, and if you attempt to create them twice, errors
     * will be thrown.
     *
     * @param boolean $resetPermissions If TRUE, all events will
     *                                  have their permissions
     *                                  wiped and reset to match
     *                                  their parent calendar.  
     *
     * @access public
     *
     */
   function saveEvents($resetPermissions)
   {

      assert(is_bool($resetPermissions));


      // save one time events 
      //
      foreach ($this->events['onetime'] as $event)
      {
         if ($resetPermissions)
            $event->resetPermissionsToParent();

         create_event($this->getID(), $event);
      }

      // save recurring events
      //
      foreach ($this->events['recurring'] as $event)
      {
         if ($resetPermissions)
            $event->resetPermissionsToParent();

         create_event($this->getID(), $event);
      }

//TODO: holidays?  todos?

   }



   /**
     * Constructs iCal representation of this calendar
     *
     * Note that the returned text ONLY includes calendar 
     * attributes, which is not, per RFC2445, a valid iCal
     * object.  The caller should either insert at least
     * one other iCal component (event, timezone, etc)
     * or use the $includeEventsForUser parameter herein.
     *
     * @param boolean $includeExtras When TRUE, all calendar
     *                               attributes are included
     *                               in the return value, even 
     *                               those that are for internal 
     *                               use only and should NOT be
     *                               part of output to the 
     *                               outside world.  When FALSE,
     *                               only RFC-allowable fields
     *                               are included (optional;
     *                               default = FALSE).
     * @param string $includeEventsForUser When non-empty, this should
     *                                     specify a user for which all
     *                                     events, holidays, and others 
     *                                     are included (only events that
     *                                     the specified user has at least
     *                                     read access to are included).  
     *                                     When not given, no events are 
     *                                     included; only the calendar itself 
     *                                     is returned (optional; default = empty).
     * @param string $icalLineDelimOverride If given, will be used instead
     *                                      the RFC-standard CRLF to terminate
     *                                      iCal lines (optional)
     *
     * @return string The iCal stream representing this calendar
     *
     * @access public
     *
     */
   function getICal($includeExtras=FALSE, $includeEventsForUser='', 
                    $icalLineDelimOverride='')
   {

      // figure out line delimiter
      //
      if (empty($icalLineDelimOverride))
         $icalLineDelim = ICAL_LINE_DELIM;
      else
         $icalLineDelim = $icalLineDelimOverride;


      $iCalVersion = '2.0';
      $sm_cal_prodid = str_replace('###VERSION###', calendar_version(), SM_CAL_PRODID);


      $iCalText = 'BEGIN:VCALENDAR' . $icalLineDelim
                . 'VERSION:' . $iCalVersion . $icalLineDelim
                . $this->prodID->getICal($icalLineDelim)
                . 'CALSCALE:GREGORIAN' . $icalLineDelim
                . 'METHOD:PUBLISH' . $icalLineDelim

                // make Apple happy
                //
                . 'X-WR-CALNAME:' . $this->getName() . $icalLineDelim
                . 'X-WR-RELCALID:' . $this->id->getValue() . $icalLineDelim;


      // only show our own custom ID and name fields 
      // if this calendar was created by this application
      //
      if ($this->getProductID() == $sm_cal_prodid)
         $iCalText .= $this->id->getICal($icalLineDelim)
//LEFT OFF HERE
//TODO: following property should have language and encoding info! (take care of in constructor??)
                    . $this->name->getICal($icalLineDelim);


      // unknown attributes are included, since they
      // were probably custom attributes defined by an
      // external source
      //
      foreach ($this->unknownAttributes as $name => $attr)

         // already printed these
         //
         if ($name != 'X-WR-CALNAME' && $name != 'X-WR-RELCALID')
            $iCalText .= $attr . $icalLineDelim;


      // include all our internal attributes?
      //
      if ($includeExtras)
      {
         $iCalText .= $this->dom->getICal($icalLineDelim)
                    . $this->type->getICal($icalLineDelim)
                    . $this->sequence->getICal($icalLineDelim)
                    . $this->createdBy->getICal($icalLineDelim)
                    . $this->createdOn->getICal($icalLineDelim)
                    . $this->lastUpdatedBy->getICal($icalLineDelim)
                    . $this->lastUpdatedOn->getICal($icalLineDelim)
                    . $this->owners->getICal($icalLineDelim) 
                    . $this->readable_users->getICal($icalLineDelim) 
                    . $this->writeable_users->getICal($icalLineDelim);
      }



      // if we need to dump all events, do that here right before ending the calendar
      //
      if (!empty($includeEventsForUser))
      {
         $this->retrieveAllEvents($includeEventsForUser);
         foreach ($this->allEvents as $event)
            $iCalText .= $event->getICal(FALSE, $icalLineDelim);
      }



      $iCalText .= 'END:VCALENDAR' . $icalLineDelim;


      // fold lines that are too long
      //
      foldICalStreamByRef($iCalText);


      return $iCalText;

   }



   /**
     * Constructs a Calendar object from the given iCal stream
     *
     * @param array $iCalData The text value to be converted to a
     *                        Calendar object, one text line in 
     *                        each array value.
     * @param string $fallbackName An additional name to be used if 
     *                             none is given above or in any of the
     *                             calendar's extra attributes (optional)
     * @param string $uploadFilename A flag that indicates the source of this
     *                               calendar; if given, it represents the 
     *                               filename from which it was uploaded,
     *                               in which case the ID (if not
     *                               given) will be constructed differently,
     *                               so that it is not create-time based.
     *                               (optional; default = empty string)
     *
     * @return object A Calendar object representing the given iCal stream.
     *
     * @access public
     *
     */
   function getCalendarFromICal($iCalText, $fallbackName='', $uploadFilename='')
   {

      global $color; 


      // strip out CRLFs from each line
      //
      foreach ($iCalText as $x => $line)
         $iCalText[$x] = str_replace(ICAL_LINE_DELIM, '', $line);


      // unfold text
      //
      unfoldICalStreamByRef($iCalText);


      $id = '';
      $dom = '';
      $type = '';
      $prodID = '';
      $name = '';
      $createdBy = '';
      $createdOn = '';
      $lastUpdatedBy = '';
      $lastUpdatedOn = '';
      $sequence = '';
      $owners = '';
      $writeable_users = '';
      $readable_users = '';
      $unknownAttributes = array();
      $embeddedEvents = array();


      // pull out properties
      //
      reset($iCalText);
      while (list($junk, $line) = each($iCalText))
      // use reset/each/next instead so we can use next below inside the loop
      // foreach ($iCalText as $line)
      {

         $property = Property::extractICalProperty($line);


         // what do we do with each property?
         //
         switch ($property->getName())
         { 

            // skip this unless it is an embedded event
            //
            case 'BEGIN':
               if ($property->getValue() == 'VEVENT')
               {

                  // build event array and pass to event parser
                  //
                  $event = array($line);
                  while (list($junk, $line) = each($iCalText))
                  {
                     $event[] = $line;
                     $property = Property::extractICalProperty($line);
                     if ($property->getName() == 'END')
                        break;
                  }
                  
                  $embeddedEvents[] = Event::getEventFromICal($event);

               }
               break;


            // just skip these
            //
            case 'END':
               break;


            // version check
            //
            case 'VERSION':
               if ($property->getValue() != '2.0')
               {
                  plain_error_message('ERROR IN CALENDAR CLASS (getCalendarFromICal): Unsupported iCal version ' . $property->getValue(), $color);
                  exit;
               }
               break;


            // calendar scale check
            //
            case 'CALSCALE':
               if ($property->getValue() != 'GREGORIAN')
               {
                  plain_error_message('ERROR IN CALENDAR CLASS (getCalendarFromICal): Unsupported calendar scale ' . $property->getValue(), $color);
                  exit;
               }
               break;


            // product ID
            //
            case 'PRODID':
               $prodID = $property;
               break;


            // cal id for calendars created with this application
            //
            case 'X-SQ-CALID': 
               $id = $property;
               break;


            // cal name for calendars created with this application
            //
            case 'X-SQ-CALNAME':
               $name = $property;
               break;


            // cal domain for calendars created with this application
            //
            case 'X-SQ-CALDOMAIN':
               $dom = $property;
               break;


            // cal type for calendars created with this application
            //
            case 'X-SQ-CALTYPE':
               $type = $property;
               break;


            // cal sequence for calendars created with this application
            //
            case 'X-SQ-CALSEQUENCE':
               $sequence = $property;
               break;


            // user who created this calendar for calendars created with this application
            //
            case 'X-SQ-CALCREATOR':
               $createdBy = $property;
               break;


            // creation date for calendars created with this application
            // (need to reparse the value as a date)
            //
            case 'X-SQ-CALCREATED':
               $createdOn = Property::extractICalProperty($line, SM_CAL_ICAL_PROPERTY_TYPE_DATE);
               break;


            // user who last modified this calendar for calendars created with this application
            //
            case 'X-SQ-CALLASTUPDATOR':
               $lastUpdatedBy = $property;
               break;


            // date last modified for calendars created with this application
            // (need to reparse the value as a date)
            //
            case 'X-SQ-CALLAST-MODIFIED':
               $lastUpdatedOn = Property::extractICalProperty($line, SM_CAL_ICAL_PROPERTY_TYPE_DATE);
               break;


            // cal owners for calendars created with this application
            //
            case 'X-SQ-CALOWNERS':
               $owners = $property;
               break;


            // cal writeable users for calendars created with this application
            //
            case 'X-SQ-CALWRITEABLEUSERS':
               $writeable_users = $property;
               break;


            // cal readable users for calendars created with this application
            //
            case 'X-SQ-CALREADABLEUSERS':
               $readable_users = $property;
               break;


            // unknown parameters just pile into this array
            //
            default:
               $unknownAttributes[$property->getName()] = $line;
               break;

         } 

      }


      $cal = new Calendar($id, $sequence, $dom, $prodID, $type, $name,
                          $createdBy, $createdOn, $lastUpdatedBy, $lastUpdatedOn,
                          $owners, $readable_users, $writeable_users,
                          $unknownAttributes, $fallbackName, $uploadFilename);


      // add events (only if any were found)
      //
      if (sizeof($embeddedEvents))
      {

//LEFT OFF HERE um, if this is the only place we call the caching function for external calendar events, will it not be able to cache events outside of this date range?  will it slow considerably if the user views months outside this range? does this help (getting it out of the FORM first)?
         sqgetGlobalVar('year', $year, SQ_FORM);
         if (empty($year))
            $year = date('Y');

         $cacheDate = mktime(12, 0, 0, 1, 7, $year + 2);

         $newEvents = array();
         $newEvents['onetime'] = array();
         $newEvents['recurring'] = array();
//TODO.... how do we store Todo items???
//         $newEvents['todo'] = array();
         foreach ($embeddedEvents as $event)
         {
            if ($event->isOneTime())
               $newEvents['onetime'][] = $event;
            else if ($event->isRecurring())
            {
               $newEvents['recurring'][] = $event;
            }
//TODO.... how do we store Todo items???
//            else if ($event->isTask())
//               $newEvents['todo'][] = $event;
         }

         // precache recurring events
         //
         foreach (array_keys($newEvents['recurring']) as $index)
            $newEvents['recurring'][$index]->forceEventOccurrenceCache($cacheDate);


         // Add all events to the calendar now.
         //
         // Setting the third parameter below to TRUE means
         // we trust that all events are legitimate and no
         // duplicates exist; imported calendars that have
         // those problems should be fixed at the source anyway.
         // This also fixes problem when importing events w/no
         // IDs quickly while looping through uploaded iCal file,
         // timestamps are likely to match, so IDs will overlap,
         // but the TRUE below fixes that
         //
         $cal->addEvents($newEvents, 'xxx', TRUE, TRUE);

      }


      return $cal;


   }



}



?>
