<?php

/**
  * SquirrelMail Shared Calendar Plugin
  * Copyright (C) 2004-2005 Paul Lesneiwski <pdontthink@angrynerds.com>
  * This program is licensed under GPL. See COPYING for details
  *
  */


/**
  * Property class
  *
  * This class mimicks the structure of an iCal property,
  * including name, any properties, and value(s).
  *
  */
class Property
{

   var $name;
   var $type;
   var $parameters = array();
   var $value;
   var $rawValue;

   // helper date stuff -- no, this date has 
   // no meaning outside of this class
   //
   var $year = 2000;
   var $month = 1;
   var $day = 1;
   var $baseDate;



   /**
     * Property constructor
     *
     * If a type is given, the constructor will attempt to parse
     * the string value into the actual data type indicated by the
     * type.
     *
     * @param string $name The name of this property
     * @param mixed $value The property value, which is either a string text
     *                     value or an array of multiple string values
     * @param array $parameters A list of parameters defined for this property,
     *                          keyed by parameter name, where values are mixed;
     *                          they may be string parameter values, or possibly
     *                          an array of multiple string values
     * @param string $type The type of this property, which should correspond
     *                     to the property type constants defined in {@link constants.php}
     *                     (optional; default is SM_CAL_ICAL_PROPERTY_TYPE_TEXT)
     *
     */
   function Property($name, $value='', $parameters=array(),
                     $type=SM_CAL_ICAL_PROPERTY_TYPE_TEXT)
   {

      $this->baseDate = mktime(0, 0, 0, $this->month, $this->day, $this->year);
      $this->name = strtoupper($name);
      $this->parameters = $parameters;
      $this->type = $type;
      $this->setValue($value);

   }
      


// ---------- PUBLIC ----------



   /**
     * Parse Value 
     *
     * Attempts to parse the given string value into the
     * actual type for this property.  See {@see getValue()}
     * for documentation of returned value types for each
     * kind of iCal value.
     *
     * @param mixed $rawTextValue The raw value to be parsed
     *                            as a string or an array of
     *                            strings
     * @param array $parameters The parameters for the value
     *                          being parsed (typically used
     *                          to determine timezone info
     *                          for timezone-based date/times)
     * @param string $typeOverride The type to be used to 
     *                             interpolate the given value
     *                             (usually given as $this->type)
     *
     * @return mixed The parsed data type (in an array if 
     *               $rawTextValue is also an array of values)
     *
     */
   function parseValue($rawTextValue, $parameters, $typeOverride='')
   {

      // call back recursively for arrays and then bail
      // unless we are dealing with an RRULE, whose 
      // value should actually be brought back into a
      // single string (stupid RFC inconsistency)
      //
      if (is_array($rawTextValue) && $typeOverride != SM_CAL_ICAL_PROPERTY_TYPE_RRULE)
      {
         $returnArray = array();
         foreach ($rawTextValue as $val)
            $returnArray[] = Property::parseValue($val, $parameters, $typeOverride);
         return $returnArray;
      }


      // no sense in parsing empty values 
//TODO: RRULE values really make a lot more sense as parameters, so I 
//      would not be surprised if some applications format their RRULEs
//      with all data as parameters with NO VALUE at all.  Do we want
//      to accomodate that here?
      //
      if (empty($rawTextValue))
         return $rawTextValue;


      global $color, $WEEKDAYS;
      $year = Property::getBaseYear();
      $month = Property::getBaseMonth();
      $day = Property::getBaseDay();



      switch ($typeOverride)
      {
//TODO: need to finish other types; for now all default to text, which is usually fine

         case SM_CAL_ICAL_PROPERTY_TYPE_BOOLEAN:

            if (strtoupper($rawTextValue) == 'TRUE') return TRUE;
            else return FALSE;


         case SM_CAL_ICAL_PROPERTY_TYPE_RRULE:

            // if we treat the value as a set of parameters, we win!
            //
            // make sure commas didn't split up value text, though
            //
            if (is_array($rawTextValue)) $rawTextValue = implode(',', $rawTextValue);
            $parameters = $this->extractICalProperty('RRULE;' . $rawTextValue . ':');
            $parameters = $parameters->parameters;


            if (!isset($parameters['FREQ']) || empty($parameters['FREQ']))
            {
               plain_error_message('Missing recurrence rule frequency in iCal stream', $color);
               exit;
            }


            if (!isset($parameters['INTERVAL']) || empty($parameters['INTERVAL']))
               $parameters['INTERVAL'] = 1;


            if (isset($parameters['BYSETPOS']) && !isset($parameters['BYSECOND'])
             && !isset($parameters['BYMINUTE']) && !isset($parameters['BYHOUR'])
             && !isset($parameters['BYDAY']) && !isset($parameters['BYMONTHDAY'])
             && !isset($parameters['BYYEARDAY']) && !isset($parameters['BYWEEKNO'])
             && !isset($parameters['BYMONTH']))
            {
               plain_error_message('BYSETPOS must have accompanying criteria in iCal stream', $color);
               exit;
            }
            if (!isset($parameters['BYSETPOS'])) $parameters['BYSETPOS'] = array();
            if (!is_array($parameters['BYSETPOS'])) 
               $parameters['BYSETPOS'] = array($parameters['BYSETPOS']);


            // change UNTIL into a timestamp
            //
            if (isset($parameters['UNTIL']))
            {
               $parameters['UNTIL'] = Property::extractICalProperty('JUNK:' . $parameters['UNTIL'], SM_CAL_ICAL_PROPERTY_TYPE_DATE);
               $parameters['UNTIL'] = $parameters['UNTIL']->getValue();  // now we have the needed (ending) timestamp
            }


            if (!isset($parameters['BYSECOND'])) $parameters['BYSECOND'] = array();
            if (!is_array($parameters['BYSECOND'])) 
               $parameters['BYSECOND'] = array($parameters['BYSECOND']);
            foreach ($parameters['BYSECOND'] as $i => $j)
               if (strlen($j) < 2) $parameters['BYSECOND'][$i] = '0' . $j;

            if (!isset($parameters['BYMINUTE'])) $parameters['BYMINUTE'] = array();
            if (!is_array($parameters['BYMINUTE'])) 
               $parameters['BYMINUTE'] = array($parameters['BYMINUTE']);
            foreach ($parameters['BYMINUTE'] as $i => $j)
               if (strlen($j) < 2) $parameters['BYMINUTE'][$i] = '0' . $j;

            if (!isset($parameters['BYHOUR'])) $parameters['BYHOUR'] = array();
            if (!is_array($parameters['BYHOUR'])) 
               $parameters['BYHOUR'] = array($parameters['BYHOUR']);
            foreach ($parameters['BYHOUR'] as $i => $j)
               if (strlen($j) < 2) $parameters['BYHOUR'][$i] = '0' . $j;


            // BYDAY may only have leading numbers when FREQ is MONTHLY or YEARLY
            //
            if (isset($parameters['BYDAY']))
            {
               if (!is_array($parameters['BYDAY'])) 
                  $parameters['BYDAY'] = array($parameters['BYDAY']);

               foreach ($parameters['BYDAY'] as $i => $aDay)
               {
                  $parameters['BYDAY'][$i] = strtoupper($aDay);
                  if ($parameters['FREQ'] != SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_MONTHLY
                   && $parameters['FREQ'] != SM_CAL_ICAL_EVENT_RECURRENCE_FREQ_YEARLY
                   && !array_key_exists(strtoupper($aDay), $WEEKDAYS))
                  {
                     plain_error_message('Bad BYDAY format in iCal stream (only weekdays allowable for frequency type ' . $parameters['FREQ'] . '): ' . $aDay, $color);
                     exit;
                  }
               }
            }
            else $parameters['BYDAY'] = array();

            if (!isset($parameters['BYMONTHDAY'])) $parameters['BYMONTHDAY'] = array();
            if (!is_array($parameters['BYMONTHDAY'])) 
               $parameters['BYMONTHDAY'] = array($parameters['BYMONTHDAY']);
//Note -- zero padding appears not to be needed (php figures it out either way, but
//        if problems pop up, this could be a place to look. If something has to be
//        changed, watch out for optional +/- signs

            if (!isset($parameters['BYYEARDAY'])) $parameters['BYYEARDAY'] = array();
            if (!is_array($parameters['BYYEARDAY'])) 
               $parameters['BYYEARDAY'] = array($parameters['BYYEARDAY']);
//Note -- zero padding appears not to be needed (php figures it out either way, but
//        if problems pop up, this could be a place to look. If something has to be
//        changed, watch out for optional +/- signs

            if (!isset($parameters['BYWEEKNO'])) $parameters['BYWEEKNO'] = array();
            if (!is_array($parameters['BYWEEKNO'])) 
               $parameters['BYWEEKNO'] = array($parameters['BYWEEKNO']);
//Note -- zero padding appears not to be needed (php figures it out either way, but
//        if problems pop up, this could be a place to look. If something has to be
//        changed, watch out for optional +/- signs

            if (!isset($parameters['BYMONTH'])) $parameters['BYMONTH'] = array();
            if (!is_array($parameters['BYMONTH'])) 
               $parameters['BYMONTH'] = array($parameters['BYMONTH']);
            foreach ($parameters['BYMONTH'] as $i => $j)
               if (strlen($j) < 2) $parameters['BYMONTH'][$i] = '0' . $j;

            // validate week start day and convert to numeric representation thereof
            //
            if (isset($parameters['WKST']))
            {
               if (!array_key_exists(strtoupper($parameters['WKST']), $WEEKDAYS))
               {
                  plain_error_message('Bad week start value in iCal stream: ' . $parameters['WKST'], $color);
                  exit;
               }
               else
                  $parameters['WKST'] = $WEEKDAYS[strtoupper($parameters['WKST'])];
            }


            // actual value is our modified "parameters" (raw text value)
            //
            return $parameters;


         case SM_CAL_ICAL_PROPERTY_TYPE_PERIOD:

            list($start, $end) = explode('/', $rawTextValue);
            $start = Property::extractICalProperty('JUNK:' . $start, SM_CAL_ICAL_PROPERTY_TYPE_DATE);
            $start = $start->getValue();  // now we have starting timestamp

            if (strpos($end, 'P') !== FALSE)
            {
               $end = Property::parseValue($end, $parameters, SM_CAL_ICAL_PROPERTY_TYPE_DURATION);
               // convert to actual date
               $end += $start;
            }
            else
            {
               $end = Property::extractICalProperty('JUNK:' . $end, SM_CAL_ICAL_PROPERTY_TYPE_DATE);
               $end = $end->getValue();  // now we have ending timestamp
            }
            return array($start, $end);


         case SM_CAL_ICAL_PROPERTY_TYPE_DURATION:

            preg_match('/^([+-]{0,1})P(\d+[WHMSD])+$/', $rawTextValue, $matches);
            $totalSeconds = 0;
            $negativeDuration = FALSE; 
            for ($i = 1; $i < sizeof($matches); $i++)
            {
               if ($matches[$i] == '-') 
               {
                  $negativeDuration = TRUE; 
                  continue;
               }


               switch (substr($matches[$i], strlen($matches[$i]) - 1))
               {

                  case 'W':
                     $totalSeconds += (substr($matches[$i], 0, strpos($matches[$i], 'W')) 
                                      * SM_CAL_WEEK_SECONDS);
                     break;

                  case 'D':
                     $totalSeconds += (substr($matches[$i], 0, strpos($matches[$i], 'D')) 
                                      * SM_CAL_DAY_SECONDS);
                     break;

                  case 'H':
                     $totalSeconds += (substr($matches[$i], 0, strpos($matches[$i], 'H')) 
                                      * 3600);
                     break;

                  case 'M':
                     $totalSeconds += (substr($matches[$i], 0, strpos($matches[$i], 'M')) 
                                      * 60);
                     break;

                  case 'S':
                     $totalSeconds += substr($matches[$i], 0, strpos($matches[$i], 'S'));
                     break;

               }
            }
            if ($negativeDuration)
               return -($totalSeconds);
            else
               return $totalSeconds;


         case SM_CAL_ICAL_PROPERTY_TYPE_DATE:

            // this should work, but just in case, we do our own parsing
            //return strtotime($rawTextValue);
            preg_match('/^(\d{4})(\d{2})(\d{2})$/', $rawTextValue, $matches);
            if (sizeof($matches) != 4) 
            {
               plain_error_message('Badly formatted DATE stamp in iCal stream', $color);
               exit;
            }

            // PHP date functions are limited to parsing years
            // starting with 1970; if we have something older
            // than that, just set it to 1970...
            //
            // TODO: does that completely b0rk such events?
            //
            if ($matches[1] < 1970) $matches[1] = 1970;

            // NOTE the following zero time is important and can't be changed,
            // because when only giving a DATE (not DATETIME), the event is an 
            // all-day "anniversary" type
            //
            return mktime(0, 0, 0, $matches[2], $matches[3], $matches[1]);

         
         case SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_LOCAL:

            preg_match('/^(\d{4})(\d{2})(\d{2})T(\d{2})(\d{2})(\d{2})$/', $rawTextValue, $matches);
            if (sizeof($matches) != 7) 
            {
               plain_error_message('Badly formatted DATE/TIME (local) stamp in iCal stream', $color);
               exit;
            }

            // PHP date functions are limited to parsing years
            // starting with 1970; if we have something older
            // than that, just set it to 1970...
            //
            // TODO: does that completely b0rk such events?
            //
            if ($matches[1] < 1970) $matches[1] = 1970;

            return mktime($matches[4], $matches[5], $matches[6], 
                          $matches[2], $matches[3], $matches[1]);

         
         case SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_UTC:

            preg_match('/^(\d{4})(\d{2})(\d{2})T(\d{2})(\d{2})(\d{2})Z$/', $rawTextValue, $matches);
            if (sizeof($matches) != 7) 
            {
               plain_error_message('Badly formatted DATE/TIME (UTC) stamp in iCal stream: ' . $rawTextValue, $color);
               exit;
            }

            // PHP date functions are limited to parsing years
            // starting with 1970; if we have something older
            // than that, just set it to 1970...
            //
            // TODO: does that completely b0rk such events?
            //
            if ($matches[1] < 1970) $matches[1] = 1970;

            return gmmktime($matches[4], $matches[5], $matches[6], 
                            $matches[2], $matches[3], $matches[1]);

         
         case SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_TZ:

            preg_match('/^(\d{4})(\d{2})(\d{2})T(\d{2})(\d{2})(\d{2})$/', $rawTextValue, $matches);
            if (sizeof($matches) != 7) 
            {
               plain_error_message('Badly formatted DATE/TIME (tz) stamp in iCal stream', $color);
               exit;
            }
            if (!isset($parameters['TZID']))
            {
               plain_error_message('Missing time zone parameter for DATE/TIME (tz) stamp in iCal stream', $color);
               exit;
            }

            // PHP date functions are limited to parsing years
            // starting with 1970; if we have something older
            // than that, just set it to 1970...
            //
            // TODO: does that completely b0rk such events?
            //
            if ($matches[1] < 1970) $matches[1] = 1970;

            $time = mktime($matches[4], $matches[5], $matches[6], 
                           $matches[2], $matches[3], $matches[1]);
            return convertTimestampToTimezone($time, $parameters['TZID']);

         
         case SM_CAL_ICAL_PROPERTY_TYPE_TIME_LOCAL:

            preg_match('/^(\d{2})(\d{2})(\d{2})$/', $rawTextValue, $matches);
            if (sizeof($matches) != 4) 
            {
               plain_error_message('Badly formatted TIME (local) stamp in iCal stream', $color);
               exit;
            }
            return mktime($matches[1], $matches[2], $matches[3], $month, $day, $year);


         case SM_CAL_ICAL_PROPERTY_TYPE_TIME_UTC:

            preg_match('/^(\d{2})(\d{2})(\d{2})Z$/', $rawTextValue, $matches);
            if (sizeof($matches) != 4) 
            {
               plain_error_message('Badly formatted TIME (UTC) stamp in iCal stream', $color);
               exit;
            }
            return gmmktime($matches[1], $matches[2], $matches[3], $month, $day, $year);


         case SM_CAL_ICAL_PROPERTY_TYPE_TIME_TZ:

            preg_match('/^(\d{2})(\d{2})(\d{2})$/', $rawTextValue, $matches);
            if (sizeof($matches) != 4) 
            {
               plain_error_message('Badly formatted TIME (tz) stamp in iCal stream', $color);
               exit;
            }
            if (!isset($parameters['TZID']))
            {
               plain_error_message('Missing time zone parameter for TIME (tz) stamp in iCal stream', $color);
               exit;
            }
            $time = mktime($matches[1], $matches[2], $matches[3], $month, $day, $year);
            return convertTimestampToTimezone($time, $parameters['TZID']);

         
         case SM_CAL_ICAL_PROPERTY_TYPE_TEXT:
         default:

            // nothing to do here
            //
            return $rawTextValue;

      }

   }



   /**
     * Get Base Year
     *
     * This is the year used in the dummy date for time-only
     * values
     *
     * @return int The base year
     *
     */
   function getBaseYear()
   {
      return $this->year;
   }



   /**
     * Get Base Month
     *
     * This is the month used in the dummy date for time-only
     * values
     *
     * @return int The base month
     *
     */
   function getBaseMonth()
   {
      return $this->month;
   }



   /**
     * Get Base Day
     *
     * This is the day used in the dummy date for time-only
     * values
     *
     * @return int The base day
     *
     */
   function getBaseDay()
   {
      return $this->day;
   }



   /**
     * Get Property Name
     *
     * @return string This property's name
     *
     */
   function getName()
   {
      return $this->name;
   }



   /**
     * Set Property Name
     *
     * @param string The new name to be assigned to this property
     *
     */
   function setName($name)
   {
      $this->name = strtoupper($name);
   }



   /**
     * Get Property Parameters
     *
     * @return array This property's parameters
     *
     */
   function getParameters()
   {
      return $this->parameters;
   }



   /**
     * Set Property Parameters
     *
     * @param array The new set of parameters to be assigned to this property
     *
     */
   function setParameters($parameters)
   {
      $this->parameters = $parameters;
   }



   /**
     * Get Property Raw Value
     *
     * @return string This property's raw value, in
     *                standard iCal format 
     *
     */
   function getRawValue()
   {
      return $this->rawValue;
   }



   /**
     * Get Property Value
     *
     * @return mixed This property's value. Note that
     *               the returned data type is based on
     *               the current property type:
     *
     *               SM_CAL_ICAL_PROPERTY_TYPE_TEXT: string
     *               SM_CAL_ICAL_PROPERTY_TYPE_DATE: int (timestamp, time is always 12:00:00)
     *               SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_LOCAL: int (timestamp)
     *               SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_UTC: int (timestamp)
     *               SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_TZ: int (timestamp)
     *               SM_CAL_ICAL_PROPERTY_TYPE_BOOLEAN: boolean
     *               SM_CAL_ICAL_PROPERTY_TYPE_DURATION: int (number of seconds)
     *               SM_CAL_ICAL_PROPERTY_TYPE_PERIOD: array (int (starting timestamp), 
     *                  int (ending timestamp (even when value was given as date/dur)))
     *               SM_CAL_ICAL_PROPERTY_TYPE_TIME_LOCAL: int (timestamp, 
     *                  based on date of Jan 1, 2000)
     *               SM_CAL_ICAL_PROPERTY_TYPE_TIME_UTC: int (timestamp, 
     *                  based on date of Jan 1, 2000)
     *               SM_CAL_ICAL_PROPERTY_TYPE_TIME_TZ: int (timestamp, 
     *                  based on date of Jan 1, 2000)
     *               SM_CAL_ICAL_PROPERTY_TYPE_RRULE: array (parsed version of 
     *                  paramters describing the recurrence rule, with possible
     *                  entries 'FREQ', 'COUNT', 'UNTIL', 'INTERVAL', 'BYSECOND',
     *                  'BYMINUTE', 'BYHOUR', 'BYDAY', 'BYMONTHDAY', 'BYYEARDAY', 
     *                  'BYWEEKNO', 'BYMONTH', 'BYSETPOS' , 'WKST')
     *               SM_CAL_ICAL_PROPERTY_TYPE_URI: string
     *               SM_CAL_ICAL_PROPERTY_TYPE_UTC_OFFSET: string
     *               SM_CAL_ICAL_PROPERTY_TYPE_BINARY: string
     *               SM_CAL_ICAL_PROPERTY_TYPE_CAL_ADDRESS: string
     *               SM_CAL_ICAL_PROPERTY_TYPE_FLOAT: string
     *               SM_CAL_ICAL_PROPERTY_TYPE_INT: string
     *
     *               Also note that many properties may have 
     *               multiple values, in which case an array
     *               is returned, where array values correspond
     *               to the types listed above.
     *
     */
   function getValue()
   {
      return $this->value;
   }



   /**
     * Set Property Value
     *
     * Sets the internal value; keeping both its raw form
     * as well as a parsed version thereof (based on the
     * type of this property).
     *
     * @param mixed $value The new value to be assigned to this property
     *
     */
   function setValue($value)
   {
      $this->rawValue = $value;
      $this->value = $this->parseValue($value, $this->parameters, $this->type);
   }



   /**
     * (re)Set Property Value
     *
     * Sets the internal value without attempting to re-parse the 
     * given value and without changing the original raw value string.
     *
     * WARNING: Use this method carefully, as it should only
     *          be used as a way of overriding the parsed value
     *          already in this object - and should never be 
     *          used when the value's content is actually changing
     *
     * @param mixed $value The new value to be assigned to this property
     *
     */
   function resetValue($value)
   {
      $this->value = $value;
   }



   /**
     * Get Property Type
     *
     * @return string This property's type
     *
     */
   function getType()
   {
      return $this->type;
   }



   /**
     * Set Property Type
     *
     * @param string The new type to be assigned to this property
     *
     */
   function setType($type)
   {
      $this->type = strtoupper($type);
   }



   /**
     * Encode text values for iCal data stream
     *
     * @param string $text The text value to be escaped
     *
     * @return string The escaped text, per RFC2445
     *
     */
   function escapeICalValue($text)
   {
   
      $text = str_replace(array('\\', ';', ',', "\n", "\N"),
                          array('\\\\', '\;', '\,', '\n', '\n'),
                          $text);
   
      return $text;
   
   }
   
   
   
   /**
     * Encode text values (by reference) for iCal data stream
     *
     * @param string $text The text value to be escaped
     *
     */
   function escapeICalValueByRef(&$text)
   {
   
      $text = str_replace(array('\\', ';', ',', "\n", "\N"),
                          array('\\\\', '\\;', '\\,', '\n', '\n'),
                          $text);
   
   }
   


   /**
     * Decode text values for iCal data stream
     *
     * @param string $text The text value to be unescaped
     *
     * @return string The unescaped text, per RFC2445
     *
     */
   function unescapeICalValue($text)
   {
   
      $text = str_replace(array('\\\\', '\\;', '\\,', '\n', '\N'),
                          array('\\', ';', ',', "\n", "\n"),
                          $text);
   
      return $text;
   
   }
   
   
   
   /**
     * Decode text values (by reference) for iCal data stream
     *
     * @param string $text The text value to be unescaped
     *
     */
   function unescapeICalValueByRef(&$text)
   {
   
      $text = str_replace(array('\\\\', '\\;', '\\,', '\n', '\N'),
                          array('\\', ';', ',', "\n", "\n"),
                          $text);
   
   }
   


   /**
     * Parses an iCal property into a Property object
     *
     * @param string $text The full property text
     * @param string $typeGuess The type that the value is
     *                          likely to be; if given (parameter
     *                          is optional and type defaults to
     *                          SM_CAL_ICAL_PROPERTY_TYPE_TEXT), 
     *                          this function will attempt to 
     *                          interpret the value as desired.  
     *                          If given as any of the following, 
     *                          this function will automatically 
     *                          determine which of these the value
     *                          is actually given in (so specifying
     *                          the value type as a DATE works even
     *                          for DATE/TIME values and TIME values):
     *                          SM_CAL_ICAL_PROPERTY_TYPE_DATE
     *                          SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_LOCAL
     *                          SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_UTC
     *                          SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_TZ
     *                          SM_CAL_ICAL_PROPERTY_TYPE_TIME_LOCAL
     *                          SM_CAL_ICAL_PROPERTY_TYPE_TIME_UTC
     *                          SM_CAL_ICAL_PROPERTY_TYPE_TIME_TZ
     *                          If a value cannot be interpreted as
     *                          requested, it will be treated as a 
     *                          TEXT value.
     *
     * @return object A Property object containing the parsed
     *                iCal data
     *
     */
   function extractICalProperty($text, $typeGuess=SM_CAL_ICAL_PROPERTY_TYPE_TEXT)
   {
   
      global $color;

   
      // extract property info
      //
      // skip over quotes when looking for
      // colon and semicolon
      //
      $insideQuotes = FALSE;
      $colonPos = strlen($text) + 1;
      $semicolonPositions = array();
      for ($i = 0; $i < strlen($text); $i++)
      {

         $char = substr($text, $i, 1);

         if ($char == '"') 
            $insideQuotes = !$insideQuotes;

         else if ($char == ';' && !$insideQuotes)
            $semicolonPositions[] = $i;

         else if ($char == ':' && !$insideQuotes)
         {
            $colonPos = $i;
            break;
         }

      }
   
   
      // extract parameters
      //
      $parameters = array();
      $temp_parameters = array();
      if (!empty($semicolonPositions))
      {

         $endNamePos = $semicolonPositions[0];

         foreach ($semicolonPositions as $i => $semicolon)
         {

            if (isset($semicolonPositions[$i + 1]))
               $nextSemicolon = $semicolonPositions[$i + 1];
            else
               $nextSemicolon = $colonPos;

            $temp_parameters[] = substr($text, $semicolon + 1, $nextSemicolon - $semicolon - 1);

         }

         foreach ($temp_parameters as $param)
         {

            $insideQuotes = FALSE;
            $commaPositions = array();
            $equalPos = strlen($param) + 1;
            for ($j = 0; $j < strlen($param); $j++)
            {

               $char = substr($param, $j, 1);

               if ($char == '"') 
                  $insideQuotes = !$insideQuotes;

               else if ($char == ',' && !$insideQuotes) 
                  $commaPositions[] = $j;

               else if ($char == '=' && !$insideQuotes)
                  $equalPos = $j;

            }


            // this will break if comma inserted before equal sign
            // but then again, a lot of this will break with wrongly
            // formatted input
            //
            $param_values = array();
            if (empty($commaPositions)) 
               $param_values[] = $param;
            else
               $commaPositions[] = strlen($param);

            foreach ($commaPositions as $j => $comma)
            {

               if (isset($commaPositions[$j - 1]))
                  $prevComma = $commaPositions[$j - 1] + 1;
               else
                  $prevComma = 0;

               $param_values[] = substr($param, $prevComma, $comma - $prevComma);

            }


            $paramName = substr($param_values[0], 0, $equalPos);
            $param_values[0] = substr($param_values[0], $equalPos + 1);
            if (sizeof($param_values) == 1) $param_values = $param_values[0];

            $parameters[strtoupper($paramName)] = $param_values;

         }

      }
      else
         $endNamePos = $colonPos;
   
   
      // property name 
      //
      $name = strtoupper(substr($text, 0, $endNamePos));


      // property value 
      //
      $value = substr($text, $colonPos + 1);
      $value = str_replace('\\,', '###COMMA_IN_VALUE###', trim($value));
      $value = explode(',', $value);
      foreach($value as $x => $val)
      {
         // avoid nasty regex with negative lookbehind 
         // assertion blah blah by just fiddling with commas
         //
         $val = str_replace('###COMMA_IN_VALUE###', '\\,', $val);
         $value[$x] = Property::unescapeICalValue($val);
      }
      if (sizeof($value) == 1) $value = $value[0];



      // adjust type if necessary
      //
      $type = $typeGuess;
      switch ($typeGuess)
      {

         case SM_CAL_ICAL_PROPERTY_TYPE_DATE:
         case SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_LOCAL:
         case SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_UTC:
         case SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_TZ:
         case SM_CAL_ICAL_PROPERTY_TYPE_TIME_LOCAL:
         case SM_CAL_ICAL_PROPERTY_TYPE_TIME_UTC:
         case SM_CAL_ICAL_PROPERTY_TYPE_TIME_TZ:
         case SM_CAL_ICAL_PROPERTY_TYPE_PERIOD:

            // auto-sense date/time types including period type
            //
            if (strpos($value, '/') !== FALSE)
            {
               $type = SM_CAL_ICAL_PROPERTY_TYPE_PERIOD;
            }
            else if (preg_match('/^\d{6}$/', $value))
            {
               if (isset($parameters['TZID']))
                  $type = SM_CAL_ICAL_PROPERTY_TYPE_TIME_TZ;
               else
                  $type = SM_CAL_ICAL_PROPERTY_TYPE_TIME_LOCAL;
            }
            else if (preg_match('/^\d{6}Z$/', $value))
            {
               $type = SM_CAL_ICAL_PROPERTY_TYPE_TIME_UTC;
            }
            else if (preg_match('/^\d{8}$/', $value))
            {
               $type = SM_CAL_ICAL_PROPERTY_TYPE_DATE;
            }
            else if (preg_match('/^\d{8}T\d{6}Z$/', $value))
            {
               $type = SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_UTC;
            }
            else if (preg_match('/^\d{8}T\d{6}$/', $value))
            {
               if (isset($parameters['TZID']))
                  $type = SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_TZ;
               else
                  $type = SM_CAL_ICAL_PROPERTY_TYPE_DATETIME_LOCAL;
            }
            break;



         case SM_CAL_ICAL_PROPERTY_TYPE_TEXT:
         default:

            // nothing to do here
            //
            break;

      }



//sm_print_r($name, $parameters, $value, $type, '=====================');
      return new Property($name, $value, $parameters, $type);

   }
   
   
   
   /**
     * Get iCal Format Data Stream
     *
     * @return string The iCal-formatted representation of this property
     *
     */
   function getICal()
   {

      // name
      //
      $iCalText = $this->getName();


      // parameters
      //
      $params = $this->getParameters();
      if (!empty($params))
      {
         foreach ($params as $param => $val)
         {

            $iCalText .= ';' . $param;
            if (!empty($val)) $iCalText .= '=';

            if (is_array($val))
            {
               $first = TRUE;
               foreach ($val as $v)
               {
                  if (!$first) $iCalText .= ',';
                  if (preg_match('/[;:,]/', $v))
                     $iCalText .= '"' . $v . '"';
                  else
                     $iCalText .= $v;
                  $first = FALSE;
               }
            }
            else
               $iCalText .= $val;

         }
      }


      // value
      //
      $val = $this->getRawValue();
      $iCalText .= ':';
      if (is_array($val))
      {
         $first = TRUE;
         foreach ($val as $v)
         {
            if (!$first) $iCalText .= ',';
            $iCalText .= ($this->getType() == SM_CAL_ICAL_PROPERTY_TYPE_RRULE ? $v
                          : $this->escapeICalValue($v));
            $first = FALSE;
         }
      }
      else
         $iCalText .= ($this->getType() == SM_CAL_ICAL_PROPERTY_TYPE_RRULE ? $val 
                       : $this->escapeICalValue($val));


      // EOL
      //
      $iCalText .= ICAL_LINE_DELIM;


      return $iCalText;

   }



}



?>
