/**
  * SquirrelMail Server Settings Backend Plugin SUID Backend
  *
  * squirrelmail_server_settings_suid_backend.c
  *
  * Copyright 2004-2010 Paul Lesniewski <paul@squirrelmail.org>
  * Copyright 2007 Dan Astoorian
  * Copyright 2001-2004 Jonathan B. Bayer
  * PAM support courtesy of Christiane Ruetten
  * IMAP authentication courtesy of David Phillips <david <at> geektech.com>
  *
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * This is a set-UID wrapper intended for PHP scripts that
  * that need read and/or write access to files not otherwise
  * accessible (or at least writable) by the web server itself.
  * Usually, those files are in the user's "home" directory (the
  * location of which is configurable), and access is jailed in
  * that "home" directory, but with certain configuration settings,
  * files can be accessed anywhere on the local server (or anywhere
  * traversable by normal filesystem means).  Configure CAREFULLY!
  *
  * Read operations are always performed by making a copy of the
  * desired file in a location where the web server can read the
  * copy (usually the SquirrelMail attachments directory, but this
  * is up to the caller).  Therefore, reading large files can be
  * problematic and should be avoided.
  *
  * Please see the README file in the same directory that this
  * file is found in for detailed information about what this
  * program does and how to configure and compile it (compilation
  * directives are not directly documented, but there are
  * configuration options that correspond and control all of
  * the possible compilation directives).
  *
  * Installation:
  *
  *    Must be owned by root, be set-UID and be executable by the
  *    web server.  The "make install" process should do this for
  *    you, but for example, after compilation:
  *
  *       # chown root:root squirrelmail_server_settings_suid_backend
  *       # chmod 4755 squirrelmail_server_settings_suid_backend
  *
  *    It should be safe to allow anyone to execute the program,
  *    especially if "AUTHORIZED_INVOKER" was enabled at compile
  *    time (configure option --enable-authorized-invoker), in which
  *    case then the executing user (real user) must have the same
  *    UID as the user defined by "AUTHORIZED_INVOKER" ("apache" by
  *    default).
  *
  *    However, it is of course also possible to be more restrictive
  *    with the program's ownership and permissions (replace "apache"
  *    with whatever user runs the web server in your environment):
  *
  *       # chown root:apache squirrelmail_server_settings_suid_backend
  *       # chmod 4750 squirrelmail_server_settings_suid_backend
  *
  * Actions:
  *
  *    file_exists directory_exists path_exists stat lstat get_file put_file
  *    delete_file
  *
  * File Manager Actions (only available if configured to allow file manager actions):
  *
  *    list_directory list_directory_detailed_stat list_directory_lstat
  *    change_permissions mkdir mkdir_recursive delete_directory
  *    delete_file_or_directory rename
  *
  * Vacation Actions (only available if configured to allow vacation actions):
  *
  *      vacation_init vacation_init_with_interval
  *
  * Usage (note that steps two and/or three are only necessary if
  * compiled with USE_MASTER_LOGIN (configuration option
  * --enable-master-login); step two is only necessary if ALSO
  * compiled with MASTER_USERNAME (configuration option
  * --enable-master-username)):
  *
  *    1) call with syntax defined below
  *    2) optionally send master username on STDIN
  *    3) optionally send master password on STDIN
  *    4) send user password on STDIN
  *
  * Syntax:
  *
  *      squirrelmail_server_settings_suid_backend host user action source destination
  *
  * Note that currently the "host" argument is not used (the caller
  * should pass some dummy value in its place).
  *
  */



// Exit status codes
//
#define ERR_OK                 0    // no error
#define ERR_NOTFOUND           1    // file not found
#define ERR_BADPASS           32    // bad password
#define ERR_USAGE             33    // usage error
#define ERR_RESTRICTED        34    // not allowed to use this program
#define ERR_REMOTEFILE        35    // illegal remote filename
#define ERR_LOCALFILE         36    // illegal local filename
#define ERR_CONFIG            37    // global configuration problem
#define ERR_USER              38    // problem with this user
#define ERR_HOME              39    // problem accessing home directory
#define ERR_SOURCEFILE        40    // problem opening/stat()ing source file
#define ERR_DESTFILE          41    // problem opening/deleting dest file
#define ERR_COPYFILE          42    // problem copying file contents
#define ERR_UNLINK            43    // problem unlinking file
#define ERR_FILETYPE          44    // not a regular file
#define ERR_EXEC              45    // exec() of an external program failed
#define ERR_NOTSUPPORTED      46    // feature not enabled
#define ERR_SOCKET            47    // problem connecting to IMAP server (socket)
#define ERR_SQL_ALLOC_ENV     48    // problem allocating ODBC environemnt handle
#define ERR_SQL_SET_ENV_ATTR  49    // problem setting ODBC environment attribute
#define ERR_SQL_ALLOC_CONN    50    // problem allocating ODBC connection handle
#define ERR_SQL_SET_CONN_ATTR 51    // problem setting ODBC connection attribute
#define ERR_SQL_CONNECT       52    // problem connecting to ODBC data source
#define ERR_SQL_ALLOC_STMT    53    // problem allocating ODBC statement handle
#define ERR_SQL_QUERY         54    // problem executing ODBC database query
#define ERR_SQL_GET_ROWS      55    // problem retrieving the number of queried rows
#define ERR_SQL_GET_COLS      56    // problem retrieving the number of queried columns
#define ERR_SQL_BAD_ROWS      57    // query returned wrong number of rows
#define ERR_SQL_BAD_COLS      58    // query returned wrong number of columns
#define ERR_SQL_GET_DATA      59    // problem obtaining data from query
#define ERR_BAD_LIBRARY       60    // UW c-client library does not support IMAP
#define ERR_RMDIR             61    // problem deleting a directory
#define ERR_PRIVILEGE        125    // unexpected privileges problem
#define ERR_UNEXPECTED       126    // other unexpected error



#if HAVE_CONFIG_H
#include "config.h"
#else
#define HAVE_SYS_TYPES_H
#define HAVE_SYS_STAT_H
#define HAVE_UNISTD_H
#define HAVE_FCNTL_H
#define HAVE_STDLIB_H
#define HAVE_STRINGS_H
#define HAVE_STRING_H
#define HAVE_LIBGEN_H
#define HAVE_DIRENT_H
#define HAVE_CRYPT_H
#define HAVE_PWD_H
#define HAVE_GRP_H
#define HAVE_ERRNO_H
#define HAVE_SHADOW_H
#define HAVE_NETINET_IN_H
#define HAVE_NETDB_H
#define HAVE_SYS_SOCKET_H
#define HAVE_ARPA_INET_H
#define HAVE_SQL_H
#define HAVE_SQLEXT_H
#define HAVE_MAIL_H
#endif // HAVE_CONFIG_H



#define BUFSIZE 512
#define BIGBUFSIZE 2048



#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif // HAVE_SYS_TYPES_H

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif // HAVE_SYS_STAT_H

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif // HAVE_UNISTD_H

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif // HAVE_FCNTL_H

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif // HAVE_STDLIB_H

#include <stdio.h>

#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif // HAVE_STRINGS_H

#ifdef HAVE_STRINGS_H
#include <string.h>
#endif // HAVE_STRING_H

#ifdef HAVE_LIBGEN_H
#include <libgen.h>
#endif // HAVE_LIBGEN_H

#ifdef HAVE_DIRENT_H
#include "dirent.h"
#endif // HAVE_DIRENT_H

#ifdef HAVE_CRYPT_H
#include <crypt.h>
#endif // HAVE_CRYPT_H

#ifdef HAVE_PWD_H
#include <pwd.h>
#endif // HAVE_PWD_H

#ifdef HAVE_GRP_H
#include <grp.h>
#endif // HAVE_GRP_H

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif // HAVE_ERRNO_H

#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif // HAVE_NETINET_IN_H

#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif // HAVE_NETDB_H

#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif // HAVE_SYS_SOCKET_H

#ifdef HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif // HAVE_ARPA_INET_H

#ifdef HAVE_SQL_H
#include <sql.h>
#endif // HAVE_SQL_H

#ifdef HAVE_SQLEXT_H
#include <sqlext.h>
#endif // HAVE_SQLEXT_H

#ifdef HAVE_MAIL_H
#include "mail.h"
#endif // HAVE_MAIL_H


// Set some sane defaults if not configured otherwise
//
#ifndef LOCAL_FILE_MODE
#define LOCAL_FILE_MODE 0600
#endif // !LOCAL_FILE_MODE

#ifndef REMOTE_FILE_MODE
#define REMOTE_FILE_MODE 0600
#endif // !REMOTE_FILE_MODE

#ifndef REMOTE_HOME_DIRECTORY_MODE
#define REMOTE_HOME_DIRECTORY_MODE 0700
#endif // !REMOTE_HOME_DIRECTORY_MODE

#ifndef PROG_UMASK
#define PROG_UMASK 0077 & ~(REMOTE_FILE_MODE|LOCAL_FILE_MODE|REMOTE_HOME_DIRECTORY_MODE)
#endif // !PROG_UMASK

#if defined(NO_AUTHORIZED_INVOKER)
#undef AUTHORIZED_INVOKER
#elif !defined(AUTHORIZED_INVOKER)
#define AUTHORIZED_INVOKER "apache"
#endif // defined(NO_AUTHORIZED_INVOKER), !defined(AUTHORIZED_INVOKER)

#if (defined(USE_SHADOW_AUTH) || defined(MASTER_USE_SHADOW_AUTH)) && defined(HAVE_SHADOW_H)
#include <shadow.h>
#endif // (defined(USE_SHADOW_AUTH) || defined(MASTER_USE_SHADOW_AUTH)) && defined(HAVE_SHADOW_H)

#if defined(USE_PAM_AUTH) || defined(MASTER_USE_PAM_AUTH)
#include <security/pam_appl.h>
#endif // defined(USE_PAM_AUTH) || defined(MASTER_USE_PAM_AUTH)

#ifdef USE_PAM_AUTH
#ifndef PAM_SERVICE
#define PAM_SERVICE "squirrelmail_server_settings_suid_backend"
#endif // !PAM_SERVICE
#endif // USE_PAM_AUTH

#ifdef MASTER_USE_PAM_AUTH
#ifndef MASTER_PAM_SERVICE
#define MASTER_PAM_SERVICE "squirrelmail_server_settings_suid_backend"
#endif // !MASTER_PAM_SERVICE
#endif // MASTER_USE_PAM_AUTH

#if defined(USE_IMAP_AUTH) || defined(USE_CCLIENT_AUTH)
#ifndef IMAP_SERVER_ADDRESS
#define IMAP_SERVER_ADDRESS "localhost"
#endif // !IMAP_SERVER_ADDRESS
#ifndef IMAP_PORT
#define IMAP_PORT 143
#endif // !IMAP_PORT
#endif // defined(USE_IMAP_AUTH) || defined(USE_CCLIENT_AUTH)

#if defined(MASTER_USE_IMAP_AUTH) || defined(MASTER_USE_CCLIENT_AUTH)
#ifndef MASTER_IMAP_SERVER_ADDRESS
#define MASTER_IMAP_SERVER_ADDRESS "localhost"
#endif // !MASTER_IMAP_SERVER_ADDRESS
#ifndef MASTER_IMAP_PORT
#define MASTER_IMAP_PORT 143
#endif // !MASTER_IMAP_PORT
#endif // defined(MASTER_USE_IMAP_AUTH) || defined(MASTER_USE_CCLIENT_AUTH)

#if !defined(ROOT_ALLOWED) && !defined(MIN_UID)
#define MIN_UID 1
#endif // !defined(ROOT_ALLOWED) && !defined(MIN_UID)

#if !defined(REMOTE_FILE_OK_CHARS) && !defined(REMOTE_FILE_OK_CHARS_ANY)
#define REMOTE_FILE_OK_CHARS \
  "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789.+-_~"
#endif // !defined(REMOTE_FILE_OK_CHARS) && !defined(REMOTE_FILE_OK_CHARS_ANY)



// Constants that refer to the arguments being passed in:
//
// squirrelmail_server_settings_suid_backend server user action source destination
//
// Currently the "host" argument is not being used
// 
#define HOST_ARG   1
#define USER_ARG   2
#define ACTION_ARG 3
#define SRC_ARG    4
#define DEST_ARG   5



// used to store the (real) UID/GID of the process
// that invoked us before we assumed root
//
static uid_t orig_uid = 0;
static gid_t orig_gid = 0;



// define handy error spew functions that don't
// do anything if SILENT is turned on
//
#ifdef SILENT
#define PRINTERROR2(format, arg1)
#define PRINTERROR3(format, arg1, arg2)
#define PRINTSYSERROR(e, s)
#else
#define PRINTERROR2(format, arg1) fprintf(stderr, format, arg1)
#define PRINTERROR3(format, arg1, arg2) fprintf(stderr, format, arg1, arg2)
#define PRINTSYSERROR(e, s) fprintf(stderr, "%s: %s", strerror(e), s)
#endif // SILENT



// ============================================================== 
// ==================== FUNCTION DEFINITIONS ==================== 
// ============================================================== 



/**
  * Report an error and exit
  *
  */
static void fail(int status_code, char *message)
{
   PRINTERROR2("%s\n", message);
   exit(status_code);
}



/**
  * Report an error including the system
  * error status, and exit
  *
  */
static void efail(int status_code, char *message)
{
   char *syserr;

   syserr = strerror(errno);
   if (!syserr)
      syserr = "Unknown error";

   PRINTERROR3("%s: %s\n", message, syserr);
   exit(status_code);
}



/**
  * Recursive mkdir
  *
  * Returns 1 upon success, otherwise fails and exits
  *
  * Based on bash shell code for mkdir(1)
  * Copyright (C) 1999-2009 Free Software Foundation, Inc.
  *
  */
static int mkdir_recursive(char *path, int mode, struct passwd *user_info)
{

   char *temp_path = strdup(path);
   char *p = temp_path;
   struct stat stat_info;
   char error_message[BUFSIZE];


   // see if the path already exists
   //
   if (stat(path, &stat_info) == 0)
   {

      // yes?  is it not a directory?
      //
      if (S_ISDIR(stat_info.st_mode) == 0)
      {
         snprintf(error_message, BUFSIZE, "mkdir_recursive(): \"%s\" exists but is not a directory", path);
         fail(ERR_HOME, error_message);
      }


      // otherwise, the path exists as a directory,
      // so there is nothing for us to do
      //
      free(temp_path);
      return 1;

   }


   // skip leading slashes
   //
   while (*p == '/') p++;


   // iterate over each directory in the path
   // 
   while (p = strchr(p, '/'))
   {

      // try seeing if what we have so far already exists
      //
      *p = '\0';
      if (stat(temp_path, &stat_info) != 0)
      {

         // nope, need to create it
         //
         if (mkdir(temp_path, mode) != 0)
         {
            snprintf(error_message, BUFSIZE, "mkdir_recursive(): mkdir(\"%s\") failed", temp_path);
            efail(ERR_HOME, error_message);
         }


         // change ownership to that of the indicated UID/GID
         //
         if (chown(temp_path, user_info->pw_uid, user_info->pw_gid) != 0)
         {

            // failed - remove directory if possible
            //
            rmdir(temp_path);
            snprintf(error_message, BUFSIZE, "mkdir_recursive(): chown(\"%s\") failed after mkdir()", temp_path);
            fail(ERR_HOME, error_message);
         }

      }

      // yes, it exists already, but is it a directory?
      //
      else if (S_ISDIR(stat_info.st_mode) == 0)
      {
         snprintf(error_message, BUFSIZE, "mkdir_recursive(): \"%s\" exists but is not a directory", temp_path);
         fail(ERR_HOME, error_message);
      }


      // restore the separator so we can iterate to next directory
      // and also skip redundant slashes
      //
      *p++ = '/';
      while (*p == '/') p++;

   }


   // there's one last path component left after we exit
   // the above loop, so create it now
   //
   if (stat(temp_path, &stat_info) != 0)  // this condition should always be true
   {

      // stat() failed; need to create...
      //
      if (mkdir(temp_path, mode) != 0)
      {
         snprintf(error_message, BUFSIZE, "mkdir_recursive(): mkdir(\"%s\") failed", temp_path);
         efail(ERR_HOME, error_message);
      }


      // change ownership to that of the indicated UID/GID
      //
      if (chown(temp_path, user_info->pw_uid, user_info->pw_gid) != 0)
      {

         // failed - remove directory if possible
         //
         rmdir(temp_path);
         snprintf(error_message, BUFSIZE, "mkdir_recursive(): chown(\"%s\") failed after mkdir()", temp_path);
         fail(ERR_HOME, error_message);
      }

   }

   // it exists already, but is it a directory?
   //
   else if (S_ISDIR(stat_info.st_mode) == 0)  // should never be true
   {
      snprintf(error_message, BUFSIZE, "mkdir_recursive(): \"%s\" exists but is not a directory", temp_path);
      fail(ERR_HOME, error_message);
   }

   free(temp_path);
   return 1;

}



/**
  * Set the real and effective user ID and group ID
  *
  * Returns 1 upon success, or 0 on error
  *
  */
static int set_real_and_effective_uid_and_gid(uid_t real_uid, gid_t real_gid,
                                              uid_t effective_uid, gid_t effective_gid)
{

   // the *effective* UID has to be 0 (root) to be
   // able to make the other uid/gid changes below
   //
   if (seteuid(0) != 0)
      return 0;

   if (setregid(real_gid, effective_gid) != 0)
      return 0;

   if (setreuid(real_uid, effective_uid) != 0)
      return 0;

   return 1;
}



/**
  * Assumes a user's credentials (sets effective UID and
  * GID) and chdir()s to her home directory.
  *
  * If compiled with CREATE_HOME_DIR and missing_ok is 1,
  * this also creates the directory if it doesn't exist.
  *
  * (If missing_ok is 1 and CREATE_HOME_DIR was not used,
  * missing_ok does not do anything (slightly counter-intuitive).)
  *
  * If retain_root is 1 then the real UID will remain as 0 (root),
  * otherwise, privileges are permanently dropped to the user's level.
  *
  * Returns 1 on success, or 0 if home directory doesn't exist and
  * missing_ok is 0; otherwise, fails and exits.
  *
  */
static int go_home(struct passwd *user_info, int missing_ok, int retain_root)
{

   char *home_directory, *current_dir;
   char error_message[BUFSIZE];
#ifdef CREATE_HOME_DIR
   struct stat stat_info;
#endif // CREATE_HOME_DIR


   // if configured to use other credentials, substitute them now (YIKES!)
   //
#ifdef OVERRIDE_UID
   user_info->pw_uid = OVERRIDE_UID;
#endif // OVERRIDE_UID
#ifdef OVERRIDE_GID
   user_info->pw_gid = OVERRIDE_GID;
#endif // OVERRIDE_GID


   // if configured to use other home directory, substitute it now (YIKES!)
   //
#ifdef OVERRIDE_HOME_DIR
   user_info->pw_dir = OVERRIDE_HOME_DIR;
#endif // OVERRIDE_HOME_DIR


   // validate home directory
   //
   if (user_info->pw_dir == NULL || user_info->pw_dir[0] != '/')
      fail(ERR_USER, "go_home(): Invalid home directory");


   // start with full root privilege - yikes!
   // (needed to change to and/or create home directories)
   //
   if (!set_real_and_effective_uid_and_gid(0, 0, 0, 0))
      efail(ERR_PRIVILEGE, "go_home(): Cannot assume super-user");


   // chdir to HOME_DIR_PREFIX (or "/" if no prefix is defined)
   // while still privileged
   //
#ifdef HOME_DIR_PREFIX
   if (chdir(HOME_DIR_PREFIX) != 0)
      efail(ERR_CONFIG, "go_home(): chdir(\"" HOME_DIR_PREFIX "\") failed");
   current_dir = HOME_DIR_PREFIX;

#else // HOME_DIR_PREFIX
   if (chdir("/") != 0)  // otherwise mkdir_recursive() will start from the cwd
      efail(ERR_CONFIG, "go_home(): chdir(\"/\") failed");
   current_dir = "/";
#endif // HOME_DIR_PREFIX


   // strip off leading slashes from home directory
   //
   home_directory = user_info->pw_dir + strspn(user_info->pw_dir, "/");


   // if home directory is now empty, it was pointing to root ("/")
   // this should be OK, because any sane configuration of this program
   // should have provided a HOME_DIR_PREFIX... if not, we presume
   // access to / is intentional, if poorly thought out
   //
   if (strcmp(home_directory, "") == 0)
      home_directory = "/";


   // create home directory if necessary
   //
#ifdef CREATE_HOME_DIR
   if (missing_ok && lstat(home_directory, &stat_info) != 0) // && errno == ENOENT)
   {

      // try to create & detect failure
      //
      if (!mkdir_recursive(home_directory, REMOTE_HOME_DIRECTORY_MODE, user_info))
      {
         snprintf(error_message, BUFSIZE, "go_home(): mkdir(\"%s\") failed", home_directory);
         efail(ERR_HOME, error_message);
      }

   }
#endif // CREATE_HOME_DIR


   // assume user's (effective) credentials
   //
   if (!set_real_and_effective_uid_and_gid(retain_root == 1 ? 0 : user_info->pw_uid,
                                           user_info->pw_gid,
                                           user_info->pw_uid,
                                           user_info->pw_gid))
      efail(ERR_PRIVILEGE, "go_home(): Cannot set effective UID to target");


   // change to the target directory
   //
   if (chdir(home_directory) != 0)
   {

      // not found?
      //
      if (errno == ENOENT && !missing_ok)
         return 0;
      else
      {
         snprintf(error_message, BUFSIZE, "go_home(): From \"%s\", chdir(\"%s\") failed", current_dir, home_directory);
         efail(ERR_HOME, error_message);
      }

   }


   // success
   //
#ifdef DEBUG
   fprintf(stderr, "go_home(): Changed to %s/%s\n", current_dir, home_directory);
#endif // DEBUG
   return 1;

}



/**
  * Validate a "remote" file path
  *
  * The file path is checked for whether or not it can contain
  * subdirectories, illegal upward directory traversal, and
  * certain configurable character sequences in the file name
  * part of the path
  *
  * See the configuration options that correspond to
  * REMOTE_SUBDIRECTORIES_OK, REMOTE_DOTFILES_REQUIRED, 
  * REMOTE_DOTFILES_BAD, REMOTE_FILE_FORWARD_PREFIX,
  * REMOTE_FILE_VACATION_PREFIX, REMOTE_FILE_PREFIX,
  * REMOTE_FILE_OK_CHARS and REMOTE_FILE_OK_CHARS_ANY.
  * Those should be --enable-remote-subdirectories-ok,
  * --enable-remote-filenames-dot, --enable-remote-file-prefix,
  * --enable-remote-file-custom-prefix, and
  * --(dis/en)able-remote-chars
  *
  * Returns a pointer to the given remote file path, which
  * will never be modified unless configured and compiled
  * to auto-adjust absolute file paths, in which case if the
  * file path is absolute, it will be modified to be a relative
  * path (prefixed with ".")
  *
  * If the path is invalid, this function fails and exits
  *
  */
static char *check_remote_file_path(char *file_path)
{

   char *base_name;
   int file_name_ok = 0;


#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   char *return_path;
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH


#ifdef REMOTE_SUBDIRECTORIES_OK
   int path_length;
   char *temp_path = strdup(file_path);
#endif // REMOTE_SUBDIRECTORIES_OK


   // check for empty path or attempt to access "." or ".."
   //
   if (file_path == NULL || file_path[0] == '\0')
      fail(ERR_REMOTEFILE, "Remote file name cannot be empty");
   if (strcmp(file_path, "..") == 0 || strcmp(file_path, ".") == 0)
      fail(ERR_REMOTEFILE, "Remote file name cannot be \".\" or \"..\"");


   // if subdirectories are allowed, slashes are ok (except
   // at the very beginning (in which case we might alter
   // it to be a relative path))
   //
#ifdef REMOTE_SUBDIRECTORIES_OK
   if (file_path[0] == '/')
   {
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return_path = malloc(strlen(file_path) + 2); // one for prepended "." and one for null char
      return_path[0] = '.';
      return_path[1] = '\0';
      strcat(return_path, file_path);
      //free(file_path);  // no - wasn't obtained with alloc() in the first place
      file_path = return_path;
#ifdef DEBUG
      fprintf(stderr, "check_remote_file_path(): Remote file path cannot be absolute; changing to \"%s\"\n", file_path);
#endif // DEBUG
#else // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      fail(ERR_REMOTEFILE, "Remote file path cannot be absolute");
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   }
   else
   {
      // this code is only here so that we can later free() this memory
      // without needing to figure out if it was malloc()ed or not
      //
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return_path = strdup(file_path);
      //free(file_path);  // no - wasn't obtained with alloc() in the first place
      file_path = return_path;
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   }


   // if subdirectories are allowed, slashes are ok, but
   // we disallow traversing parent directories; i.e.,
   // the path cannot start with "../", contain "/../",
   // or end in "/.."
   //
   path_length = strlen(file_path);

   if (strncmp(file_path, "../", 3) == 0
    || strstr(file_path, "/../") != NULL
    || (path_length >= 3 && strncmp(&file_path[path_length - 3], "/..", 3) == 0))
      fail(ERR_REMOTEFILE, "Remote file path cannot traverse \"..\"");

   base_name = basename(temp_path); // use temp_path because basename() may modify args


   // subdirectories not allowed?  then we allow no slashes whatsoever
   //
#else // REMOTE_SUBDIRECTORIES_OK
   if (strchr(file_path, '/') != NULL)
      fail(ERR_REMOTEFILE, "Remote file name cannot contain \"/\"");

   base_name = file_path;

#endif // REMOTE_SUBDIRECTORIES_OK


   // check if "dot files" are required or banned
   //
#ifdef REMOTE_DOTFILES_REQUIRED
   if (base_name[0] != '.')
      fail(ERR_REMOTEFILE, "Remote file name must start with a dot (\".\")");
#endif // REMOTE_DOTFILES_REQUIRED
#ifdef REMOTE_DOTFILES_BAD
   if (base_name[0] == '.')
      fail(ERR_REMOTEFILE, "Remote file name cannot start with a dot (\".\")");
#endif // REMOTE_DOTFILES_BAD


   // check if any remote file name prefixes are required
   //
   file_name_ok = 0; 

#ifdef REMOTE_FILE_FORWARD_PREFIX
   if (strncmp(base_name, ".forward", 8) == 0)
      file_name_ok = 1; 
#endif // REMOTE_FILE_FORWARD_PREFIX

#ifdef REMOTE_FILE_VACATION_PREFIX
   if (strncmp(base_name, ".vacation", 9) == 0)
      file_name_ok = 1; 
#endif // REMOTE_FILE_VACATION_PREFIX

#ifdef REMOTE_FILE_PREFIX
   if (strncmp(base_name, REMOTE_FILE_PREFIX, strlen(REMOTE_FILE_PREFIX)) == 0)
      file_name_ok = 1; 
#endif // REMOTE_FILE_PREFIX


   // if no file name restrictions are in place, give it an automatic pass
   //
#if !defined(REMOTE_FILE_FORWARD_PREFIX) && !defined(REMOTE_FILE_VACATION_PREFIX) && !defined(REMOTE_FILE_PREFIX)
   file_name_ok = 1; 
#endif // !defined(REMOTE_FILE_FORWARD_PREFIX) && !defined(REMOTE_FILE_VACATION_PREFIX) && !defined(REMOTE_FILE_PREFIX)


   if (!file_name_ok)
      fail(ERR_REMOTEFILE, "Remote file name begins with illegal characters");


   // finally, check for illegal characters ANYWHERE in the path
   //
#ifdef REMOTE_FILE_OK_CHARS
#ifdef REMOTE_SUBDIRECTORIES_OK
   if (strspn(file_path, "/" REMOTE_FILE_OK_CHARS) != strlen(file_path))
      fail(ERR_REMOTEFILE, "Remote file name contains illegal character(s)");
#else // REMOTE_SUBDIRECTORIES_OK
   if (strspn(file_path, REMOTE_FILE_OK_CHARS) != strlen(file_path))
      fail(ERR_REMOTEFILE, "Remote file name contains illegal character(s)");
#endif // REMOTE_SUBDIRECTORIES_OK
#endif // REMOTE_FILE_OK_CHARS


#ifdef REMOTE_SUBDIRECTORIES_OK
   free(temp_path);
#endif // REMOTE_SUBDIRECTORIES_OK
   return file_path;

}



/**
  * Validate a "local" file path
  *
  * The path must be a full, absolute path
  * and might be restricted to a certain base
  * if configured as such
  *
  * If the path is invalid, this function fails and exits
  *
  */
static void check_local_file_path(char *file_path)
{

#ifdef LOCAL_FILE_PREFIX
   char error_message[BUFSIZE];
#endif // LOCAL_FILE_PREFIX


   // need absolute path
   //
   if (file_path[0] != '/')
      fail(ERR_LOCALFILE, "Local file path must be an absolute path");


   // restrict base path if needed
   //
#ifdef LOCAL_FILE_PREFIX
   if (strncmp(file_path, LOCAL_FILE_PREFIX, strlen(LOCAL_FILE_PREFIX)) != 0)
   {
      snprintf(error_message, BUFSIZE, "Local file path is restricted to \"%s\"", LOCAL_FILE_PREFIX);
      fail(ERR_LOCALFILE, error_message);
   }
#endif // LOCAL_FILE_PREFIX

}



/**
  * Copy contents of SOURCE_FILE to DEST_FILE, then close
  * both files
  *
  * SOURCE_FILE IN File descriptor to an *already open* source file
  * DEST_FILE   IN File descriptor to an *already open* destination file
  *
  * Returns NULL if copy succeeded, otherwise an error
  * string is returned.
  *
  */
static char *copy_file(int SOURCE_FILE, int DEST_FILE)
{

   ssize_t size;
   char buf[BUFSIZE];


   // loop through, copying data until we reach EOF for the source
   //
   while ((size = read(SOURCE_FILE, buf, BUFSIZE)) > 0)
   {
      if (write(DEST_FILE, buf, size) != size)
      {
         close(SOURCE_FILE);
         close(DEST_FILE);
         return "Error while writing file";
      }
   }

   // make sure there were no read errors
   // 
   if (size == -1)
   {
      close(SOURCE_FILE);
      close(DEST_FILE);
      return "Error reading file";
   }


   // close both file descriptors
   //
   if (close(SOURCE_FILE) == -1 || close(DEST_FILE) == -1)
      return "Error closing files";


   // success
   //
   return NULL;

}



/**
  * Create a string suitable for inclusion in an environment
  * ("VAR=value")
  *
  * variable_name  IN The variable identifier
  * variable_value IN The varialbe's value
  *
  * Returns the needed string (fails and does not return
  * if something goes wrong)
  *
  */
static char *mkenv(char *variable_name, char *variable_value)
{

   size_t return_string_length;
   char *return_string;


   // allocate string memory
   //
   return_string_length = strlen(variable_name) + strlen(variable_value) + 2;
   return_string = malloc(return_string_length);
   if (return_string == NULL)
      fail(ERR_UNEXPECTED, "mkenv(): malloc() failed");


   // create the string
   //
   if (snprintf(return_string, return_string_length, "%s=%s",
                variable_name, variable_value) != return_string_length - 1) // NULL is not counted by snprintf()
      fail(ERR_UNEXPECTED, "mkenv(): snprintf() failed");


   // success
   //
   return return_string;

}



// =================================================== 
// ==================== BEGIN PAM ==================== 
#if defined(USE_PAM_AUTH) || defined(MASTER_USE_PAM_AUTH)


/**
  * This PAM code is mostly taken from inn2.0-beta
  *
  */
static int pass_conv(int num_msg, const struct pam_message **msgm,
                     struct pam_response **response, void *appdata_ptr)
{
   int i;

   *response = malloc(num_msg * sizeof(struct pam_response));
   if (*response == NULL)
      return PAM_CONV_ERR;

   for (i = 0; i < num_msg; i++)
   {
      (*response)[i].resp = (void *) strdup((char *)appdata_ptr);
      (*response)[i].resp_retcode = 0;
   }

   return PAM_SUCCESS;
}



/**
  * Authenticate a user/password pair using PAM
  *
  * Returns 1 upon success, 0 on error/not authenticated
  *
  */
static int auth_pam(char *username, char *password, char *service)
{
   pam_handle_t *pamh;
   struct pam_conv conv;
   int status;

   conv.conv = pass_conv;
   conv.appdata_ptr = password;
   status = pam_start(service, username, &conv, &pamh);
   if (status != PAM_SUCCESS)
   {
      PRINTERROR2("pam_start failed: %s\n", pam_strerror(pamh, status));
      return 0;
   }
   status = pam_authenticate(pamh, PAM_SILENT);
   if (status != PAM_SUCCESS)
   {
      PRINTERROR2("pam_authenticate failed: %s\n", pam_strerror(pamh, status));
      return 0;
   }
   status = pam_acct_mgmt(pamh, PAM_SILENT);
   if (status != PAM_SUCCESS)
   {
      PRINTERROR2("pam_acct_mgmt failed: %s\n", pam_strerror(pamh, status));
      return 0;
   }
   status = pam_end(pamh, status);
   if (status != PAM_SUCCESS)
   {
      PRINTERROR2("pam_end failed: %s\n", pam_strerror(pamh, status));
      return 0;
   }

   // if we get this far, the user successfully authenticated
   //
   return 1;
}


#endif // defined(USE_PAM_AUTH) || defined(MASTER_USE_PAM_AUTH)
// ==================== END PAM ==================== 
// ================================================= 



// ======================================================== 
// ==================== BEGIN C-CLIENT ==================== 
#if defined(USE_CCLIENT_AUTH) || defined(MASTER_USE_CCLIENT_AUTH)


// these are needed in this scope due to the way UW c-client callbacks happen
//
static char *callback_username, *callback_password;


#ifndef CCLIENT_MAILBOX_FLAGS
#define CCLIENT_MAILBOX_FLAGS "/norsh"
#endif



/**
  * Authenticate a user/password pair by attempting
  * to log into an IMAP server using the UW c-client library
  *
  * Returns 1 upon success, 0 on error/not authenticated
  *
  */
static int auth_cclient_imap(char *username, char *password,
                             char *imap_server_address, int imap_port, char *extra)
{

   char mailbox_name[MAILTMPLEN];
   MAILSTREAM *imap_stream;
   DRIVER *driver;


   // we expect these to be accessed before this function returns
   //
   callback_username = username;
   callback_password = password;


   // initialize all of the c-client drivers
   //
   #include "linkage.c"


   // ensure the c-client library supports IMAP
   //
   driver = (DRIVER*)mail_parameters(NIL, GET_DRIVER, (void*)"imap");
   if (!driver)
      fail(ERR_BAD_LIBRARY, "UW c-client library does not support IMAP");


   // reduce timeout/retries (default is 2 seconds in configure.ac)
   //
#ifdef CCLIENT_IMAP_TIMEOUT
   mail_parameters(NIL, SET_OPENTIMEOUT, (void*)CCLIENT_IMAP_TIMEOUT);
   mail_parameters(NIL, SET_READTIMEOUT, (void*)CCLIENT_IMAP_TIMEOUT);
   mail_parameters(NIL, SET_WRITETIMEOUT, (void*)CCLIENT_IMAP_TIMEOUT);
   mail_parameters(NIL, SET_CLOSETIMEOUT, (void*)CCLIENT_IMAP_TIMEOUT);
   mail_parameters(NIL, SET_RSHTIMEOUT, (void*)CCLIENT_IMAP_TIMEOUT);
#endif
   mail_parameters(NIL, SET_MAXLOGINTRIALS, (void*)1);


   // extra may be null
   //
   if (!extra) extra = "";


   // build mailbox string
   //
   snprintf(mailbox_name, MAILTMPLEN, "{%s:%d/service=imap/user=%s%s%s}INBOX",
           imap_server_address, imap_port, username, CCLIENT_MAILBOX_FLAGS, extra);

#ifdef DEBUG
   fprintf(stderr, "Attempting to connect using c-client mailbox_name: %s\n", mailbox_name);
#endif

   // try to connect
   //
   imap_stream = mail_open(NIL, mailbox_name, NIL);

   if (imap_stream != NIL)
   {
      mail_close(imap_stream);
      return 1;
   }
   else
      fail(ERR_USER, "Failed to authenticate user");

}



/**
  * UW c-client IMAP login callback - supply the needed username and password
  *
  */
void mm_login(NETMBX *mailbox, char *username, char *password, long trial)
{

#ifdef DEBUG
   fprintf(stderr, "c-client mm_login: {%s/%s/user=\"%s\"}\n", mailbox->host, mailbox->service, mailbox->user);
#if DEBUG > 1
   fprintf(stderr, "c-client mm_login -> %s %s\n", callback_username, callback_password);
#else
   fprintf(stderr, "c-client mm_login -> %s\n", callback_username);
#endif
#endif

   strncpy(username, callback_username, MAILTMPLEN);
   strncpy(password, callback_password, MAILTMPLEN);

}



/**
  * UW c-client logging callback
  *
  * Currently, we only use this for debugging purposes
  *
  */
void mm_log(char *string, long error_flag)
{

#ifdef DEBUG
   char *error_flag_name;
   switch ((short)error_flag)
   {
      case NIL:     error_flag_name = "NIL";   break;
      case PARSE:   error_flag_name = "PARSE"; break;
      case WARN:    error_flag_name = "WARN";  break;
      case ERROR:   error_flag_name = "ERROR"; break;
      default:      error_flag_name = "?";     break;
   }

   fprintf(stderr, "c-client mm_log: %s: %s\n", error_flag_name, string);
#endif

}



/**
  * UW c-client notify callback
  *
  * Currently, we only use this for debugging purposes
  *
  */
void mm_notify(MAILSTREAM *stream, char *string, long error_flag)
{
   mm_log(string, error_flag);
}



/**
  * These are c-client functions that we don't need to implement
  *
  */
void mm_flags (MAILSTREAM *stream,unsigned long number){}
void mm_status (MAILSTREAM *stream,char *mailbox,MAILSTATUS *status){}
void mm_searched (MAILSTREAM *stream,unsigned long number){}
void mm_exists (MAILSTREAM *stream,unsigned long number){}
void mm_expunged (MAILSTREAM *stream,unsigned long number){}
void mm_list (MAILSTREAM *stream,int delimiter,char *name,long attributes) {}
void mm_lsub (MAILSTREAM *stream,int delimiter,char *name,long attributes) {}
void mm_dlog (char *string){}
void mm_critical (MAILSTREAM *stream) {}
void mm_nocritical (MAILSTREAM *stream) {}
long mm_diskerror (MAILSTREAM *stream,long errcode,long serious) {}
void mm_fatal (char *string) {}


#endif // defined(USE_CCLIENT_AUTH) || defined(MASTER_USE_CCLIENT_AUTH)
// ==================== END C-CLIENT ==================== 
// ====================================================== 



// ==================================================== 
// ==================== BEGIN IMAP ==================== 
#if defined(USE_IMAP_AUTH) || defined(MASTER_USE_IMAP_AUTH)


/**
  * Open and connect to a socket at the given address/port
  *
  * Returns a file descriptor to the socket or -1 upon
  * generic failure or -2 upon failure where a system
  * error should be available
  *
  */
static int socket_connect(const char *host_string, int port)
{
   struct in_addr host_address;
   struct hostent *host_entity;
   struct sockaddr_in socket_address;
   int socket_descriptor;

   // convert host address string into internal
   // data structure representation
   //
   if (inet_aton(host_string, &host_address) == 0)
   {
      if ((host_entity = gethostbyname(host_string)) == NULL)
         return -1;
      host_address = *((struct in_addr *)host_entity->h_addr);
   }

   // prepare the socket
   //
   socket_address.sin_family = AF_INET;
   socket_address.sin_addr = host_address;
   socket_address.sin_port = htons(port);
   memset(&socket_address.sin_zero, 0, 8);

   // open socket
   //
   if ((socket_descriptor = socket(AF_INET, SOCK_STREAM, 0)) == -1)
      return -2;

   // bind/connect to the socket
   //
   if (connect(socket_descriptor, (struct sockaddr *)&socket_address,
               sizeof(struct sockaddr)) == -1)
   {
      close(socket_descriptor);
      return -2;
   }

   // success; return socket descriptor
   //
   return socket_descriptor;
}



/**
  * Authenticate a user/password pair by attempting
  * to log into an IMAP server with them
  *
  * Returns 1 upon success, 0 on error/not authenticated
  *
  */
static int auth_imap(char *username, char *password,
                     char *imap_server_address, int imap_port)
{
   int socket;
   char buf[4096];
   FILE *file_pointer;
 
   // connect to the IMAP server
   //
#ifdef DEBUG
   fprintf(stderr, "Attempting to connect to IMAP address: %s (%d)\n", imap_server_address, imap_port);
#endif // DEBUG
   socket = socket_connect(imap_server_address, imap_port);
   if (socket == -1)
      fail(ERR_SOCKET, "Could not connect to IMAP server");
   if (socket == -2)
      efail(ERR_SOCKET, "Could not connect to IMAP server");
   file_pointer = fdopen(socket, "r+");
 
   if (fgets(buf, sizeof(buf), file_pointer) == NULL)
   {
      fclose(file_pointer);
      efail(ERR_SOCKET, "Could not read from IMAP server socket");
   }

   fprintf(file_pointer, "A1 LOGIN %s %s\r\n", username, password);

   if (fgets(buf, sizeof(buf), file_pointer) == NULL)
   {
      fclose(file_pointer);
      efail(ERR_SOCKET, "Could not read from IMAP server socket");
   }

   // alright, log out nicely
   //
   fprintf(file_pointer, "A2 LOGOUT\r\n");
   fclose(file_pointer);

   // show what the IMAP server had to say if needed
   //
#ifdef DEBUG
   fprintf(stderr, "IMAP response: %s\n", buf);
#endif

   // an "OK" from the IMAP server means the user
   // logged in successfully
   //
   if (strstr(buf, "OK") == NULL)
      return 0;

   // if we get this far, the user successfully authenticated
   //
   return 1;
}


#endif // defined(USE_IMAP_AUTH) || defined(MASTER_USE_IMAP_AUTH)
// ==================== END IMAP ==================== 
// ================================================== 



// ========================================================================= 
// ==================== BEGIN FILE MANAGER TYPE ACTIONS ==================== 
#ifdef ALLOW_FILE_MANAGER_ACTIONS


/**
  * Retrieve a directory listing and store resulting
  * entry names in the given local file, one per line
  *
  * user_info      IN user account information; home
  *                   directory therein tells us where
  *                   to start looking for the directory
  *                   and the UID and GID therein specify
  *                   what user is looking for the directory
  *                   (we assume that UID/GID when
  *                   inspecting the file directory)
  * directory      IN the directory to list
  * dest_file_path IN the path to the local file
  *                   where the directory listing
  *                   should be stored
  * stat_entries   IN when 1, each entry is run through stat()
  *                   and the lines in the output file will
  *                   contain double pipe-separated fields
  *                   describing each directory entry in this order:
  *                         mode
  *                         UID
  *                         GID
  *                         size
  *                         atime
  *                         mtime
  *                         ctime
  *                         path (all but the last path element)
  *                         base name (last path element)
  *                   For example:
  *    33188||510||510||55||1215598228||1209472879||1273860228||subdir/dir2||filename
  * use_lstat      IN when 1, use lstat (instead of stat),
  *                   wherein links themselves are stat'ed
  *                   instead of their targets (only relevant
  *                   when stat_entries is also 1)
  *
  * Returns 0 (ERR_OK) upon success, or non-zero
  * when a problem is encountered (including when
  * the directory is not found (1 (ERR_NOTFOUND)),
  * which may not be a fatal error, depending on
  * the caller's context)
  *
  */
static int list_directory(struct passwd *user_info,
                          char *directory, char *dest_file_path,
                          int stat_entries, int use_lstat)
{

   DIR *DIRECTORY_HANDLE;
   struct dirent *entry;
   int DEST_FILE;
   struct stat stat_info;
   int stat_result;
   char path[BIGBUFSIZE];
   char output_line[BIGBUFSIZE];


#ifdef DEBUG
   fprintf(stderr, "list_directory(): Listing directory: \"%s\" into \"%s\"\n", directory, dest_file_path);
#endif


   // check that the requested path is safe
   //
   directory = check_remote_file_path(directory);
   check_local_file_path(dest_file_path);


   // start in the user's home directory
   //
   // (after this call, our effective and
   // real UID/GID are all that of the
   // target user)
   //
   if (!go_home(user_info, 0, 1))
   {
#ifdef DEBUG
      fprintf(stderr, "list_directory(): Home directory not found\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      free(directory);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return ERR_NOTFOUND;
   }


   // open directory
   //
   if ((DIRECTORY_HANDLE = opendir(directory)) == NULL)
      efail(ERR_SOURCEFILE, "list_directory(): Could not open directory");


   // we want to open the destination (local) file as the calling user
   //
   if (!set_real_and_effective_uid_and_gid(orig_uid, orig_gid, orig_uid, orig_gid))
   {
      closedir(DIRECTORY_HANDLE);
      fail(ERR_PRIVILEGE, "list_directory(): Could not assume previous user privileges");
   }


   // open desination (local) file
   //
   DEST_FILE = open(dest_file_path, O_WRONLY | O_CREAT | O_TRUNC, LOCAL_FILE_MODE);
   if (DEST_FILE == -1)
   {
      closedir(DIRECTORY_HANDLE);
      fail(ERR_DESTFILE, "list_directory(): open() destination file failed");
   }


   // read the directory contents and write them to the output file
   //
   while (entry = readdir(DIRECTORY_HANDLE))
   {

      // get detailed listing?
      //
      if (stat_entries)
      {

         snprintf(path, BIGBUFSIZE, "%s/%s", directory, entry->d_name);
         if (use_lstat)
            stat_result = lstat(path, &stat_info);
         else
            stat_result = stat(path, &stat_info);

         if (stat_result == 0)
         {

            // output line format (double pipe-separated):
            //
            //   mode
            //   UID
            //   GID
            //   size
            //   atime
            //   mtime
            //   ctime
            //   path (all but the last path element)
            //   base name (last path element)
            //
            snprintf(output_line, BIGBUFSIZE, "%d||%d||%d||%d||%d||%d||%d||%s||%s\n",
                     stat_info.st_mode,
                     stat_info.st_uid,
                     stat_info.st_gid,
                     stat_info.st_size,
                     stat_info.st_atime,
                     stat_info.st_mtime,
                     stat_info.st_ctime,
                     directory,
                     entry->d_name);

         }


         // an error occurred with stat(); was it because
         // nothing was found at that path?
         // (use output_line as error message string)
         //
         else if (errno == ENOENT)
         {
            snprintf(output_line, BIGBUFSIZE, "list_directory(): Could not stat() directory entry \"%s\" (not found)", entry->d_name);
            close(DEST_FILE);
            closedir(DIRECTORY_HANDLE);
            fail(ERR_SOURCEFILE, output_line);
         }


         // otherwise, something strange happened
         // (use output_line as error message string)
         //
         else
         {
            snprintf(output_line, BIGBUFSIZE, "list_directory(): Could not stat() directory entry \"%s\"", entry->d_name);
            close(DEST_FILE);
            closedir(DIRECTORY_HANDLE);
            fail(ERR_SOURCEFILE, output_line);
         }

      }


      // simple directory listing of entry names only
      //
      else
         snprintf(output_line, BIGBUFSIZE, "%s\n", entry->d_name);


      // write directory entry to file
      //
      if (write(DEST_FILE, output_line, strlen(output_line)) != strlen(output_line))
      {
         close(DEST_FILE);
         closedir(DIRECTORY_HANDLE);
         fail(ERR_DESTFILE, "list_directory(): Error while writing entry");
      }

   }


   // close file descriptor and directory handle
   //
   if (close(DEST_FILE) == -1)
      efail(ERR_DESTFILE, "list_directory(): Error closing file");
   if (closedir(DIRECTORY_HANDLE) == -1)
      efail(ERR_SOURCEFILE, "list_directory(): Error closing directory");


   // done
   //
#ifdef DEBUG
   fprintf(stderr, "list_directory(): ... OK\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   free(directory);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   return ERR_OK;

}



/**
  * Change permissions on a file or directory
  *
  * user_info      IN user account information; home
  *                   directory therein tells us where
  *                   to start looking for the path and
  *                   the UID and GID therein specify what
  *                   user is looking for the path (we
  *                   assume that UID/GID when inspecting
  *                   the path)
  * path           IN the path to change permissions on
  * mode           IN the desired new (octal) permissions
  *
  * Returns 0 (ERR_OK) upon success, or non-zero
  * when a problem is encountered (including when
  * the path is not found (1 (ERR_NOTFOUND)))
  *
  */
static int change_permissions(struct passwd *user_info, char *path, char *mode_string)
{

   int mode = atoi(mode_string);


#ifdef DEBUG
   fprintf(stderr, "change_permissions(): Changing permissions of \"%s\" to %o\n", path, mode);
#endif


   // check that the requested path is safe
   //
   path = check_remote_file_path(path);


   // start in the user's home directory
   //
   // (after this call, our effective and
   // real UID/GID are all that of the
   // target user)
   //
   if (!go_home(user_info, 0, 0))
   {
#ifdef DEBUG
      fprintf(stderr, "change_permissions(): Home directory not found\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      free(path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return ERR_NOTFOUND;
   }


   // try to change permissions
   //
   if (chmod(path, mode) != 0)
      efail(ERR_SOURCEFILE, "change_permissions(): Could not change permissions");


   // no errors; success
   //
#ifdef DEBUG
   fprintf(stderr, "change_permissions(): ... OK\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   free(path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   return ERR_OK;

}



/**
  * Rename/move a file or directory
  *
  * user_info        IN user account information; home
  *                     directory therein tells us where
  *                     to start looking for the path and
  *                     the UID and GID therein specify what
  *                     user is looking for the path (we
  *                     assume that UID/GID when inspecting
  *                     the path)
  * source_path      IN the original (source) path to move
  * destination_path IN the desired new path specification
  *
  * Returns 0 (ERR_OK) upon success, or non-zero
  * when a problem is encountered
  *
  */
static int rename_file_or_directory(struct passwd *user_info, char *source_path,
                                    char *destination_path)
{

#ifdef DEBUG
   fprintf(stderr, "rename_file_or_directory(): Moving \"%s\" to \"%s\"\n", source_path, destination_path);
#endif


   // check that the requested paths are safe
   //
   source_path = check_remote_file_path(source_path);
   destination_path = check_remote_file_path(destination_path);


   // start in the user's home directory
   //
   // (after this call, our effective and
   // real UID/GID are all that of the
   // target user)
   //
   if (!go_home(user_info, 0, 0))
   {
#ifdef DEBUG
      fprintf(stderr, "rename_file_or_directory(): Home directory not found\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      free(source_path);  // remote paths may have been altered
      free(destination_path);  // remote paths may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return ERR_NOTFOUND;
   }


   // try to rename/move the path
   //
   if (rename(source_path, destination_path) != 0)
      efail(ERR_DESTFILE, "rename_file_or_directory(): Could not move");


   // no errors; success
   //
#ifdef DEBUG
   fprintf(stderr, "rename_file_or_directory(): ... OK\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   free(source_path);  // remote paths may have been altered
   free(destination_path);  // remote paths may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   return ERR_OK;

}



/**
  * Create a directory
  *
  * user_info      IN user account information; home
  *                   directory therein tells us where
  *                   to start looking for the path and
  *                   the UID and GID therein specify what
  *                   user is looking for the path (we
  *                   assume that UID/GID when inspecting
  *                   the path)
  * path           IN the path (directory) to create
  * mode           IN the desired new (octal) permissions
  * recursive      IN when 0, create just one directory - the
  *                   last item in the given path (the rest of
  *                   the path must exist); otherwise, create
  *                   any and all non-existent directories in
  *                   the given path (mkdir -p)
  *
  * Returns 0 (ERR_OK) upon success, or non-zero
  * when a problem is encountered
  *
  */
static int make_directory(struct passwd *user_info, char *path,
                          char *mode_string, int recursive)
{

   int mode = atoi(mode_string);


#ifdef DEBUG
   if (recursive)
      fprintf(stderr, "make_directory(): Creating directory (recursively): \"%s\" with permissions %o\n", path, mode);
   else
      fprintf(stderr, "make_directory(): Creating directory (non-recursively): \"%s\" with permissions %o\n", path, mode);
#endif


   // check that the requested path is safe
   //
   path = check_remote_file_path(path);


   // start in the user's home directory
   //
   // (after this call, our effective and
   // real UID/GID are all that of the
   // target user)
   //
   if (!go_home(user_info, 0, 0))
   {
#ifdef DEBUG
      fprintf(stderr, "make_directory(): Home directory not found\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      free(path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return ERR_NOTFOUND;
   }


   // simple mkdir?
   //
   if (!recursive)
   {
      if (mkdir(path, mode) != 0)
         efail(ERR_SOURCEFILE, "make_directory(): Could not create directory");
   }


   // recursive mkdir
   //
   else
   {
      // technically, this function will never return if there were any errors
      //
      if (!mkdir_recursive(path, mode, user_info))
         fail(ERR_SOURCEFILE, "make_directory(): Could not create directory");
   }


   // no errors; success
   //
#ifdef DEBUG
   fprintf(stderr, "make_directory(): ... OK\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   free(path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   return ERR_OK;

}



//TODO -- if we change this to allow recursive rmdir, should we have a configure flag that allows it? (?? is that necessary??)..................    and have yet another configure argument that allows blasting non-empty directories (rm -rf) (??)
/**
  * Delete a directory
  *
TODO: Possibly merge functions delete_file, delete_directory, delete_file_or_directory
  * user_info IN user account information; home
  *              directory therein tells us where
  *              to start looking for the path and
  *              the UID and GID therein specify what
  *              user is looking for the path (we
  *              assume that UID/GID when inspecting
  *              the path)
  * path      IN the path (directory) to delete
  *
  * Returns 0 (ERR_OK) upon success, or non-zero
  * when a problem is encountered (including when
  * the directory is not found (1 (ERR_NOTFOUND)),
  * which may not be a fatal error, depending on
  * the caller's context)
  *
  */
static int delete_directory(struct passwd *user_info, char *path)
{

   struct stat stat_info;
   char error_message[BUFSIZE];


#ifdef DEBUG
   fprintf(stderr, "delete_directory(): Deleting remote directory \"%s\"\n", path);
#endif


   // safety-check path
   //
   path = check_remote_file_path(path);


   // start in the user's home directory
   //
   // (after this call, our effective and
   // real UID/GID are all that of the
   // target user)
   //
   if (!go_home(user_info, 0, 0))
   {
#ifdef DEBUG
      fprintf(stderr, "delete_directory(): Home directory not found\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      free(path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return ERR_NOTFOUND;
   }


   // detect if the path goes to a directory or link
   // which can be deleted
   //
   if (lstat(path, &stat_info) == 0)
   {

      if (S_ISDIR(stat_info.st_mode))
      {
         if (rmdir(path) != 0)
         {
            snprintf(error_message, BUFSIZE, "delete_directory(): rmdir() \"%s\" failed", path);
            efail(ERR_RMDIR, error_message);
         }
      }
#ifdef S_ISLNK
      else if (S_ISLNK(stat_info.st_mode))
      {

         // for links, we'll only allow them to be deleted if they
         // point to a directory
//TODO: uh, is that too restrictive?
         //
         if (stat(path, &stat_info) == 0)
         {

            if (S_ISDIR(stat_info.st_mode))
            {
               if (unlink(path) == -1)
               {
                  snprintf(error_message, BUFSIZE, "delete_directory(): unlink() \"%s\" failed", path);
                  efail(ERR_UNLINK, error_message);
               }
            }
            else
               fail(ERR_FILETYPE, "delete_directory(): Target is not a directory");

         }
         else
            fail(ERR_UNEXPECTED, "delete_directory(): Unexpected error - lstat() worked but stat() did not");
      }
#endif // S_ISLNK
      else
         fail(ERR_FILETYPE, "delete_directory(): Target is not a directory");

   }
   else if (errno == ENOENT)
   {
#ifdef DEBUG
      fprintf(stderr, "delete_directory(): Directory not found\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      free(path);  // remote path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return ERR_NOTFOUND;
   }
   else
   {
      snprintf(error_message, BUFSIZE, "delete_directory(): lstat() of \"%s\" failed", path);
      efail(ERR_DESTFILE, error_message);
   }


   // success
   //
#ifdef DEBUG
   fprintf(stderr, "delete_directory(): ... OK\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   free(path);  // remote path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   return ERR_OK;

}



//TODO -- if we change this to allow recursive rmdir, should we have a configure flag that allows it? (?? is that necessary??)..................    and have yet another configure argument that allows blasting non-empty directories (rm -rf) (??)
/**
  * Delete a file or directory
  *
TODO: Possibly merge functions delete_file, delete_directory, delete_file_or_directory
  * user_info IN user account information; home
  *              directory therein tells us where
  *              to start looking for the path and
  *              the UID and GID therein specify what
  *              user is looking for the path (we
  *              assume that UID/GID when inspecting
  *              the path)
  * path      IN the path to delete
  *
  * Returns 0 (ERR_OK) upon success, or non-zero
  * when a problem is encountered (including when
  * the file or directory is not found (1
  * (ERR_NOTFOUND)), which may not be a fatal
  * error, depending on the caller's context)
  *
  */
static int delete_file_or_directory(struct passwd *user_info, char *path)
{

   struct stat stat_info;
   char error_message[BUFSIZE];


#ifdef DEBUG
   fprintf(stderr, "delete_file_or_directory(): Deleting remote file or directory \"%s\"\n", path);
#endif


   // safety-check path
   //
   path = check_remote_file_path(path);


   // start in the user's home directory
   //
   // (after this call, our effective and
   // real UID/GID are all that of the
   // target user)
   //
   if (!go_home(user_info, 0, 0))
   {
#ifdef DEBUG
      fprintf(stderr, "delete_file_or_directory(): Home directory not found\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      free(path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return ERR_NOTFOUND;
   }


   // use appropriate delete call depending on
   // what kind of entity the path points to
   //
   if (lstat(path, &stat_info) == 0)
   {

      if (S_ISDIR(stat_info.st_mode))
      {
         if (rmdir(path) != 0)
         {
            snprintf(error_message, BUFSIZE, "delete_file_or_directory(): rmdir() \"%s\" failed", path);
            efail(ERR_RMDIR, error_message);
         }
      }
      else if (S_ISREG(stat_info.st_mode)
#ifdef S_ISLNK
       || S_ISLNK(stat_info.st_mode)
#endif // S_ISLNK
      )
      {
         if (unlink(path) == -1)
         {
            snprintf(error_message, BUFSIZE, "delete_file_or_directory(): unlink() \"%s\" failed", path);
            efail(ERR_UNLINK, error_message);
         }
      }
      else
         fail(ERR_FILETYPE, "delete_file_or_directory(): Target is neither a file nor a directory");

   }
   else if (errno == ENOENT)
   {
#ifdef DEBUG
      fprintf(stderr, "delete_file_or_directory(): File or directory not found\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      free(path);  // remote path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return ERR_NOTFOUND;
   }
   else
   {
      snprintf(error_message, BUFSIZE, "delete_file_or_directory(): lstat() of \"%s\" failed", path);
      efail(ERR_DESTFILE, error_message);
   }


   // success
   //
#ifdef DEBUG
   fprintf(stderr, "delete_file_or_directory(): ... OK\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   free(path);  // remote path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   return ERR_OK;

}


#endif // ALLOW_FILE_MANAGER_ACTIONS
// ==================== END FILE MANAGER TYPE ACTIONS ==================== 
// ======================================================================= 



// ======================================================= 
// ==================== BEGIN ACTIONS ==================== 


/**
  * stat() a path and store resulting information
  * in the given local file
  *
  * Specifically, these items are stored on each
  * line in the output file:
  *
  *   base name (last path element)
  *   path (all but the last path element)
  *   mode
  *   UID
  *   GID
  *   size
  *   atime
  *   mtime
  *   ctime
  *
  * user_info      IN user account information; home
  *                   directory therein tells us where
  *                   to start looking for the path
  *                   and the UID and GID therein specify
  *                   what user is looking for the path
  *                   (we assume that UID/GID when
  *                   inspecting the file path)
  * path           IN the path to stat
  * dest_file_path IN the path to the local file
  *                   where the stat information
  *                   should be stored
  * use_lstat      IN when 1, use lstat (instead of stat),
  *                   wherein links themselves are stat'ed
  *                   instead of their targets
  *
  * Returns 0 (ERR_OK) upon success, or non-zero
  * when a problem is encountered (including when
  * the path is not found (1 (ERR_NOTFOUND)),
  * which may not be a fatal error, depending on
  * the caller's context)
  *
  */
static int stat_path(struct passwd *user_info, char *path,
                     char *dest_file_path, int use_lstat)
{

   struct stat stat_info;
   int stat_result;
   int DEST_FILE;
   char file_output[BIGBUFSIZE];
   char *temp_path1 = strdup(path);
   char *temp_path2 = strdup(path);


#ifdef DEBUG
   if (use_lstat)
      fprintf(stderr, "stat_path(): Using lstat() to check path: \"%s\" into \"%s\"\n", path, dest_file_path);
   else
      fprintf(stderr, "stat_path(): Using stat() to check path: \"%s\" into \"%s\"\n", path, dest_file_path);
#endif


   // check that the requested path is safe
   //
   path = check_remote_file_path(path);
   check_local_file_path(dest_file_path);


   // start in the user's home directory
   //
   // (after this call, our effective and
   // real UID/GID are all that of the
   // target user)
   //
   if (!go_home(user_info, 0, 1))
   {
#ifdef DEBUG
      fprintf(stderr, "stat_path(): Home directory not found\n");
#endif
      free(temp_path1);
      free(temp_path2);
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      free(path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return ERR_NOTFOUND;
   }


   // test for the path
   //
   if (use_lstat)
      stat_result = lstat(path, &stat_info);
   else
      stat_result = stat(path, &stat_info);

   if (stat_result == 0)
   {

      // we want to open the destination (local) file as the calling user
      //
      if (!set_real_and_effective_uid_and_gid(orig_uid, orig_gid, orig_uid, orig_gid))
         efail(ERR_PRIVILEGE, "stat_path(): Could not assume previous user privileges");


      // open desination (local) file
      //
      DEST_FILE = open(dest_file_path, O_WRONLY | O_CREAT | O_TRUNC, LOCAL_FILE_MODE);
      if (DEST_FILE == -1)
         efail(ERR_DESTFILE, "stat_path(): open() destination file failed");


      // file output (per line):
      //
      //   base name (last path element)
      //   path (all but the last path element)
      //   mode
      //   UID
      //   GID
      //   size
      //   atime
      //   mtime
      //   ctime
      //
      snprintf(file_output, BIGBUFSIZE, "%s\n%s\n%d\n%d\n%d\n%d\n%d\n%d\n%d",
               basename(temp_path1), // use temp_path because basename() may modify args
               dirname(temp_path2), // use temp_path because dirname() may modify args
               stat_info.st_mode,
               stat_info.st_uid,
               stat_info.st_gid,
               stat_info.st_size,
               stat_info.st_atime,
               stat_info.st_mtime,
               stat_info.st_ctime);
      if (write(DEST_FILE, file_output, strlen(file_output)) != strlen(file_output))
      {
         close(DEST_FILE);
         fail(ERR_DESTFILE, "stat_path(): Error while writing file");
      }


      // close file descriptor
      //
      if (close(DEST_FILE) == -1)
         efail(ERR_DESTFILE, "stat_path(): Error closing file");

   }


   // an error occurred with stat(); was it because
   // nothing was found at that path?
   //
   else if (errno == ENOENT)
   {
#ifdef DEBUG
      fprintf(stderr, "stat_path(): ... Not found\n", path);
#endif
      free(temp_path1);
      free(temp_path2);
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      free(path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return ERR_NOTFOUND;
   }


   // otherwise, something strange happened
   //
   else
      efail(ERR_SOURCEFILE, "Problem with stat()");


   // no errors; path exists
   //
#ifdef DEBUG
   fprintf(stderr, "stat_path(): ... OK\n");
#endif
   free(temp_path1);
   free(temp_path2);
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   free(path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   return ERR_OK;

}



/**
  * Determines if a path exists
  *
  * user_info IN user account information; home
  *              directory therein tells us where
  *              to start looking for the path
  *              and the UID and GID therein specify
  *              what user is looking for the path
  *              (we assume that UID/GID when
  *              inspecting the file path)
  * path      IN the path to look for
  * is_file   IN when 1, the path must point to a
  *              regular file; otherwise, function
  *              should fail
  * is_dir    IN when 1, the path must point to a
  *              directory; otherwise, function
  *              should fail
  *
  * Returns 0 (ERR_OK) upon success, or non-zero
  * when a problem is encountered (including when
  * the path is not found (1 (ERR_NOTFOUND)),
  * which may not be a fatal error, depending on
  * the caller's context)
  *
  */
static int path_exists(struct passwd *user_info, char *path,
                       int is_file, int is_dir)
{

   struct stat stat_info;


   // is_file and is_dir are mutually exclusive
   //
   if (is_file && is_dir)
      fail(ERR_UNEXPECTED, "path_exists() cannot be called with both is_file and is_dir");


#ifdef DEBUG
   if (is_file)
      fprintf(stderr, "path_exists(): Checking for regular file at path: %s\n", path);
   else if (is_dir)
      fprintf(stderr, "path_exists(): Checking for directory at path: %s\n", path);
   else
      fprintf(stderr, "path_exists(): Checking for path: %s\n", path);
#endif


   // check that the requested path is safe
   //
   path = check_remote_file_path(path);


   // start in the user's home directory
   //
   // (after this call, our effective and
   // real UID/GID are all that of the
   // target user)
   //
   if (!go_home(user_info, 0, 0))
   {
#ifdef DEBUG
      fprintf(stderr, "path_exists(): Home directory not found\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      free(path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return ERR_NOTFOUND;
   }


   // test for the path
   //
   if (stat(path, &stat_info) == 0)
   {

      // we found *something*, if needed, check to see if it's a file
      // 
      if (is_file && !S_ISREG(stat_info.st_mode))
         fail(ERR_FILETYPE, "path_exists(): Target is not a regular file");


      // or if needed, check to see if it's a directory
      // 
      if (is_dir && !S_ISDIR(stat_info.st_mode))
         fail(ERR_FILETYPE, "path_exists(): Target is not a direcory");

   }


   // an error occurred with stat(); was it because
   // nothing was found at that path?
   //
   else if (errno == ENOENT)
   {
#ifdef DEBUG
      fprintf(stderr, "path_exists(): ... Not found\n", path);
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      free(path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return ERR_NOTFOUND;
   }


   // otherwise, something strange happened
   //
   else
      efail(ERR_SOURCEFILE, "Problem with stat()");


   // no errors; path exists
   //
#ifdef DEBUG
   fprintf(stderr, "path_exists(): ... Yes\n", path);
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   free(path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   return ERR_OK;

}



/**
  * Retrieve a file's contents, copying them into
  * the specified "local" file
  *
  * user_info IN user account information; home
  *              directory therein tells us where
  *              to start looking for the file
  *              and the UID and GID therein specify
  *              what user is looking for the file
  *              (we assume that UID/GID when
  *              inspecting the file path)
  * source_file_path IN the path to the file to retrieve
  * dest_file_path   IN the path to the file where the
  *                     copy should be placed
  *
  * Returns 0 (ERR_OK) upon success, or non-zero
  * when a problem is encountered (including when
  * the file is not found (1 (ERR_NOTFOUND)),
  * which may not be a fatal error, depending on
  * the caller's context)
  *
  */
static int get_file(struct passwd *user_info, char *source_file_path,
                    char *dest_file_path)
{

   int SOURCE_FILE, DEST_FILE;
   char *copy_result;


#ifdef DEBUG
   fprintf(stderr, "get_file(): Retrieving remote file \"%s\" into local file \"%s\"\n", source_file_path, dest_file_path);
#endif


   // safety-check file paths
   //
   source_file_path = check_remote_file_path(source_file_path);
   check_local_file_path(dest_file_path);


   // start in the user's home directory
   //
   // (after this call, our effective and
   // real UID/GID are all that of the
   // target user)
   //
   if (!go_home(user_info, 0, 1))
   {
#ifdef DEBUG
      fprintf(stderr, "get_file(): Home directory not found\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      free(source_file_path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return ERR_NOTFOUND;
   }


   // open source (remote) file
   //
   SOURCE_FILE = open(source_file_path, O_RDONLY);
   if (SOURCE_FILE == -1)
   {
      if (errno == ENOENT)
      {
#ifdef DEBUG
         fprintf(stderr, "get_file(): File not found\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
         free(source_file_path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
         return ERR_NOTFOUND;
      }
      else
         efail(ERR_SOURCEFILE, "get_file(): open() source file failed");
   }


   // we want to open the destination (local) file as the calling user
   //
   if (!set_real_and_effective_uid_and_gid(orig_uid, orig_gid, orig_uid, orig_gid))
      efail(ERR_PRIVILEGE, "get_file(): Could not assume previous user privileges");


   // open desination (local) file
   //
   DEST_FILE = open(dest_file_path, O_WRONLY | O_CREAT | O_TRUNC, LOCAL_FILE_MODE);
   if (DEST_FILE == -1)
      efail(ERR_DESTFILE, "get_file(): open() destination file failed");


   // copy contents
   //
   if ((copy_result = copy_file(SOURCE_FILE, DEST_FILE)) != NULL)
   {
      unlink(dest_file_path);  // remove incomplete destination file
      fail(ERR_COPYFILE, copy_result);
   }


   // success
   //
#ifdef DEBUG
   fprintf(stderr, "get_file(): ... OK\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   free(source_file_path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   return ERR_OK;

}



/**
  * "Put" file: Transfer a "local" file's contents
  * into the specified "remote" file
  *
  * user_info IN user account information; home
  *              directory therein tells us where
  *              to start looking for the file
  *              and the UID and GID therein specify
  *              what user is looking for the file
  *              (we assume that UID/GID when
  *              inspecting the file path)
  * source_file_path IN the path to the file to retrieve
  * dest_file_path   IN the path to the file where the
  *                     copy should be placed
  *
  * Returns 0 (ERR_OK) upon success, or non-zero
  * when a problem is encountered (including when
  * the file is not found (1 (ERR_NOTFOUND)),
  * which may not be a fatal error, depending on
  * the caller's context)
  *
  */
static int put_file(struct passwd *user_info, char *source_file_path,
                    char *dest_file_path)
{

   int SOURCE_FILE, DEST_FILE;
   char *copy_result;


#ifdef DEBUG
   fprintf(stderr, "put_file(): Storing local file \"%s\" into remote file \"%s\"\n", source_file_path, dest_file_path);
#endif


   // safety-check file paths
   //
   check_local_file_path(source_file_path);
   dest_file_path = check_remote_file_path(dest_file_path);


   // open source (local) file as calling user
   //
   if (!set_real_and_effective_uid_and_gid(0, orig_gid, orig_uid, orig_gid))
      efail(ERR_PRIVILEGE, "put_file(): Could not change privileges");

   SOURCE_FILE = open(source_file_path, O_RDONLY);
   if (SOURCE_FILE == -1)
   {
      if (errno == ENOENT)
      {
#ifdef DEBUG
         fprintf(stderr, "put_file(): File not found\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
         free(dest_file_path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
         return ERR_NOTFOUND;
      }
      else
         efail(ERR_SOURCEFILE, "put_file(): open() source file failed");
   }


   // change to the user's home directory and
   // assume the user's privileges for opening
   // the remote file
   //
   // (after this call, our effective and
   // real UID/GID are all that of the
   // target user)
   //
   if (!go_home(user_info, 1, 0))
      fail(ERR_DESTFILE, "put_file(): unknown error");  // should never be possible to get here


   // open destination (remote) file as target user
   //
   DEST_FILE = open(dest_file_path, O_WRONLY | O_CREAT | O_TRUNC, REMOTE_FILE_MODE);
   if (DEST_FILE == -1)
      efail(ERR_DESTFILE, "put_file(): open() destination file failed");


   // copy contents
   //
   if ((copy_result = copy_file(SOURCE_FILE, DEST_FILE)) != NULL)
   {
      unlink(dest_file_path);  // remove incomplete destination file
      fail(ERR_COPYFILE, copy_result);
   }


   // success
   //
#ifdef DEBUG
   fprintf(stderr, "put_file(): ... OK\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   free(dest_file_path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   return ERR_OK;

}



/**
  * Delete (remote) file
  *
TODO: Possibly merge functions delete_file, delete_directory, delete_file_or_directory
  * user_info IN user account information; home
  *              directory therein tells us where
  *              to find the file and the UID and
  *              GID therein specify what user owns
  *              the file (we assume that UID/GID
  *              when deleting it)
  * file_path IN the path to the file to delete
  *
  * Returns 0 (ERR_OK) upon success, or non-zero
  * when a problem is encountered (including when
  * the file is not found (1 (ERR_NOTFOUND)),
  * which may not be a fatal error, depending on
  * the caller's context)
  *
  */
static int delete_file(struct passwd *user_info, char *file_path)
{

   struct stat stat_info;
   char error_message[BUFSIZE];


#ifdef DEBUG
   fprintf(stderr, "delete_file(): Deleting remote file \"%s\"\n", file_path);
#endif


   // safety-check file path
   //
   file_path = check_remote_file_path(file_path);


   // start in the user's home directory
   //
   // (after this call, our effective and
   // real UID/GID are all that of the
   // target user)
   //
   if (!go_home(user_info, 0, 0))
   {
#ifdef DEBUG
      fprintf(stderr, "delete_file(): Home directory not found\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      free(file_path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return ERR_NOTFOUND;
   }


   // detect if the path goes to a file or link
   // which can be deleted
   //
   if (lstat(file_path, &stat_info) == 0)
   {

      if (S_ISREG(stat_info.st_mode)
#ifdef S_ISLNK
       || S_ISLNK(stat_info.st_mode)
#endif // S_ISLNK
      )
      {
//TODO: we could first check that if it's a link that it points to a real file before allowing detele...  *SHOULD* we be doing that??
         if (unlink(file_path) == -1)
         {
            snprintf(error_message, BUFSIZE, "delete_file(): unlink() \"%s\" failed", file_path);
            efail(ERR_UNLINK, error_message);
         }
      }
      else
         fail(ERR_FILETYPE, "delete_file(): Target is not a regular file");

   }
   else if (errno == ENOENT)
   {
#ifdef DEBUG
      fprintf(stderr, "delete_file(): File not found\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      free(file_path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
      return ERR_NOTFOUND;
   }
   else
   {
      snprintf(error_message, BUFSIZE, "delete_file(): lstat() of \"%s\" failed", file_path);
      efail(ERR_DESTFILE, error_message);
   }


   // success
   //
#ifdef DEBUG
   fprintf(stderr, "delete_file(): ... OK\n");
#endif
#ifdef AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   free(file_path);  // remote file path may have been altered
#endif // AUTO_ADJUST_ABSOLUTE_REMOTE_PATH
   return ERR_OK;

}


// ==================== END ACTIONS ==================== 
// ===================================================== 



// ======================================================== 
// ==================== BEGIN VACATION ==================== 
#ifdef VACATION_PATH


/**
  * Execute the "vacation" program in order to initialize
  * a user's autoresponder
  *
  * Typically, this means we will be running "vacation -I".
  *
  * The environment in which it is run is stripped down to
  * just having a USER, PATH and HOME -- no other variables.
  *
  * user_info    IN user account information; home
  *                 directory therein tells us from
  *                 where to execute the "vacation"
  *                 program and the UID and GID to
  *                 use while doing so.
  * use_interval IN when 1, specify the reply interval
  *                 when initializing... the desired
  *                 interval is given as the "interval"
  *                 argument as follows (otherwise, the
  *                 "interval" argument is ignored)
  * interval     IN the interval to be used when
  *                 initializing (only if "use_interval
  *                 is 1) (commonly, this will be the
  *                 number of days, but can also be
  *                 "infinite" - this depends entirely
  *                 on your "vacation" implementation)
  *                 NOTE also that VACATION_INTERVAL_FLAG
  *                 must be compiled in for intervals
  *                 to work at all.
  *
  * Under normal circumstances, this will never return - the
  * current process will be replaced with the executing "vacation"
  * program.  If an error occurs, an error status code (non-zero)
  * may be returned.
  *
  */
static int vacation_init(struct passwd *user_info, int use_interval, char *interval)
{

   int i;
   char *vacation_environment_variables[4];
   char *vacation_arguments[6];

#ifdef VACATION_USE_RESOLVED_HOME
   char current_directory[BUFSIZE], *current_directory_pointer;
#endif // VACATION_USE_RESOLVED_HOME


#ifdef DEBUG
   fprintf(stderr, "vacation_init(): Initializing autoresponder\n");
#endif


   // start in the user's home directory
   //
   // (after this call, our effective and
   // real UID/GID are all that of the
   // target user)
   //
   if (!go_home(user_info, 1, 0))
   {
#ifdef DEBUG
      fprintf(stderr, "vacation_init(): Home directory not found\n");
#endif
      return ERR_NOTFOUND;
   }


   // argument counter
   //
   i = 0;


   // add the path to the "vacation" program as
   // the first argument as is customary
   //
   vacation_arguments[i++] = VACATION_PATH;


   // specify the needed "vacation" argument
   //
#ifndef VACATION_INIT_FLAG
#define VACATION_INIT_FLAG "-I"
#endif // !VACATION_INIT_FLAG
   vacation_arguments[i++] = VACATION_INIT_FLAG;


#ifdef VACATION_INTERVAL_FLAG
   if (use_interval)
   {
      vacation_arguments[i++] = VACATION_INTERVAL_FLAG;
      vacation_arguments[i++] = interval;
   }
#endif // VACATION_INTERVAL_FLAG


   // if needed, also pass the username to "vacation"
   //
#ifdef VACATION_USER_ARGUMENT
   vacation_arguments[i++] = user_info->pw_name;
#endif // VACATION_USER_ARGUMENT


   // make sure to NULL-terminate argument list
   //
   vacation_arguments[i++] = NULL;


   // environment counter
   //
   i = 0;


   // construct environment PATH variable
   //
#ifndef VACATION_ENVIRONMENT_PATH
#define VACATION_ENVIRONMENT_PATH "/bin:/usr/bin"
#endif // VACATION_ENVIRONMENT_PATH
   vacation_environment_variables[i++] = "PATH=" VACATION_ENVIRONMENT_PATH;


   // construct environment HOME variable
   //
#ifdef VACATION_USE_RESOLVED_HOME

   // in this case, use the current working directory
   //
   // NOTE that if HOME_DIR_PREFIX or OVERRIDE_HOME_DIR or
   // other home directory manipulations have been used,
   // the "vacation" program will probably have to be
   // customized to know how to find the user's home
   // directory in the same way
   //
   current_directory_pointer = getcwd(current_directory, BUFSIZE);
   if (!current_directory_pointer)
      efail(ERR_UNEXPECTED, "vacation_init(): getcwd() failed");
   vacation_environment_variables[i++] = mkenv("HOME", current_directory_pointer);

#else // VACATION_USE_RESOLVED_HOME

   // just set HOME to user's looked-up home directory
   //
   vacation_environment_variables[i++] = mkenv("HOME", user_info->pw_dir);
#endif // VACATION_USE_RESOLVED_HOME


   // construct environment USER variable
   //
   vacation_environment_variables[i++] = mkenv("USER", user_info->pw_name);


   // make sure to NULL-terminate environment variable list
   //
   vacation_environment_variables[i++] = NULL;


   // now execute the vacation program... note that execve()
   // never returns if all is well
   //
#ifdef DEBUG
   fprintf(stderr, "vacation_init(): Calling \"vacation\" program; process being replaced\n");
#endif
   execve(VACATION_PATH, vacation_arguments, vacation_environment_variables);
   efail(ERR_EXEC, "vacation_init(): execve() failed");

}


#endif // VACATION_PATH
// ==================== END VACATION ==================== 
// ====================================================== 



// ========================================================== 
// ==================== BEGIN SQL (ODBC) ==================== 
#ifdef USER_LOOKUP_SQL


/**
  * Concatenate optional SQL diagnostic message (when DEBUG
  * is enabled) onto the given message string(s) and return
  *
  */
static char *get_sql_diagnostics(char *message, char *debug_message,
                                 SQLHANDLE handle, SQLSMALLINT handle_type)
{

   // get additional diagnostic messages if needed
   //
#ifdef DEBUG
   SQLCHAR sql_state[7];
   SQLINTEGER diagnostic_code;
   SQLCHAR *diagnostic_message;
   SQLSMALLINT diagnostic_message_len;
   SQLRETURN sql_return_value;
   char *combined_message;
   char *full_message;
   int full_message_len;


   // first, just combine regular and debug messages
   //
   combined_message = malloc(strlen(message) + strlen(debug_message) + 4);
   sprintf(combined_message, "%s (%s)", message, debug_message);


   // now just get diagnostic message length so we can allocate a buffer for it
   //
   sql_return_value = SQLGetDiagRec(handle_type, handle, 1, sql_state,
                                    &diagnostic_code, NULL, 0,
                                    &diagnostic_message_len);
   if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
   {
      // give up; can't get diagnostic message
      return combined_message;
   }
   else
   {

      // now we can get the message
      //
      diagnostic_message = malloc(diagnostic_message_len + 1);  // +1 for null termination
      sql_return_value = SQLGetDiagRec(handle_type, handle, 1, sql_state,
                                       &diagnostic_code, diagnostic_message,
                                       diagnostic_message_len + 1, NULL);
      if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
      {
         // give up; can't get diagnostic message
         return combined_message;
      }
      else
      {
         // (add 50 to length for null termination and format string in
         // the sprintf() below... shouldn't need more than that ever
         // unless unixODBC has some really strange diagnostic codes)
         //
         full_message_len = diagnostic_message_len + 50 + strlen(combined_message);
         full_message = malloc(full_message_len);
         snprintf(full_message, full_message_len, "%s: (%d) %s\n", combined_message,
                                                                   diagnostic_code,
                                                                   diagnostic_message);
         return full_message;
      }
   }


#else // DEBUG
   return message;


#endif // DEBUG

}



/**
  * Connect to a ODBC database and query for user account info
  *
  */
static struct passwd *look_up_user_sql(char *username, char *query, char *dsn)
{

   struct passwd *user_info;

   SQLRETURN sql_return_value;
   SQLHENV sql_environment_handle;
   SQLHDBC sql_connection_handle;
   SQLHSTMT sql_statement_handle;
   char *message;
   char *debug_message;
   int message_len;
   char *final_query;
   char *pos;
   SQLINTEGER row_count;
   SQLSMALLINT column_count;
   char *password = malloc(BUFSIZE);
   char *home_directory = malloc(BIGBUFSIZE);
   SQLUINTEGER uid;
   SQLUINTEGER gid;


   // don't allow any quotes in username - easy way to avoid SQL injection
   //
   if (strstr(username, "'") != NULL || strstr(username, "\"") != NULL)
      fail(ERR_USER, "Bad characters in username");


   // allocate environment handle
   //
   sql_return_value = SQLAllocHandle(SQL_HANDLE_ENV, SQL_NULL_HANDLE, &sql_environment_handle);
   if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
   {
      message = get_sql_diagnostics("Failed to allocate environment handle", "", sql_environment_handle, SQL_HANDLE_ENV);
      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
      fail(ERR_SQL_ALLOC_ENV, message);
   }


   // declare ODBC v3.0
   //
   sql_return_value = SQLSetEnvAttr(sql_environment_handle, SQL_ATTR_ODBC_VERSION, (void*)SQL_OV_ODBC3, 0);
   if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
   {
      message = get_sql_diagnostics("SQLSetEnvAttr() failed", "", sql_environment_handle, SQL_HANDLE_ENV);
      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
      fail(ERR_SQL_SET_ENV_ATTR, message);
   }


   // allocate connection handle
   //
   sql_return_value = SQLAllocHandle(SQL_HANDLE_DBC, sql_environment_handle, &sql_connection_handle);
   if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
   {
      message = get_sql_diagnostics("Failed to allocate connection handle", "", sql_environment_handle, SQL_HANDLE_ENV);
      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
      fail(ERR_SQL_ALLOC_CONN, message);
   }


// TODO: do we want to set a login timeout value?
//   sq_return_value = SQLSetConnectAttr(sql_connection_handle, SQL_LOGIN_TIMEOUT, (SQLPOINTER *)5, 0);
//   if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
//   {
//      message = get_sql_diagnostics("SQLSetConnectAttr() failed", "", sql_connection_handle, SQL_HANDLE_DBC);
//      SQLFreeHandle(SQL_HANDLE_DBC, sql_connection_handle);
//      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
//      fail(ERR_SQL_SET_CONN_ATTR, message);
//   }


   // connect to desired DSN
   //
   sql_return_value = SQLConnect(sql_connection_handle, dsn, SQL_NTS, NULL, SQL_NTS, NULL, SQL_NTS);
   if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
   {
      debug_message = malloc(strlen(dsn) + 3);
      sprintf(debug_message, "\"%s\"", dsn);
      message = get_sql_diagnostics("Failed to connect to data source", debug_message, sql_connection_handle, SQL_HANDLE_DBC);
      SQLFreeHandle(SQL_HANDLE_DBC, sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
      fail(ERR_SQL_CONNECT, message);
   }


   // allocate statement handle
   //
   sql_return_value = SQLAllocHandle(SQL_HANDLE_STMT, sql_connection_handle, &sql_statement_handle);
   if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
   {
      message = get_sql_diagnostics("Failed to allocate statement handle", "", sql_connection_handle, SQL_HANDLE_DBC);
      SQLDisconnect(sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_DBC, sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
      fail(ERR_SQL_ALLOC_STMT, message);
   }


   // replace "%u" in the query with the username
   //
   final_query = malloc(strlen(query) + strlen(username));
   if ((pos = strstr(query, "%u")) != NULL)
   {
      strncpy(final_query, query, pos - query);
      final_query[pos - query] = '\0';
      strcat(final_query, username);
      strcat(final_query, pos + 2);  // 2 is strlen("%u")
   }
   else
      strcpy(final_query, query);


   // execute the database query
   //
   sql_return_value = SQLExecDirect(sql_statement_handle, final_query, SQL_NTS);
   if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
   {
      debug_message = malloc(strlen(final_query) + 3);
      sprintf(debug_message, "\"%s\"", final_query);
      message = get_sql_diagnostics("Failed to query database", debug_message, sql_statement_handle, SQL_HANDLE_STMT);
      SQLFreeHandle(SQL_HANDLE_STMT, sql_statement_handle);
      SQLDisconnect(sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_DBC, sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
      fail(ERR_SQL_QUERY, message);
   }


   // get number of rows
   //
   sql_return_value = SQLRowCount(sql_statement_handle, &row_count);
   if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
   {
      debug_message = malloc(strlen(final_query) + 3);
      sprintf(debug_message, "\"%s\"", final_query);
      message = get_sql_diagnostics("Could not retrieve the number of rows for query", debug_message, sql_statement_handle, SQL_HANDLE_STMT);
      SQLFreeHandle(SQL_HANDLE_STMT, sql_statement_handle);
      SQLDisconnect(sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_DBC, sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
      fail(ERR_SQL_GET_ROWS, message);
   }


   // get number of columns
   //
   sql_return_value = SQLNumResultCols(sql_statement_handle, &column_count);
   if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
   {
      debug_message = malloc(strlen(final_query) + 3);
      sprintf(debug_message, "\"%s\"", final_query);
      message = get_sql_diagnostics("Could not retrieve the number of columns for query", debug_message, sql_statement_handle, SQL_HANDLE_STMT);
      SQLFreeHandle(SQL_HANDLE_STMT, sql_statement_handle);
      SQLDisconnect(sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_DBC, sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
      fail(ERR_SQL_GET_COLS, message);
   }


   // one and only one row with five columns should be
   // returned - anything else is an error condition
   //
   if (row_count != 1)
   {
#ifdef DEBUG
      message_len = strlen(final_query) + 75;
      message = malloc(message_len);
      snprintf(message, message_len, "Query returned wrong number of rows (\"%s\" returned %d rows)", final_query, row_count);
#else // DEBUG
      message = "Query returned wrong number of rows";
#endif // DEBUG
      SQLFreeHandle(SQL_HANDLE_STMT, sql_statement_handle);
      SQLDisconnect(sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_DBC, sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
      fail(ERR_SQL_BAD_ROWS, message);
   }
   if (column_count != 5)
   {
#ifdef DEBUG
      message_len = strlen(final_query) + 81;
      message = malloc(message_len);
      snprintf(message, message_len, "Query returned wrong number of columns (\"%s\" returned %d columns)", final_query, column_count);
#else // DEBUG
      message = "Query returned wrong number of columns";
#endif // DEBUG
      SQLFreeHandle(SQL_HANDLE_STMT, sql_statement_handle);
      SQLDisconnect(sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_DBC, sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
      fail(ERR_SQL_BAD_COLS, message);
   }


   // columns are expected to be in this order:
   // user name, password, UID, GID and home directory
   //
   sql_return_value = SQLFetch(sql_statement_handle);
   if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
   {
      debug_message = malloc(strlen(final_query) + 3);
      sprintf(debug_message, "\"%s\"", final_query);
      message = get_sql_diagnostics("Could not retrieve query data", debug_message, sql_statement_handle, SQL_HANDLE_STMT);
      SQLFreeHandle(SQL_HANDLE_STMT, sql_statement_handle);
      SQLDisconnect(sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_DBC, sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
      fail(ERR_SQL_GET_DATA, message);
   }


   // get password
   //
   sql_return_value = SQLGetData(sql_statement_handle, 2, SQL_C_CHAR, password, BUFSIZE, NULL);
   if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
   {
      debug_message = malloc(strlen(final_query) + 3);
      sprintf(debug_message, "\"%s\"", final_query);
      message = get_sql_diagnostics("Could not retrieve account password", debug_message, sql_statement_handle, SQL_HANDLE_STMT);
      SQLFreeHandle(SQL_HANDLE_STMT, sql_statement_handle);
      SQLDisconnect(sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_DBC, sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
      fail(ERR_SQL_GET_DATA, message);
   }
   

   // get UID
   //
   sql_return_value = SQLGetData(sql_statement_handle, 3, SQL_C_ULONG, &uid, sizeof(uid), NULL);
   if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
   {
      debug_message = malloc(strlen(final_query) + 3);
      sprintf(debug_message, "\"%s\"", final_query);
      message = get_sql_diagnostics("Could not retrieve account UID", debug_message, sql_statement_handle, SQL_HANDLE_STMT);
      SQLFreeHandle(SQL_HANDLE_STMT, sql_statement_handle);
      SQLDisconnect(sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_DBC, sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
      fail(ERR_SQL_GET_DATA, message);
   }
   

   // get GID
   //
   sql_return_value = SQLGetData(sql_statement_handle, 4, SQL_C_ULONG, &gid, sizeof(gid), NULL);
   if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
   {
      debug_message = malloc(strlen(final_query) + 3);
      sprintf(debug_message, "\"%s\"", final_query);
      message = get_sql_diagnostics("Could not retrieve account GID", debug_message, sql_statement_handle, SQL_HANDLE_STMT);
      SQLFreeHandle(SQL_HANDLE_STMT, sql_statement_handle);
      SQLDisconnect(sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_DBC, sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
      fail(ERR_SQL_GET_DATA, message);
   }
   

   // get home directory
   //
   sql_return_value = SQLGetData(sql_statement_handle, 5, SQL_C_CHAR, home_directory, BIGBUFSIZE, NULL);
   if ((sql_return_value != SQL_SUCCESS) && (sql_return_value != SQL_SUCCESS_WITH_INFO))
   {
      debug_message = malloc(strlen(final_query) + 3);
      sprintf(debug_message, "\"%s\"", final_query);
      message = get_sql_diagnostics("Could not retrieve account home directory", debug_message, sql_statement_handle, SQL_HANDLE_STMT);
      SQLFreeHandle(SQL_HANDLE_STMT, sql_statement_handle);
      SQLDisconnect(sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_DBC, sql_connection_handle);
      SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);
      fail(ERR_SQL_GET_DATA, message);
   }
   

   // done with the database connection
   //
   SQLFreeHandle(SQL_HANDLE_STMT, sql_statement_handle);
   SQLDisconnect(sql_connection_handle);
   SQLFreeHandle(SQL_HANDLE_DBC, sql_connection_handle);
   SQLFreeHandle(SQL_HANDLE_ENV, sql_environment_handle);


   // fill in a passwd struct with the data from the database
   //
   user_info = (struct passwd *)malloc(sizeof(struct passwd));
   user_info->pw_name = username;
   user_info->pw_passwd = password;
   user_info->pw_uid = uid;
   user_info->pw_gid = gid;
   user_info->pw_dir = home_directory;


   // done
   //
   free(final_query);
   return user_info;

}


#endif // USER_LOOKUP_SQL
// ==================== END SQL (ODBC) ==================== 
// ======================================================== 



/**
  * Look up user information: UID, GID, user name, password
  *
  * Returns a passwd struct with at least the
  * username (pw_name), UID (pw_uid), GID (pw_gid),
  * and "home" directory (pw_dir) accurately
  * populated.  These values may or may not correspond
  * to a real user account on the system depending on
  * the mode of authentication.
  *
  * In the case of an authentication error, NULL is
  * returned.
  *
  */
static struct passwd *look_up_user(char *username)
{

   struct passwd *user_info;

#ifdef USE_SHADOW_AUTH
   struct spwd *shadow_user_info;
#endif // USE_SHADOW_AUTH



   // SQL lookup
   //
#ifdef USER_LOOKUP_SQL
#ifdef DEBUG
   fprintf(stderr, "Account lookup using SQL database\n");
#endif // DEBUG
   user_info = look_up_user_sql(username, USER_LOOKUP_SQL_QUERY, USER_LOOKUP_SQL_DSN);



   // password database lookup
   //
#else // USER_LOOKUP_SQL
#ifdef DEBUG
   fprintf(stderr, "Account lookup using local password database\n");
#endif // DEBUG
   if ((user_info = getpwnam(username)) == NULL)
      fail(ERR_USER, "getpwnam() failed");

#ifdef USE_SHADOW_AUTH
#ifdef DEBUG
   fprintf(stderr, "Retrieving password from shadow password database\n");
#endif // DEBUG
   if ((shadow_user_info = getspnam(username)) == NULL)
      fail(ERR_USER, "getspnam() failed");
   user_info->pw_passwd = shadow_user_info->sp_pwdp;
#endif // USE_SHADOW_AUTH



#endif // USER_LOOKUP_SQL
 

   // success
   //
#ifdef DEBUG
   fprintf(stderr, "Found account\n");
#endif // DEBUG
   return user_info;

}



/**
  * Authenticate the requested user
  *
  * Returns a passwd struct with at least the
  * username (pw_name), UID (pw_uid), GID (pw_gid),
  * and "home" directory (pw_dir) accurately
  * populated.  These values may or may not correspond
  * to a real user account on the system depending on
  * the mode of authentication.
  *
  * NOTE that (for security) the "pw_passwd" entry in
  *      the returned passwd struct) will be zero-ed
  *      out after this function completes
  *
  * In the case of an authentication error, NULL is
  * returned.
  *
  */
static struct passwd *authenticate_user(char *username, char *password)
{

   // look up user account info
   //
   struct passwd *user_info = look_up_user(username);


#if !defined(USE_IMAP_AUTH) && !defined(USE_CCLIENT_AUTH) && !defined(USE_PAM_AUTH)
   char *test_password;
#endif // !defined(USE_IMAP_AUTH) && !defined(USE_CCLIENT_AUTH) && !defined(USE_PAM_AUTH)


   // is the root user allowable?
   //
#ifndef ROOT_ALLOWED
   if (strcmp(username, "root") == 0)
   {
      memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
      fail(ERR_USER, "Super-user not allowed");
   }
#endif // ROOT_ALLOWED


   // check for minimum UID if needed
   //
#ifdef MIN_UID
   if (user_info->pw_uid < MIN_UID)
   {
      memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
      fail(ERR_USER, "UID too small");
   }
#endif // MIN_UID



   // ================================================================
   //
   // authenticate against UW cclient library
   //
#if defined(USE_CCLIENT_AUTH)
#ifdef DEBUG
   fprintf(stderr, "User authentication using IMAP (UW cclient)\n");
#endif // DEBUG
   if (!auth_cclient_imap(username, password, IMAP_SERVER_ADDRESS, IMAP_PORT, NULL))
   {
      memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
      fail(ERR_BADPASS, "UW c-client IMAP authentication failed");
   }



   // ================================================================
   //
   // authenticate against IMAP directly
   //
#elif defined(USE_IMAP_AUTH)
#ifdef DEBUG
   fprintf(stderr, "User authentication using IMAP\n");
#endif // DEBUG
   if (!auth_imap(username, password, IMAP_SERVER_ADDRESS, IMAP_PORT))
   {
      memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
      fail(ERR_BADPASS, "IMAP authentication failed");
   }



   // ================================================================
   //
   // authenticate against PAM
   //
#elif defined(USE_PAM_AUTH)
#ifdef DEBUG
   fprintf(stderr, "User authentication using PAM\n");
#endif // DEBUG
   if (!auth_pam(username, password, PAM_SERVICE))
   {
      memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
      fail(ERR_BADPASS, "PAM authentication failed");
   }



   // ================================================================
   //
   // authenticate against local crypt() function
   //
#else // defined(USE_CCLIENT_AUTH), defined(USE_IMAP_AUTH), defined(USE_PAM_AUTH)
#ifdef DEBUG
   fprintf(stderr, "User authentication using local crypt()\n");
#endif // DEBUG

#if defined(USER_LOOKUP_SQL)
#define PASSWORD_LOCATION "SQL database"
#elif defined(USE_SHADOW_AUTH)
#define PASSWORD_LOCATION "/etc/shadow"
#else // defined(USER_LOOKUP_SQL), defined(USE_SHADOW_AUTH)
#define PASSWORD_LOCATION "/etc/passwd"
#endif // defined(USER_LOOKUP_SQL), defined(USE_SHADOW_AUTH)

   // validate the password
   //
   test_password = crypt(password, user_info->pw_passwd);
   if (strcmp(test_password, user_info->pw_passwd) != 0)
   {
      memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
      fail(ERR_BADPASS, "Password does not match " PASSWORD_LOCATION);
   }



#endif // defined(USE_CCLIENT_AUTH), defined(USE_IMAP_AUTH), defined(USE_PAM_AUTH)



   // disallow users without a password if desired
   //
#ifndef EMPTY_PASSWORD_OK
   if (user_info->pw_passwd == NULL || user_info->pw_passwd[0] == '\0')
   {
      memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
      fail(ERR_BADPASS, "Access not allowed for this account");
   }
#endif // !EMPTY_PASSWORD_OK


   // success (before returning, wipe out password just to be extra safe)
   //
   memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
#ifdef DEBUG
   fprintf(stderr, "User has been authenticated\n");
#endif // DEBUG
   return user_info;

}



/**
  * Look up master user login information: UID, GID, user name, password
  *
  * Returns a passwd struct with at least the
  * username (pw_name), UID (pw_uid), GID (pw_gid),
  * and "home" directory (pw_dir) accurately
  * populated.  These values may or may not correspond
  * to a real user account on the system depending on
  * the mode of authentication.
  *
  * In the case of an authentication error, NULL is
  * returned.
  *
  */
static struct passwd *look_up_master_user(char *username)
{

   struct passwd *user_info;

#ifdef MASTER_USE_SHADOW_AUTH
   struct spwd *shadow_user_info;
#endif // MASTER_USE_SHADOW_AUTH



   // SQL lookup
   //
#ifdef MASTER_USER_LOOKUP_SQL
#ifdef DEBUG
   fprintf(stderr, "Master account lookup using SQL database\n");
#endif // DEBUG
   user_info = look_up_user_sql(username, MASTER_USER_LOOKUP_SQL_QUERY, MASTER_USER_LOOKUP_SQL_DSN);



   // password database lookup
   //
#else // MASTER_USER_LOOKUP_SQL
#ifdef DEBUG
   fprintf(stderr, "Master account lookup using local password database\n");
#endif // DEBUG
   if ((user_info = getpwnam(username)) == NULL)
      fail(ERR_USER, "getpwnam() failed");

#ifdef MASTER_USE_SHADOW_AUTH
#ifdef DEBUG
   fprintf(stderr, "Retrieving master password from shadow password database\n");
#endif // DEBUG
   if ((shadow_user_info = getspnam(username)) == NULL)
      fail(ERR_USER, "getspnam() failed");
   user_info->pw_passwd = shadow_user_info->sp_pwdp;
#endif // MASTER_USE_SHADOW_AUTH



#endif // MASTER_USER_LOOKUP_SQL
 

   // success
   //
#ifdef DEBUG
   fprintf(stderr, "Found master account\n");
#endif // DEBUG
   return user_info;

}



/**
  * Authenticate the master user login
  *
  * Returns a passwd struct with at least the
  * username (pw_name), UID (pw_uid), GID (pw_gid),
  * and "home" directory (pw_dir) accurately
  * populated.  These values may or may not correspond
  * to a real user account on the system depending on
  * the mode of authentication.
  *
  * NOTE that (for security) the "pw_passwd" entry in
  *      the returned passwd struct) will be zero-ed
  *      out after this function completes
  *
  * In the case of an authentication error, NULL is
  * returned.
  *
  */
static struct passwd *authenticate_master_user(char *username, char *password)
{

   // look up user account info
   //
   struct passwd *user_info = look_up_master_user(username);


#if !defined(MASTER_USE_IMAP_AUTH) && !defined(MASTER_USE_CCLIENT_AUTH) && !defined(MASTER_USE_PAM_AUTH)
   char *test_password;
#endif // !defined(MASTER_USE_IMAP_AUTH) && !defined(MASTER_USE_CCLIENT_AUTH) && !defined(MASTER_USE_PAM_AUTH)


   // is the root user allowable?
   //
#ifndef MASTER_ROOT_ALLOWED
   if (strcmp(username, "root") == 0)
   {
      memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
      fail(ERR_USER, "Super-user not allowed for master login");
   }
#endif // MASTER_ROOT_ALLOWED


   // check for minimum UID if needed
   //
#ifdef MASTER_MIN_UID
   if (user_info->pw_uid < MASTER_MIN_UID)
   {
      memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
      fail(ERR_USER, "Master UID too small");
   }
#endif // MASTER_MIN_UID



   // ================================================================
   //
   // authenticate against UW cclient library
   //
#if defined(MASTER_USE_CCLIENT_AUTH)
#ifdef DEBUG
   fprintf(stderr, "Master login authentication using IMAP (UW cclient)\n");
#endif // DEBUG
   if (!auth_cclient_imap(username, password, MASTER_IMAP_SERVER_ADDRESS, MASTER_IMAP_PORT, NULL))
   {
      memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
      fail(ERR_BADPASS, "UW c-client IMAP authentication failed");
   }



   // ================================================================
   //
   // authenticate against IMAP directly
   //
#elif defined(MASTER_USE_IMAP_AUTH)
#ifdef DEBUG
   fprintf(stderr, "Master login authentication using IMAP\n");
#endif // DEBUG
   if (!auth_imap(username, password, MASTER_IMAP_SERVER_ADDRESS, MASTER_IMAP_PORT))
   {
      memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
      fail(ERR_BADPASS, "IMAP authentication failed");
   }



   // ================================================================
   //
   // authenticate against PAM
   //
#elif defined(MASTER_USE_PAM_AUTH)
#ifdef DEBUG
   fprintf(stderr, "Master login authentication using PAM\n");
#endif // DEBUG
   if (!auth_pam(username, password, MASTER_PAM_SERVICE))
   {
      memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
      fail(ERR_BADPASS, "PAM authentication failed");
   }



   // ================================================================
   //
   // authenticate against local crypt() function
   //
#else // defined(MASTER_USE_CCLIENT_AUTH), defined(MASTER_USE_IMAP_AUTH), defined(MASTER_USE_PAM_AUTH)
#ifdef DEBUG
   fprintf(stderr, "Master login authentication using local crypt()\n");
#endif // DEBUG

#if defined(MASTER_USER_LOOKUP_SQL)
#define PASSWORD_LOCATION "SQL database"
#elif defined(MASTER_USE_SHADOW_AUTH)
#define PASSWORD_LOCATION "/etc/shadow"
#else // defined(MASTER_USER_LOOKUP_SQL), defined(MASTER_USE_SHADOW_AUTH)
#define PASSWORD_LOCATION "/etc/passwd"
#endif // defined(MASTER_USER_LOOKUP_SQL), defined(MASTER_USE_SHADOW_AUTH)

   // validate the password
   //
   test_password = crypt(password, user_info->pw_passwd);
   if (strcmp(test_password, user_info->pw_passwd) != 0)
   {
      memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
      fail(ERR_BADPASS, "Master password does not match " PASSWORD_LOCATION);
   }



#endif // defined(MASTER_USE_CCLIENT_AUTH), defined(MASTER_USE_IMAP_AUTH), defined(MASTER_USE_PAM_AUTH)



   // disallow users without a password if desired
   //
#ifndef MASTER_EMPTY_PASSWORD_OK
   if (user_info->pw_passwd == NULL || user_info->pw_passwd[0] == '\0')
   {
      memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
      fail(ERR_BADPASS, "Access not allowed for this master login");
   }
#endif // !MASTER_EMPTY_PASSWORD_OK


   // success (before returning, wipe out password just to be extra safe)
   //
   memset(user_info->pw_passwd, 0, sizeof(user_info->pw_passwd));
#ifdef DEBUG
   fprintf(stderr, "Master login has been authenticated\n");
#endif // DEBUG
   return user_info;

}



// ============================================== 
// ==================== MAIN ==================== 
// ============================================== 


/**
  * Main application entry point
  *
  * Returns 0 (ERR_OK) upon successful completion,
  * otherwise some other non-zero error number is
  * returned.
  *
  */
int main (int arg_count, char *argv[])
{

   int status;
   char *action;
   struct passwd *user_info;
   struct passwd *invoker_user_info;

   char password[BUFSIZE];
   size_t password_length;


#ifdef USE_MASTER_LOGIN
   char master_username[BUFSIZE];
   char master_password[BUFSIZE];
   size_t username_length;
   struct passwd *master_user_info;
#endif // USE_MASTER_LOGIN


   // set umask
   //
   umask(PROG_UMASK);


   // clear out supplementary group IDs
   //
   if (setgroups(0, NULL) < 0)
      efail(ERR_PRIVILEGE, "setgroups(0, NULL)");


   // save off the real UID/GID of the invoking process
   //
   orig_uid = getuid();
   orig_gid = getgid();


   // Validate invoking user/process
   //
   // If configured to limit access to a single real user,
   // verify that that is the case.  The root user is always
   // OK, however, we'll still verify that the restricted
   // user is a valid user, mostly for the sake of
   // troubleshooting.
   //
#ifdef AUTHORIZED_INVOKER
   if ((invoker_user_info = getpwnam(AUTHORIZED_INVOKER)) == NULL)
      fail(ERR_CONFIG, "Invalid authorized invoker");

   if (orig_uid != 0)
   {
      if (invoker_user_info->pw_uid != orig_uid)
         fail(ERR_RESTRICTED, "Invalid real user");
   }

   // remove any information about the invoker from memory, just to be safe
   //
   memset(invoker_user_info, 0, sizeof(invoker_user_info));  
#endif // AUTHORIZED_INVOKER


   // check that we have at least an action argument
   //
   if (arg_count < ACTION_ARG + 1)
      fail(ERR_USAGE, "Incorrect usage");


#ifdef USE_MASTER_LOGIN

#ifdef MASTER_USERNAME
   snprintf(master_username, BUFSIZE, MASTER_USERNAME);

#else // MASTER_USERNAME
   // read master username from stdin
   //
   if (fgets(master_username, BUFSIZE, stdin) == NULL)
      fail(ERR_USER, "Could not read master username");

   username_length = strlen(master_username);
   if (master_username[username_length - 1] == '\n')
      master_username[username_length - 1] = '\0';
   else
      fail(ERR_USER, "Could not read master username");
#endif // MASTER_USERNAME

   // read master password from stdin
   //
   if (fgets(master_password, BUFSIZE, stdin) == NULL)
      fail(ERR_BADPASS, "Could not read master password");

   password_length = strlen(master_password);
   if (master_password[password_length - 1] == '\n')
      master_password[password_length - 1] = '\0';
   else
      fail(ERR_BADPASS, "Could not read master password");


#ifdef DEBUG
   fprintf(stderr, "Validating master login...\n");
#endif // DEBUG


   // validate the master login
   //
   if ((master_user_info = authenticate_master_user(master_username, master_password)) == NULL)
   {
      memset(master_password, 0, sizeof(master_password));
      fail(ERR_USER, "Failed to authenticate master login");
   }


#ifdef DEBUG
   fprintf(stderr, "... master login OK\n");
#endif // DEBUG


   // purge all master user info so it doesn't stick around in memory
   //
   memset(master_username, 0, sizeof(master_username));  
   memset(master_password, 0, sizeof(master_password));  
   memset(master_user_info, 0, sizeof(master_user_info));  
#endif // USE_MASTER_LOGIN


   // read password from stdin
   //
   if (fgets(password, BUFSIZE, stdin) == NULL)
      fail(ERR_BADPASS, "Could not read password");

   password_length = strlen(password);
   if (password[password_length - 1] == '\n')
      password[password_length - 1] = '\0';
   else
      fail(ERR_BADPASS, "Could not read password");


   // validate the requested user
   //
   if ((user_info = authenticate_user(argv[USER_ARG], password)) == NULL)
   {
      memset(password, 0, sizeof(password));
      fail(ERR_USER, "Failed to authenticate user");
   }


   // purge password so it doesn't stick around in memory
   //
   memset(password, 0, sizeof(password));  


   // now we're about ready to go do what we were asked to do
   //
   status = ERR_UNEXPECTED;
   action = argv[ACTION_ARG];


   // file_exists action
   //
   if (strcmp(action, "file_exists") == 0)
   {
      if (arg_count != SRC_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = path_exists(user_info, argv[SRC_ARG], 1, 0);
   }

   // directory_exists action
   //
   else if (strcmp(action, "directory_exists") == 0)
   {
      if (arg_count != SRC_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = path_exists(user_info, argv[SRC_ARG], 0, 1);
   }

   // path_exists action
   //
   else if (strcmp(action, "path_exists") == 0)
   {
      if (arg_count != SRC_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = path_exists(user_info, argv[SRC_ARG], 0, 0);
   }

   // stat action
   //
   else if (strcmp(action, "stat") == 0)
   {
      if (arg_count != DEST_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = stat_path(user_info, argv[SRC_ARG], argv[DEST_ARG], 0);
   }

   // lstat action
   //
   else if (strcmp(action, "lstat") == 0)
   {
      if (arg_count != DEST_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = stat_path(user_info, argv[SRC_ARG], argv[DEST_ARG], 1);
   }

   // get_file action
   //
   else if (strcmp(action, "get_file") == 0)
   {
      if (arg_count != DEST_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = get_file(user_info, argv[SRC_ARG], argv[DEST_ARG]);
   }

   // put_file action
   //
   else if (strcmp(action, "put_file") == 0)
   {
      if (arg_count != DEST_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = put_file(user_info, argv[SRC_ARG], argv[DEST_ARG]);
   }

   // delete_file action
   //
   else if (strcmp(action, "delete_file") == 0)
   {
      if (arg_count != SRC_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = delete_file(user_info, argv[SRC_ARG]);
   }

#ifdef ALLOW_FILE_MANAGER_ACTIONS
   // list_directory action
   //
   else if (strcmp(action, "list_directory") == 0)
   {
      if (arg_count != DEST_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = list_directory(user_info, argv[SRC_ARG], argv[DEST_ARG], 0, 0);
   }

   // list_directory_detailed_stat action
   //
   else if (strcmp(action, "list_directory_detailed_stat") == 0)
   {
      if (arg_count != DEST_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = list_directory(user_info, argv[SRC_ARG], argv[DEST_ARG], 1, 0);
   }

   // list_directory_detailed_lstat action
   //
   else if (strcmp(action, "list_directory_detailed_lstat") == 0)
   {
      if (arg_count != DEST_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = list_directory(user_info, argv[SRC_ARG], argv[DEST_ARG], 1, 1);
   }

   // change_permissions action
   //
   else if (strcmp(action, "change_permissions") == 0)
   {
      if (arg_count != DEST_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = change_permissions(user_info, argv[SRC_ARG], argv[DEST_ARG]);
   }

   // mkdir action
   //
   else if (strcmp(action, "mkdir") == 0)
   {
      if (arg_count != DEST_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = make_directory(user_info, argv[SRC_ARG], argv[DEST_ARG], 0);
   }

   // mkdir_recursive action
   //
   else if (strcmp(action, "mkdir_recursive") == 0)
   {
      if (arg_count != DEST_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = make_directory(user_info, argv[SRC_ARG], argv[DEST_ARG], 1);
   }

   // delete_directory action
   //
   else if (strcmp(action, "delete_directory") == 0)
   {
      if (arg_count != SRC_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = delete_directory(user_info, argv[SRC_ARG]);
   }

   // delete_file_or_directory action
   //
   else if (strcmp(action, "delete_file_or_directory") == 0)
   {
      if (arg_count != SRC_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = delete_file_or_directory(user_info, argv[SRC_ARG]);
   }

   // rename action
   //
   else if (strcmp(action, "rename") == 0)
   {
      if (arg_count != DEST_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = rename_file_or_directory(user_info, argv[SRC_ARG], argv[DEST_ARG]);
   }

#else // ALLOW_FILE_MANAGER_ACTIONS
   // file manager actions not supported when not compiled in
   //
   else if (strcmp(action, "list_directory") == 0
         || strcmp(action, "list_directory_detailed_stat") == 0
         || strcmp(action, "list_directory_detailed_lstat") == 0
         || strcmp(action, "change_permissions") == 0
         || strcmp(action, "mkdir") == 0
         || strcmp(action, "mkdir_recursive") == 0
         || strcmp(action, "delete_directory") == 0
         || strcmp(action, "delete_file_or_directory") == 0
         || strcmp(action, "rename") == 0)
   {
      fail(ERR_USAGE, "Feature not enabled");
   }
#endif // ALLOW_FILE_MANAGER_ACTIONS

#ifdef VACATION_PATH
   // vacation_init action
   //
   else if (strcmp(action, "vacation_init") == 0)
   {
      if (arg_count != ACTION_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = vacation_init(user_info, 0, "");
   }

#ifdef VACATION_INTERVAL_FLAG
   // vacation_init_with_interval action
   //
   else if (strcmp(action, "vacation_init_with_interval") == 0)
   {
      if (arg_count != SRC_ARG + 1)
         fail(ERR_USAGE, "Incorrect usage");
      status = vacation_init(user_info, 1, argv[SRC_ARG]);
   }
#else // VACATION_INTERVAL_FLAG
   // vacation actions not supported when not compiled in
   //
   else if (strcmp(action, "vacation_init_with_interval") == 0)
   {
      fail(ERR_USAGE, "Feature not enabled");
   }
#endif // VACATION_INTERVAL_FLAG

#else // VACATION_PATH
   // vacation actions not supported when not compiled in
   //
   else if (strcmp(action, "vacation_init") == 0)
   {
      fail(ERR_USAGE, "Feature not enabled");
   }
#endif // VACATION_PATH

   // unknown action
   //
   else
   {
#ifdef DEBUG
      fprintf(stderr, "Bad action: %s\n", action);
#endif // DEBUG
      fail(ERR_USAGE, "Action not found");
   }


   // done!
   //
#ifdef USER_LOOKUP_SQL
   free(user_info->pw_dir);
   free(user_info->pw_passwd);
   free(user_info);
#endif // USER_LOOKUP_SQL
   exit(status);

}


