<?php

/**
  * SquirrelMail Server Settings Backend Plugin
  * Copyright (c) 2008-2010 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage server_settings_backend
  *
  */



/**
  * Retrieves a setting from a SQL database.
  *
  * If the query returns more than one result, the
  * results will be returned as an array.  When no
  * results were found or the value is empty/null,
  * NULL is returned.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $dsn   The data source name that provides
  *                       the needed credentials to log into
  *                       the database.
  * @param string  $abstraction_type The name of the Pear database
  *                                  abstraction layer.  Currently,
  *                                  one of "DB" and "MDB2" is supported.
  * @param string  $query The query that will extract the needed
  *                       information from the database.
  * @param boolean $quiet When TRUE, suppresses any error
  *                       messages and makes this function
  *                       return NULL without doing anything
  *                       else (OPTIONAL; default = FALSE).
  *
  * @return mixed The setting value if it was retrieved normally,
  *               or NULL if the query did not return any results
  *               or if an error occurred (and $quiet is TRUE).
  *               
  */
function retrieve_server_setting_from_sql($dsn, $abstraction_type,
                                          $query, $quiet=FALSE)
{

   // ready the database connection
   //
   $db = ssb_get_database_connection($dsn, $abstraction_type, $quiet);
   if ($quiet && is_null($db))
      return NULL;


   // run the query
   //
   $error = '';
   if (strtoupper($abstraction_type) == 'DB')
      $result = $db->getAll($query);
   else if (strtoupper($abstraction_type) == 'MDB2')
      $result = $db->queryAll($query);


   if (PEAR::isError($result))
   {
      if ($quiet) return NULL;
      sq_change_text_domain('server_settings_backend');

      // show query and database error when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("Could not complete query to database (%s): %s"), $query, $result->getMessage());
      else
         $error = _("Could not query database");

      sq_change_text_domain('squirrelmail');
   }


   // check for unexpected results (only want one column per row)
   //
/* We now allow mulitple-column SELECTs ---------------------------------
   if (empty($error) && !empty($result[0]) && sizeof($result[0]) != 1)
   {
      if ($quiet) return NULL;
      sq_change_text_domain('server_settings_backend');

      // show query and database error when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("Query returned malformed results (%s): instead of one column, %s columns were returned"), $query, sizeof($result[0]));
      else
         $error = _("Malformed database query");

      sq_change_text_domain('squirrelmail');
   }
-------------------------------------------- */


   // spit out error if any
   //
   if (!empty($error))
   {
      global $color;
      $ret = plain_error_message($error, $color);
      if (check_sm_version (1, 5, 2))
      {
         echo $ret;
         global $oTemplate;
         $oTemplate->display('footer.tpl');
      }
      exit;
   }


   // multi-column lookups just have to be returned as is; the
   // caller is responsible for validating and parsing them further
   //
   if (isset($result[0]) && is_array($result[0]) && sizeof($result[0]) > 1)
      return $result;


   // combine multiple results?
   //
   if (sizeof($result) > 1)
   {
      $ret = array();
      foreach ($result as $row)
         $ret[] = $row[0];
      //return serialize($ret);
      return $ret;
   }


   // MySQL seems to return null values as an empty string, so, hoping
   // that this does not create problems with other DBs or when the
   // setting really does contain a blank string, we'll return NULL
   // when the value is empty...
   //
   else if (sizeof($result) == 0 || empty($result[0][0]))
      return NULL;
   else
      return $result[0][0];

}



/**
  * Runs a SQL query.  This function can be used for any
  * query that the caller does not expect any return value
  * more than an indication of success/failure.
  * 
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $dsn   The data source name that provides
  *                       the needed credentials to log into
  *                       the database.
  * @param string  $abstraction_type The name of the Pear database
  *                                  abstraction layer.  Currently,
  *                                  one of "DB" and "MDB2" is supported.
  * @param string  $query The query to be executed.
  * @param boolean $quiet When TRUE, suppresses any error
  *                       messages and makes this function
  *                       return NULL without doing anything
  *                       else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE if the query was successfully executed
  *               or NULL if it was not (and $quiet is TRUE).
  *
  */
function execute_sql_query($dsn, $abstraction_type,
                           $query, $quiet=FALSE)
{

   // ready the database connection
   //
   $db = ssb_get_database_connection($dsn, $abstraction_type, $quiet);
   if ($quiet && is_null($db))
      return NULL;


   // run the query
   //
   $result = $db->query($query);


   if (PEAR::isError($result))
   {
      if ($quiet) return NULL;
      sq_change_text_domain('server_settings_backend');

      // show query and database error when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("Could not complete query to database (%s): %s"), $query, $result->getMessage());
      else
         $error = _("Could not query database");

      sq_change_text_domain('squirrelmail');
      global $color;
      $ret = plain_error_message($error, $color);
      if (check_sm_version (1, 5, 2))
      {
         echo $ret;
         global $oTemplate;
         $oTemplate->display('footer.tpl');
      }
      exit;
   }

   return TRUE;

}



/**
  * Get a database connection
  *
  * @param string  $dsn   The DSN to be used to connect.
  * @param string  $abstraction_type The name of the Pear database
  *                                  abstraction layer.  Currently,
  *                                  one of "DB" and "MDB2" is supported.
  * @param boolean $quiet When TRUE, suppresses any error
  *                       messages and makes this function
  *                       return NULL without doing anything
  *                       else (OPTIONAL; default = FALSE).
  *
  * @return mixed The database connection handle or NULL if
  *               an error occurred and $quiet is TRUE.
  *
  */
function ssb_get_database_connection($dsn, $abstraction_type, $quiet=FALSE)
{

   global $ssb_db_connections;


   $pear = ssb_get_pear_db($abstraction_type, $quiet);
   if ($quiet && is_null($pear))
      return NULL;


   // make a new connection if needed; exit if failure
   //
   $cache_key = $dsn . $abstraction_type;
   if (empty($ssb_db_connections[$cache_key]))
   {

      // force new connection for each DSN - yikes, but in MDB2 we
      // need it in order to keep multiple DSNs apart from one another
      //
      if (strpos($dsn, 'new_link=') === FALSE)
         if (strpos($dsn, '?') === FALSE)
            $dsn .= '?new_link=true';
         else
            $dsn .= '&new_link=true';

      if (strtoupper($abstraction_type) == 'DB')
         $ssb_db_connections[$cache_key] = DB::connect($dsn);
      else if (strtoupper($abstraction_type) == 'MDB2')
         $ssb_db_connections[$cache_key] = MDB2::connect($dsn);

      if (PEAR::isError($ssb_db_connections[$cache_key]))
      {
         if ($quiet)
         {
            unset($ssb_db_connections[$cache_key]);
            return NULL;
         }
         sq_change_text_domain('server_settings_backend');

         // show DSN and database error when in debug mode
         //
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not connect to database (DSN: %s): %s"), $dsn, $ssb_db_connections[$cache_key]->getMessage());
         else
            $error = _("Could not connect to database");

         unset($ssb_db_connections[$cache_key]);

         sq_change_text_domain('squirrelmail');
         global $color;
         $ret = plain_error_message($error, $color);
         if (check_sm_version (1, 5, 2))
         {
            echo $ret;
            global $oTemplate;
            $oTemplate->display('footer.tpl');
         }
         exit;
      }

      if (strtoupper($abstraction_type) == 'DB')
         $ssb_db_connections[$cache_key]->setFetchMode(DB_FETCHMODE_ORDERED);
      else if (strtoupper($abstraction_type) == 'MDB2')
         $ssb_db_connections[$cache_key]->setFetchMode(MDB2_FETCHMODE_ORDERED);

   }


   // return connection
   //
   return $ssb_db_connections[$cache_key];

}



/**
  * Include Pear database abstraction layer
  *
  * @param string  $abstraction_type The name of the Pear database
  *                                  abstraction layer.  Currently,
  *                                  one of "DB" and "MDB2" is supported.
  * @param boolean $quiet When TRUE, suppresses any error
  *                       messages and makes this function
  *                       return NULL without doing anything
  *                       else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE if no problems occured, or NULL if
  *               an error occurred and $quiet is TRUE.
  *
  */
function ssb_get_pear_db($abstraction_type, $quiet=FALSE)
{

   static $been_here = array();
   if (!empty($been_here[$abstraction_type])) return TRUE;
   $been_here[$abstraction_type] = TRUE;


   $error = '';
   switch (strtoupper($abstraction_type))
   {
      case 'DB':
         $db_file = 'DB.php';
         break;

      case 'MDB2':
         $db_file = 'MDB2.php';
         break;

      default:
         if ($quiet) return NULL;
         sq_change_text_domain('server_settings_backend');
         $error = sprintf(_("Unknown database abstraction layer: \"%s\""), $abstraction_type);
         sq_change_text_domain('squirrelmail');
         global $color;
         $ret = plain_error_message($error, $color);
         if (check_sm_version (1, 5, 2))
         {
            echo $ret;
            global $oTemplate;
            $oTemplate->display('footer.tpl');
         }
         exit;
   }


   // mask include errors if not in debug mode
   //
   global $sm_debug_mode, $server_settings_backend_debug;
   server_settings_backend_init();
   if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
   if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
      $if_statement = 'return !include_once(\'' . $db_file . '\');';
   else
      $if_statement = 'return !@include_once(\'' . $db_file . '\');';

   if (eval($if_statement))
   {
      if ($quiet) return NULL;
      sq_change_text_domain('server_settings_backend');
      $error = _("Could not find Pear DB or MDB2 library");
      sq_change_text_domain('squirrelmail');
      global $color;
      $ret = plain_error_message($error, $color);
      if (check_sm_version (1, 5, 2))
      {
         echo $ret;
         global $oTemplate;
         $oTemplate->display('footer.tpl');
      }
      exit;
   }

   return TRUE;

}



