<?php

/**
  * SquirrelMail Server Settings Backend Plugin
  * Copyright (c) 2008-2010 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage server_settings_backend
  *
  */



/**
  * Retrieves a setting from a FTP server/account.
  *
  * If the target file has more than one occurance of
  * the setting, the results will be returned as a
  * set of nested arrays.  When no results were found,
  * NULL is returned.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param mixed   $FTP           A handle to the FTP connection
  *                               if one has already been opened.
  *                               If not given as a resource (handle),
  *                               a new connection attempt will be made.
  *                               If a new connection was opened, it
  *                               will be closed herein; if not, the
  *                               connection will not be closed by this
  *                               function unless a fatal error occurs.
  *                               UPDATE: due to the use of connection
  *                               caching, we'll avoid closing the
  *                               connection under all circumstances.
  * @param string  $host          The target FTP hostname.
  * @param int     $port          The port to be used when connecting.
  * @param boolean $use_ssl       Whether or not to use SSL when connecting.
  * @param boolean $passive       Whether or not to use passive mode.
  * @param string  $mode          The FTP transfer mode.
  * @param string  $user          The username to log in with.
  * @param string  $pass          The password to log in with.
  * @param string  $directory     The directory to change to after login.
  *                               If empty, no directory change is attempted.
  * @param string  $remote_file   The target (remote) FTP file name.
  * @param string  $parse_pattern The regular expression pattern
  *                               needed to parse the setting value
  *                               out of the file.
  * @param string  $pattern_group_number The group number of the regular
  *                                      expression parsing pattern that
  *                                      contains ONLY the actual value
  *                                      (and not, for example, also the
  *                                      setting name).
  * @param boolean $treat_as_empty_when_not_found When TRUE, if the target
  *                                               file is non-existent, the
  *                                               return value is as if the
  *                                               setting was not found in
  *                                               the file (OPTIONAL;
  *                                               default = FALSE).
  * @param boolean $quiet         When TRUE, suppresses any error
  *                               messages and makes this function
  *                               return NULL without doing anything
  *                               else (OPTIONAL; default = FALSE).
  * @param boolean $return_file_contents When TRUE, the normal return
  *                                      value is placed inside an array
  *                                      to which the file contents are
  *                                      added as the second and last
  *                                      array element, and this two-
  *                                      element array is then returned
  *                                      (OPTIONAL; default = FALSE).
  *
  * @return mixed An array containing the setting value and its offset in
  *               the target file (in that order) if it was retrieved
  *               normally and was only found once in the file (scalar
  *               value), or an array of arrays, where the sub-arrays
  *               contain each setting value and its offset in the target
  *               file (in that order) if the setting is found more than
  *               once in the file (the caller should check if the return
  *               value's first array element is an array itself to
  *               determine if the result is non-scalar), or NULL if the
  *               setting is not found in the file or if an error occurred
  *               (and $quiet is TRUE).  Note that if $return_file_contents
  *               is TRUE, the normal return value described above will be
  *               placed inside a two-element array, and the file contents
  *               placed as the second array element, and that is returned.
  *
  */
function retrieve_server_setting_from_ftp($FTP, $host, $port, $use_ssl, $passive,
                                          $mode, $user, $pass, $directory,
                                          $remote_file, $parse_pattern,
                                          $pattern_group_number,
                                          $treat_as_empty_when_not_found=FALSE,
                                          $quiet=FALSE,
                                          $return_file_contents=FALSE)
{

   $error = '';

   $file = ftp_get_file($FTP, $host, $port, $use_ssl, $passive, $mode,
                        $user, $pass, $directory, $remote_file,
                        $treat_as_empty_when_not_found, $quiet);


   // first deal with file-not-found
   //
   if (is_null($file))
   {

      if ($treat_as_empty_when_not_found)
      {
         // parse_value() found in functions.php
         //
         $ret = parse_value('', $parse_pattern, $pattern_group_number, $quiet);
         if ($return_file_contents)
            $ret = array($ret, '');
         return $ret;
      }

      if ($quiet) return NULL;

      sq_change_text_domain('server_settings_backend');
      $error = _("FTP error when attempting to retrieve a setting from the server");
      sq_change_text_domain('squirrelmail');

   }


   // parse out our target value from the file
   //
   else
   {

      // parse_value_from_file() found in functions.php
      //
      $ret = parse_value_from_file($file, $parse_pattern,
                                   $pattern_group_number, $quiet,
                                   $return_file_contents);
      unlink($file);
      return $ret;

   }


   // spew error
   //
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Stores a file in a FTP server/account.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param mixed   $FTP               A handle to the FTP connection
  *                                   if one has already been opened.
  *                                   If not given as a resource (handle),
  *                                   a new connection attempt will be made.
  *                                   If a new connection was opened, it
  *                                   will be closed herein; if not, the
  *                                   connection will not be closed by this
  *                                   function unless a fatal error occurs.
  *                                   UPDATE: due to the use of connection
  *                                   caching, we'll avoid closing the
  *                                   connection under all circumstances.
  * @param string  $file_contents     The contents of the file to be
  *                                   placed on the server.
  * @param boolean $allow_overwrite   When TRUE, any existing file at
  *                                   $path will be overwritten with
  *                                   $contents, otherwise, an existing
  *                                   file will create an error.
  * @param string  $host              The target FTP hostname.
  * @param int     $port              The port to be used when connecting.
  * @param boolean $use_ssl           Whether or not to use SSL when connecting.
  * @param boolean $passive           Whether or not to use passive mode.
  * @param string  $mode              The FTP transfer mode.
  * @param string  $user              The username to log in with.
  * @param string  $pass              The password to log in with.
  * @param string  $directory         The directory to change to after login.
  *                                   If empty, no directory change is attempted.
  *                                   It is acceptable to specify nothing
  *                                   here and give a full path for $remote_file.
  * @param string  $remote_file       The target (remote) FTP file name.
  * @param boolean $delete_when_empty When TRUE, if the file contents
  *                                   are empty, the file will be
  *                                   deleted instead of left as an
  *                                   empty file (OPTIONAL; default = FALSE).
  * @param boolean $quiet             When TRUE, suppresses any error
  *                                   messages and makes this function
  *                                   return NULL without doing anything
  *                                   else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when completed normally, or NULL if an error
  *               occurred (and $quiet is TRUE).
  *
  */
function ftp_put_file($FTP, $file_contents, $allow_overwrite, $host,
                      $port, $use_ssl, $passive, $mode, $user, $pass,
                      $directory, $remote_file, $delete_when_empty=FALSE,
                      $quiet=FALSE)
{

   $error = '';
   $close_connection = FALSE;
   $cwd = '.';
   if ($remote_file[strlen($remote_file) - 1] == '/'
    || $remote_file[strlen($remote_file) - 1] == '\\')
      $remote_file = substr($remote_file, 0, -1);
//TODO: path separator below may not work under all O/Ses(?)
   $full_path = (empty($directory) ? $remote_file : $directory . '/' . $remote_file);


   // connect to FTP server
   //
   if (!is_resource($FTP))
   {
      $FTP = ssb_ftp_connect($host, $port, $use_ssl, $passive,
                             $user, $pass, $directory, $quiet);
      $close_connection = TRUE;

      // don't do anything if not connected
      //
      if (is_null($FTP))
         return NULL;
   }


   // if already connected, note directory we started in
   // and then make sure we're in the right directory
   //
   else if (!($cwd = ftp_pwd($FTP)) || (!empty($directory) && !sq_call_function_suppress_errors('ftp_chdir', array($FTP, $directory))))
   {
      if ($quiet) return NULL;
      sq_change_text_domain('server_settings_backend');

      // show connect details when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("Failed to change to directory \"%s\" on FTP host \"%s\""), $directory, $host);
      else
         $error = _("Unable to change directories on FTP server");

      sq_change_text_domain('squirrelmail');
   }


   // override code above that might ask for closing
   // the connection when this function exits...
   // we want to cache all our connections, and closing
   // a connection will b0rk someone else who wants to
   // use the same connection
   //
   $close_connection = FALSE;


   // if overwriting not allowed, bail if path already exists
   //
   if (empty($error)
    && !$allow_overwrite
    && ftp_path_exists($FTP, $host, $port, $use_ssl, $passive, $user,
                       $pass, $full_path, FALSE, $quiet))
   {
      if ($quiet)
      {
         if ($close_connection) ftp_close($FTP);
         else @ftp_chdir($cwd);
         return NULL;
      }
      sq_change_text_domain('server_settings_backend');

      // show connect details when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("File or directory \"%s\" already exists on FTP host \"%s\"; cannot overwrite"), $full_path, $host);
      else
         $error = _("File or directory already exists; cannot overwrite");

      sq_change_text_domain('squirrelmail');
   }


   // if file is empty, should we just delete it?
   //
   if (empty($error) && $delete_when_empty)
   {

      $trimmed_file_contents = trim($file_contents);

      if (empty($trimmed_file_contents))
      {

         // only way this can return as non-TRUE is if $quiet
         // and an error occured, so if it does, just return NULL
         //
         if (!ssb_ftp_delete($FTP, $host, $port, $use_ssl, $passive, $user,
                           $pass, $directory, $remote_file, FALSE, FALSE, $quiet))
         {
            if ($close_connection) ftp_close($FTP);
            else @ftp_chdir($cwd);
            return NULL;
         }

         if ($close_connection) ftp_close($FTP);
         else @ftp_chdir($cwd);
         return TRUE;
      }

   }


   // get a temp file for uploading the target file
   //
   if (empty($error))
   {

      $temp = ssb_get_temp_file($quiet);
      if (is_null($temp)) // implied $quiet is TRUE
      {
         if ($close_connection) ftp_close($FTP);
         else @ftp_chdir($cwd);
         return NULL;
      }
      list($file_path, $FILE) = $temp;


      // now, write the file contents out to the local file
      //
      if (sq_call_function_suppress_errors('fwrite', array($FILE, $file_contents)) === FALSE)
      {
         if ($quiet)
         {
            if ($close_connection) ftp_close($FTP);
            else @ftp_chdir($cwd);
            return NULL;
         }
         sq_change_text_domain('server_settings_backend');
         $error = _("File error when attempting to write to a temporary file");
         sq_change_text_domain('squirrelmail');
         fclose($FILE);
         unlink($file_path);
      }


      // and put the file on the server
      //
      else
      {

         fclose($FILE);
         if ($mode == 'ASCII') $mode = FTP_ASCII;
         else $mode = FTP_BINARY;
         if (!sq_call_function_suppress_errors('ftp_put', array($FTP, $remote_file, $file_path, $mode)))
         {
            if ($quiet)
            {
               if ($close_connection) ftp_close($FTP);
               else @ftp_chdir($cwd);
               return NULL;
            }
            sq_change_text_domain('server_settings_backend');
            $error = _("FTP error when attempting to store a file on the server");
            sq_change_text_domain('squirrelmail');
            unlink($file_path);
         }


         // success
         //
         else 
         {
            unlink($file_path);
            if ($close_connection) ftp_close($FTP);
            else @ftp_chdir($cwd);
            return TRUE;
         }

      }

   }


   // spew error
   //
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   @ftp_close($FTP);
   exit;

}



/**
  * Retrieves a file from a FTP server/account.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param mixed   $FTP           A handle to the FTP connection
  *                               if one has already been opened.
  *                               If not given as a resource (handle),
  *                               a new connection attempt will be made.
  *                               If a new connection was opened, it
  *                               will be closed herein; if not, the
  *                               connection will not be closed by this
  *                               function unless a fatal error occurs.
  *                               UPDATE: due to the use of connection
  *                               caching, we'll avoid closing the
  *                               connection under all circumstances.
  * @param string  $host          The target FTP hostname.
  * @param int     $port          The port to be used when connecting.
  * @param boolean $use_ssl       Whether or not to use SSL when connecting.
  * @param boolean $passive       Whether or not to use passive mode.
  * @param string  $mode          The FTP transfer mode.
  * @param string  $user          The username to log in with.
  * @param string  $pass          The password to log in with.
  * @param string  $directory     The directory the target file is in.
  *                               If empty, no directory change is attempted.
  *                               It is acceptable to specify nothing
  *                               here and give a full path for $remote_file.
  * @param string  $remote_file   The target (remote) FTP file name.
  * @param boolean $ignore_non_existent When TRUE, if the file does not exist,
  *                                     NULL is returned, when FALSE and the
  *                                     file does not exist, an error is
  *                                     triggered (OPTIONAL; default = FALSE).
  * @param boolean $quiet         When TRUE, suppresses any error
  *                               messages and makes this function
  *                               return NULL without doing anything
  *                               else (OPTIONAL; default = FALSE).
  *
  * @return mixed A string containing the full file path to the local
  *               copy of the file (typically in the SquirrelMail
  *               attachments directory) if it was retrieved normally,
  *               or NULL if an error occurred and $quiet is TRUE or
  *               NULL if the file does not exist and $ignore_non_existent
  *               is TRUE.
  *
  */
function ftp_get_file($FTP, $host, $port, $use_ssl, $passive, $mode, $user,
                      $pass, $directory, $remote_file,
                      $ignore_non_existent=FALSE, $quiet=FALSE)
{

   $error = '';
   $close_connection = FALSE;
   $cwd = '.';
   if ($remote_file[strlen($remote_file) - 1] == '/'
    || $remote_file[strlen($remote_file) - 1] == '\\')
      $remote_file = substr($remote_file, 0, -1);
      

   // connect to FTP server
   //
   if (!is_resource($FTP))
   {
      $FTP = ssb_ftp_connect($host, $port, $use_ssl, $passive,
                             $user, $pass, $directory, $quiet);
      $close_connection = TRUE;

      // don't do anything if not connected
      //
      if (is_null($FTP))
         return NULL;
   }


   // if already connected, note directory we started in
   // and then make sure we're in the right directory
   //
   else if (!($cwd = ftp_pwd($FTP)) || (!empty($directory) && !sq_call_function_suppress_errors('ftp_chdir', array($FTP, $directory))))
   {
      if ($quiet) return NULL;
      sq_change_text_domain('server_settings_backend');

      // show connect details when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("Failed to change to directory \"%s\" on FTP host \"%s\""), $directory, $host);
      else
         $error = _("Unable to change directories on FTP server");

      sq_change_text_domain('squirrelmail');
   }


   // override code above that might ask for closing
   // the connection when this function exits...
   // we want to cache all our connections, and closing
   // a connection will b0rk someone else who wants to
   // use the same connection
   //
   $close_connection = FALSE;


   if (empty($error))
   {

      // get a temp file for downloading the target file
      //
      $temp = ssb_get_temp_file($quiet);
      if (is_null($temp)) // implied $quiet is TRUE
      {
         if ($close_connection) ftp_close($FTP);
         else @ftp_chdir($cwd);
         return NULL;
      }
      list($file_path, $FILE) = $temp;


      // now, actually pull file
      //
      if ($mode == 'ASCII') $mode = FTP_ASCII;
      else $mode = FTP_BINARY;


      // don't want to dump errors anywhere when missing file
      // means nothing to us
      //
      if ($ignore_non_existent || $quiet)
      {

         if (!@ftp_fget($FTP, $FILE, $remote_file, $mode))
         {
            if ($close_connection) ftp_close($FTP);
            else @ftp_chdir($cwd);
            return NULL;
         }

      }
      else
      {

         if (!sq_call_function_suppress_errors('ftp_fget', array($FTP, $FILE, $remote_file, $mode)))
         {

            // show connect details when in debug mode
            //
            global $sm_debug_mode, $server_settings_backend_debug;
            server_settings_backend_init();
            if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
            if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
               $error = sprintf(_("Could not retrieve file \"%s\" in directory \"%s\" on FTP host \"%s\""), $remote_file, $directory, $host);
            else
               $error = _("FTP error when attempting to retrieve file");

            sq_change_text_domain('squirrelmail');

            if ($close_connection) ftp_close($FTP);
            else @ftp_chdir($cwd);
            fclose($FILE);
            unlink($file_path);

         }

      }

   }


   // success
   //
   if (empty($error))
   {
      if ($close_connection) ftp_close($FTP);
      else @ftp_chdir($cwd);
      fclose($FILE);
      return $file_path;
   }


   // spew error
   //
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Deletes a file or directory on a FTP server/account.  If the
  * file or directory does not exist, behavior will be as if it
  * was correctly deleted.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param mixed   $FTP           A handle to the FTP connection
  *                               if one has already been opened.
  *                               If not given as a resource (handle),
  *                               a new connection attempt will be made.
  *                               If a new connection was opened, it
  *                               will be closed herein; if not, the
  *                               connection will not be closed by this
  *                               function unless a fatal error occurs.
  *                               UPDATE: due to the use of connection
  *                               caching, we'll avoid closing the
  *                               connection under all circumstances.
  * @param string  $host          The target FTP hostname.
  * @param int     $port          The port to be used when connecting.
  * @param boolean $use_ssl       Whether or not to use SSL when connecting.
  * @param boolean $passive       Whether or not to use passive mode.
  * @param string  $user          The username to log in with.
  * @param string  $pass          The password to log in with.
  * @param string  $directory     The directory the target file or directory
  *                               is in.  It is acceptable to specify nothing
  *                               here and give a full path for $target.
  * @param string  $target        The target (remote) FTP file or directory
  *                               name.
  * @param boolean $is_directory  When TRUE, the target being deleted is
  *                               assumed to be a directory.  When FALSE,
  *                               it is assumed to be a file.  This
  *                               information is determined by actually
  *                               inspecting the target if at all possible,
  *                               but if not supported by the FTP server,
  *                               the value passed in here will be used.
  * @param boolean $autodetect    When FALSE, $is_directory will be used
  *                               unquestioningly.  When TRUE, the $path
  *                               will be inspected to try to determine
  *                               if it is a file, directory, etc.
  *                               (OPTIONAL; default = TRUE)
  * @param boolean $quiet         When TRUE, suppresses any error
  *                               messages and makes this function
  *                               return NULL without doing anything
  *                               else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when completed normally, or NULL if an error
  *               occurred (and $quiet is TRUE).
  *
  */
function ssb_ftp_delete($FTP, $host, $port, $use_ssl, $passive, $user,
                        $pass, $directory, $target, $is_directory,
                        $autodetect=TRUE, $quiet=FALSE)
{

   $error = '';
   $close_connection = FALSE;
   $cwd = '.';
   if ($target[strlen($target) - 1] == '/'
    || $target[strlen($target) - 1] == '\\')
      $target = substr($target, 0, -1);
//TODO: path separator below may not work under all O/Ses(?)
   $full_path = (empty($directory) ? $target : $directory . '/' . $target);


   // grab information about target
   //
   $file_info = ftp_path_exists(FALSE, $host, $port, $use_ssl,
                                $passive, $user, $pass,
                                $full_path, TRUE, $quiet);
   if (is_null($file_info)) return NULL;  // pass along error ($quiet must be TRUE)


   // if file doesn't even exist in the first place, just bail -
   // our job is already done for us
   //
   if (!$file_info)
      return TRUE;


   // if possible, try to determine $is_directory on our own
   //
   if ($autodetect)
   {
      if (!empty($file_info['type']) && $file_info['type'] == 'directory')
         $is_directory = TRUE;
      else if (!empty($file_info['type']) && $file_info['type'] == 'file')
         $is_directory = FALSE;
      else if (!empty($file_info['type']))
         $is_directory = FALSE;  // for now, we'll assume we can delete other types too but can throw an error here if need be
   }


   if ($is_directory) $delete_function = 'ftp_rmdir';
   else $delete_function = 'ftp_delete';


   // connect to FTP server
   //
   if (!is_resource($FTP))
   {
      $FTP = ssb_ftp_connect($host, $port, $use_ssl, $passive,
                             $user, $pass, $directory, $quiet);

      // don't do anything if not connected
      //
      if (is_null($FTP))
         return NULL;

      $close_connection = TRUE;
   }


   // if already connected, note directory we started in
   // and then make sure we're in the right directory
   //
   else if (!($cwd = ftp_pwd($FTP)) || (!empty($directory) && !sq_call_function_suppress_errors('ftp_chdir', array($FTP, $directory))))
   {
      if ($quiet) return NULL;
      sq_change_text_domain('server_settings_backend');
   
      // show connect details when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("Failed to change to directory \"%s\" on FTP host \"%s\""), $directory, $host);
      else
         $error = _("Unable to change directories on FTP server");
   
      sq_change_text_domain('squirrelmail');
   }


   // override code above that might ask for closing
   // the connection when this function exits...
   // we want to cache all our connections, and closing
   // a connection will b0rk someone else who wants to
   // use the same connection
   //
   $close_connection = FALSE;


   // now, actually delete file/directory
   //
   if (empty($error)
    && !sq_call_function_suppress_errors($delete_function, array($FTP, $target)))
   {
      if ($quiet)
      {
         if ($close_connection) ftp_close($FTP);
         else @ftp_chdir($cwd);
         return NULL;
      }
      sq_change_text_domain('server_settings_backend');
   
      // show connect details when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("Failed to delete file or directory \"%s\" from FTP host \"%s\""), $target, $host);
      else
         $error = _("Error deleting file or directory");

      sq_change_text_domain('squirrelmail');
   }


   // success
   //
   if (empty($error))
   {
      if ($close_connection) ftp_close($FTP);
      else @ftp_chdir($cwd);
      return TRUE;
   }


   // spew error
   //
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   @ftp_close($FTP);
   exit;

}



/**
  * Retrieves file listing of a directory on a FTP server/account.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param mixed   $FTP           A handle to the FTP connection
  *                               if one has already been opened.
  *                               If not given as a resource (handle),
  *                               a new connection attempt will be made.
  *                               If a new connection was opened, it
  *                               will be closed herein; if not, the
  *                               connection will not be closed by this
  *                               function unless a fatal error occurs.
  *                               UPDATE: due to the use of connection
  *                               caching, we'll avoid closing the
  *                               connection under all circumstances.
  * @param string  $host          The target FTP hostname.
  * @param int     $port          The port to be used when connecting.
  * @param boolean $use_ssl       Whether or not to use SSL when connecting.
  * @param boolean $passive       Whether or not to use passive mode.
  * @param string  $user          The username to log in with.
  * @param string  $pass          The password to log in with.
  * @param string  $directory     The directory to get the listing of.
  *                               If empty, no directory change is attempted.
  * @param boolean $quiet         When TRUE, suppresses any error
  *                               messages and makes this function
  *                               return NULL without doing anything
  *                               else (OPTIONAL; default = FALSE).
  *
  * @return mixed NULL when an error occurred and $quiet is TRUE, otherwise
  *               when the listing was retrieved normally, an array is
  *               returned containing sub-arrays which each represent a
  *               file or directory in the listing.  These sub-arrays
  *               will contain possibly several different key-value
  *               pairs, each for a different file attribute, but should
  *               always at least contain a "name" key whose value is
  *               the file or directory name.  Other possible attributes
  *               are: "type" ("file", "directory", "link", "pipe",
  *               "socket"), "owner", "group", "size", "permissions",
  *               "date" (date last modified), "link_saki" (link
  *               destination), "numeric_permissions".
  *
  */
function ftp_get_directory_listing($FTP, $host, $port, $use_ssl, $passive,
                                   $user, $pass, $directory, $quiet=FALSE)
{

   $error = '';
   $close_connection = FALSE;
   if ($directory[strlen($directory) - 1] == '/'
    || $directory[strlen($directory) - 1] == '\\')
      $directory = substr($directory, 0, -1);


   // connect to FTP server
   //
   if (!is_resource($FTP))
   {
      $FTP = ssb_ftp_connect($host, $port, $use_ssl, $passive,
                             $user, $pass, '', $quiet);
      $close_connection = TRUE;

      // don't do anything if not connected
      //
      if (is_null($FTP))
         return NULL;
   }


   // override code above that might ask for closing
   // the connection when this function exits...
   // we want to cache all our connections, and closing
   // a connection will b0rk someone else who wants to
   // use the same connection
   //
   $close_connection = FALSE;


   // get a "raw" listing
   //
   $file_list = sq_call_function_suppress_errors('ftp_rawlist', array($FTP, $directory));


   if (!is_array($file_list))
   {
      if ($quiet)
      {
         if ($close_connection) ftp_close($FTP);
         return NULL;
      }
      sq_change_text_domain('server_settings_backend');
   
      // show connect details when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("Failed to retrieve file list of \"%s\" from FTP host \"%s\""), $directory, $host);
      else
         $error = _("Error retrieving file list");

      sq_change_text_domain('squirrelmail');
   }

   else
   {

      // if we can't determine the system type, get
      // a basic directory listing instead
      //
      if (!($host_system_type = ftp_systype($FTP))
//TODO: what are the possible values here???  how will windoze-based systems' raw list be formatted differently than *nix ones?  need feedback
        || ($host_system_type != 'UNIX'
         && $host_system_type != 'Windows_NT'))
      {
         $host_system_type = 'HOST SYSTEM TYPE UNKONWN';

         if (!($file_list = sq_call_function_suppress_errors('ftp_nlist', array($FTP, $directory))))
         {
            if ($quiet)
            {
               if ($close_connection) ftp_close($FTP);
               else @ftp_chdir($cwd);
               return NULL;
            }
            sq_change_text_domain('server_settings_backend');
   
            // show connect details when in debug mode
            //
            global $sm_debug_mode, $server_settings_backend_debug;
            server_settings_backend_init();
            if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
            if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
               $error = sprintf(_("Failed to retrieve file list of \"%s\" from FTP host \"%s\""), $directory, $host);
            else
               $error = _("Error retrieving file list");
      
            sq_change_text_domain('squirrelmail');
         }
      }


      if (empty($error))
      {

         $parsed_file_list = array();
         foreach ($file_list as $file)
         {
            switch ($host_system_type)
            {
               case 'HOST SYSTEM TYPE UNKONWN':
                  $parsed_file_list[] = array('name' => $file);
                  break;


               case 'Windows_NT':
//TODO: this is untested
                  $parsed_file = parse_unix_style_ftp_raw_directory_listing($file, $quiet);
                  if (!empty($parsed_file))
                     $parsed_file_list[] = $parsed_file;
                  break;


               case 'UNIX':
                  $parsed_file = parse_unix_style_ftp_raw_directory_listing($file, $quiet);
                  if (!empty($parsed_file))
                     $parsed_file_list[] = $parsed_file;
                  break;
            }
         }

         if ($close_connection) ftp_close($FTP);
         return $parsed_file_list;

      }

   }


   // spew error
   //
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   @ftp_close($FTP);
   exit;

}



/**
  * Determines if a certain file or directory exists on a FTP server/account.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param mixed   $FTP           A handle to the FTP connection
  *                               if one has already been opened.
  *                               If not given as a resource (handle),
  *                               a new connection attempt will be made.
  *                               If a new connection was opened, it
  *                               will be closed herein; if not, the
  *                               connection will not be closed by this
  *                               function unless a fatal error occurs.
  *                               UPDATE: due to the use of connection
  *                               caching, we'll avoid closing the
  *                               connection under all circumstances.
  * @param string  $host          The target FTP hostname.
  * @param int     $port          The port to be used when connecting.
  * @param boolean $use_ssl       Whether or not to use SSL when connecting.
  * @param boolean $passive       Whether or not to use passive mode.
  * @param string  $user          The username to log in with.
  * @param string  $pass          The password to log in with.
  * @param string  $path          The path to inspect for existence.
  * @param boolean $return_info   When TRUE, information about the path
  *                               is returned instead of just a boolean
  *                               return value.
  * @param boolean $quiet         When TRUE, suppresses any error
  *                               messages and makes this function
  *                               return NULL without doing anything
  *                               else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE if the path exists and $return_info is FALSE,
  *               FALSE if the path does not exist or NULL if an error
  *               occurred and $quiet is TRUE, or when the path exists and
  *               $return_info is TRUE, an array containing the path's
  *               file/directory attributes is returned.  This array
  *               will contain possibly several different key-value pairs,
  *               each for a different file attribute, but should always
  *               at least contain a "name" key whose value is the file
  *               or directory name.  Other possible attributes are:
  *               "type" ("file", "directory", "link", "pipe", "socket"),
  *               "owner", "group", "size", "permissions", "date_modified"
  *               (date last modified), "date_accessed" (date last accessed),
  *               "date_changed" (date last changed), "link_saki" (link
  *               destination), "numeric_permissions".
  *
  */
function ftp_path_exists($FTP, $host, $port, $use_ssl, $passive, $user,
                         $pass, $path, $return_info, $quiet=FALSE)
{

   $error = '';
   $close_connection = FALSE;
   if ($path[strlen($path) - 1] == '/'
    || $path[strlen($path) - 1] == '\\')
      $path = substr($path, 0, -1);
   $directory = dirname($path);
   $file = basename($path);



   // connect to FTP server
   //
   if (!is_resource($FTP))
   {
      $FTP = ssb_ftp_connect($host, $port, $use_ssl, $passive,
                             $user, $pass, '', $quiet);
      $close_connection = TRUE;

      // don't do anything if not connected
      //
      if (is_null($FTP))
         return NULL;
   }


   // override code above that might ask for closing
   // the connection when this function exits...
   // we want to cache all our connections, and closing
   // a connection will b0rk someone else who wants to
   // use the same connection
   //
   $close_connection = FALSE;


   // if we don't need any details to return, just get a simple file list
   //
   if (!$return_info)
   {

      if (!($file_list = sq_call_function_suppress_errors('ftp_nlist', array($FTP, $directory))))
      {
         if ($quiet)
         {
            if ($close_connection) ftp_close($FTP);
            return NULL;
         }
         sq_change_text_domain('server_settings_backend');
  
         // show connect details when in debug mode
         //
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Failed to retrieve file list of \"%s\" from FTP host \"%s\""), $directory, $host);
         else
            $error = _("Error retrieving file list");
 
         sq_change_text_domain('squirrelmail');
      }

      if (empty($error))
      {
         if ($close_connection) ftp_close($FTP);
         return (in_array($file, $file_list));
      }

   }


   // get a normal directory listing and look for our target therein
   //
   if (empty($error))
   {
      $file_list = ftp_get_directory_listing($FTP, $host, $port, $use_ssl,
                                             $passive, $user, $pass,
                                             $directory, $quiet);

      if (!is_array($file_list))
      {
         if ($quiet)
         {
            if ($close_connection) ftp_close($FTP);
            return NULL;
         }
         sq_change_text_domain('server_settings_backend');
      
         // show connect details when in debug mode
         //
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Failed to retrieve file list of \"%s\" from FTP host \"%s\""), $directory, $host);
         else
            $error = _("Error retrieving file list");
   
         sq_change_text_domain('squirrelmail');
      }
      else
      {
         foreach ($file_list as $file_info)
         {
            if (!empty($file_info['name'])
             && $file == $file_info['name'])
            {
               if ($close_connection) ftp_close($FTP);
               return $file_info;
            }
         }
         if ($close_connection) ftp_close($FTP);
         return FALSE;
      }

   }


   // spew error
   //
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   @ftp_close($FTP);
   exit;

}



/**   
  * Parses a (space-delimited) unix-style file listing
  *      
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *   
  * @param string $file_info The file listing information all
  *                          crammed into one string (to be parsed).
  * @param boolean $quiet    When TRUE, suppresses any error
  *                          messages and makes this function
  *                          return NULL without doing anything
  *                          else (OPTIONAL; default = FALSE).
  *
  * @return mixed NULL when an error occurred and $quiet is TRUE, otherwise
  *               an array is returned containing several different key-value
  *               pairs, each for a different file attribute.  At a minimum,
  *               for any actual file or directory, it should always contain
  *               a "name" key whose value is the file or directory name.
  *               Other possible attributes are: "type" ("file", "directory",
  *               "link", "pipe", "socket"), "owner", "group", "size",
  *               "permissions", "date_modified" (date last modified),
  *               "date_accessed" (date last accessed), "date_changed"
  *               (date last changed), "link_saki" (link destination),
  *               "numeric_permissions".  If there is any junk or unexpected
  *               formatting in the given $file_info input, it may be
  *               ignored and an empty array returned if possible, and
  *               errors are only triggered if more serious problems occur.
  *            
  */           
function parse_unix_style_ftp_raw_directory_listing($file_info, $quiet=FALSE)
{

   $error = '';

   // usually there are nine fields
   //
   $attributes = preg_split("/\s+/", $file_info, 9);
   foreach ($attributes as $key => $value) $attributes[$key] = trim($value);
   if (strtolower($attributes[0]) == 'total' || sizeof($attributes) != 9)
      return array();

   $file_attributes['permissions']     = $attributes[0];
   $file_attributes['number_of_links'] = $attributes[1];
   $file_attributes['size']            = $attributes[4];
   $file_attributes['date_modified']   = strtotime($attributes[5] . ' '
                                                 . $attributes[6] . ' '
                                                 . $attributes[7]);
   $file_attributes['name']            = $attributes[8];

   if (is_numeric($attributes[2]))
      $file_attributes['uid'] = $attributes[2];
   else
      $file_attributes['owner'] = $attributes[2];

   if (is_numeric($attributes[3]))
      $file_attributes['gid'] = $attributes[3];
   else
      $file_attributes['group'] = $attributes[3];

   if (preg_match('/(.*?)\s*->\s*(.*)/', $file_attributes['name'], $matches))
   {
      if (!empty($matches[1]))
         $file_attributes['name'] = $matches[1];
      if (!empty($matches[2]))
         $file_attributes['link_saki'] = $matches[2];
   }

   if ($file_attributes['permissions'][0] == 'd')
      $file_attributes['type'] = 'directory';
   else if ($file_attributes['permissions'][0] == 'l')
      $file_attributes['type'] = 'link';
   else if ($file_attributes['permissions'][0] == 's')
      $file_attributes['type'] = 'socket';
   else if ($file_attributes['permissions'][0] == 'p')
      $file_attributes['type'] = 'pipe';
   else
      $file_attributes['type'] = 'file';  // -, b, c

   // parse_file_permissions is found in functions.php
   //
   $file_attributes['numeric_permissions'] = decoct(parse_permissions($file_attributes['permissions']));

   //$file_attributes['date_string'] = date('Y/m/d H:i', $file_attributes['date_modified']);
   //$file_attributes['date_string'] = date('M d, Y g:ia', $file_attributes['date_modified']);

   return $file_attributes;
            
} 



/**
  * Changes permissions on a file or directory on a FTP server/account.
  *
  * Implemented using a SITE CHMOD command, which is supported on
  * many, but not all, FTP servers.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param mixed   $FTP           A handle to the FTP connection
  *                               if one has already been opened.
  *                               If not given as a resource (handle),
  *                               a new connection attempt will be made.
  *                               If a new connection was opened, it
  *                               will be closed herein; if not, the
  *                               connection will not be closed by this
  *                               function unless a fatal error occurs.
  *                               UPDATE: due to the use of connection
  *                               caching, we'll avoid closing the
  *                               connection under all circumstances.
  * @param string  $host          The target FTP hostname.
  * @param int     $port          The port to be used when connecting.
  * @param boolean $use_ssl       Whether or not to use SSL when connecting.
  * @param boolean $passive       Whether or not to use passive mode.
  * @param string  $user          The username to log in with.
  * @param string  $pass          The password to log in with.
  * @param string  $path          The file or directory to apply the
  *                               permissions to; should be a full path.
  * @param int     $mode          The octal permissions number representing
  *                               the permissions to apply.
  * @param boolean $quiet         When TRUE, suppresses any error
  *                               messages and makes this function
  *                               return NULL without doing anything
  *                               else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when completed normally, or NULL if an error
  *               occurred (and $quiet is TRUE).
  *
  */
function ssb_ftp_chmod($FTP, $host, $port, $use_ssl, $passive, $user,
                       $pass, $path, $mode, $quiet=FALSE)
{

   $error = '';
   $close_connection = FALSE;
   if ($path[strlen($path) - 1] == '/'
    || $path[strlen($path) - 1] == '\\')
      $path = substr($path, 0, -1);


   // connect to FTP server
   //
   if (!is_resource($FTP))
   {
      $FTP = ssb_ftp_connect($host, $port, $use_ssl, $passive,
                             $user, $pass, '', $quiet);
      $close_connection = TRUE;

      // don't do anything if not connected
      //
      if (is_null($FTP))
         return NULL;
   }


   // override code above that might ask for closing
   // the connection when this function exits...
   // we want to cache all our connections, and closing
   // a connection will b0rk someone else who wants to
   // use the same connection
   //
   $close_connection = FALSE;


   // use PHP5's native command
   //
   if (function_exists('ftp_chmod') && ftp_chmod($FTP, $mode, $path) === FALSE)
   {
      if ($quiet)
      {
         if ($close_connection) ftp_close($FTP);
         return NULL;
      }
      sq_change_text_domain('server_settings_backend');
   
      // show connect details when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("Failed to change permissions on \"%s\" on FTP host \"%s\""), $path, $host);
      else
         $error = _("Error changing permissions");

      sq_change_text_domain('squirrelmail');
   }


   // PHP4 - use ftp_site()
   //
   else if (!function_exists('ftp_chmod')
         && !ftp_site($FTP, sprintf('CHMOD %o %s', $mode, $path)))
   {
      if ($quiet)
      {
         if ($close_connection) ftp_close($FTP);
         return NULL;
      }
      sq_change_text_domain('server_settings_backend');
   
      // show connect details when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("Failed to change permissions on \"%s\" on FTP host \"%s\""), $path, $host);
      else
         $error = _("Error changing permissions");

      sq_change_text_domain('squirrelmail');
   }


   // success
   //
   if (empty($error))
   {
      if ($close_connection) ftp_close($FTP);
      return TRUE;
   }


   // spew error
   //
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   @ftp_close($FTP);
   exit;

}


 
/**
  * Creates a directory on a FTP server/account.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param mixed   $FTP           A handle to the FTP connection
  *                               if one has already been opened.
  *                               If not given as a resource (handle),
  *                               a new connection attempt will be made.
  *                               If a new connection was opened, it
  *                               will be closed herein; if not, the
  *                               connection will not be closed by this
  *                               function unless a fatal error occurs.
  *                               UPDATE: due to the use of connection
  *                               caching, we'll avoid closing the
  *                               connection under all circumstances.
  * @param string  $host          The target FTP hostname.
  * @param int     $port          The port to be used when connecting.
  * @param boolean $use_ssl       Whether or not to use SSL when connecting.
  * @param boolean $passive       Whether or not to use passive mode.
  * @param string  $user          The username to log in with.
  * @param string  $pass          The password to log in with.
  * @param string  $directory     The directory to create.
  * @param int     $mode          The octal permissions number to
  *                               apply to the newly created directory,
  *                               which may be ignored depending on the
  *                               FTP server.
  * @param boolean $quiet         When TRUE, suppresses any error
  *                               messages and makes this function
  *                               return NULL without doing anything
  *                               else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when completed normally, or NULL if an error
  *               occurred (and $quiet is TRUE).
  *
  */
function ssb_ftp_mkdir($FTP, $host, $port, $use_ssl, $passive,
                       $user, $pass, $directory, $mode, $quiet=FALSE)
{

   $error = '';
   $close_connection = FALSE;
   if ($directory[strlen($directory) - 1] == '/'
    || $directory[strlen($directory) - 1] == '\\')
      $directory = substr($directory, 0, -1);


   // connect to FTP server
   //
   if (!is_resource($FTP))
   {
      $FTP = ssb_ftp_connect($host, $port, $use_ssl, $passive,
                             $user, $pass, '', $quiet);
      $close_connection = TRUE;

      // don't do anything if not connected
      //
      if (is_null($FTP))
         return NULL;
   }


   // override code above that might ask for closing
   // the connection when this function exits...
   // we want to cache all our connections, and closing
   // a connection will b0rk someone else who wants to
   // use the same connection
   //
   $close_connection = FALSE;


   // create the directory
   //
   if (!sq_call_function_suppress_errors('ftp_mkdir', array($FTP, $directory)))
   {
      if ($quiet)
      {
         if ($close_connection) ftp_close($FTP);
         return NULL;
      }
      sq_change_text_domain('server_settings_backend');
   
      // show connect details when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("Failed to create directory on FTP host \"%s\""), $host);
      else
         $error = _("Error creating directory");

      sq_change_text_domain('squirrelmail');
   }


   // attempt to change the permissions on the new directory
   //
   else if (!ssb_ftp_chmod($FTP, $host, $port, $use_ssl, $passive,
                           $user, $pass, $directory, $mode, TRUE))
   {
      /* no-op, we'll just ignore problems setting permissions
         since some servers don't support it */
   }


   // success
   //
   if (empty($error))
   {
      if ($close_connection) ftp_close($FTP);
      return TRUE;
   }


   // spew error
   //
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   @ftp_close($FTP);
   exit;

}



/**
  * Rename/move a file or directory on a FTP server/account.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param mixed   $FTP             A handle to the FTP connection
  *                                 if one has already been opened.
  *                                 If not given as a resource (handle),
  *                                 a new connection attempt will be made.
  *                                 If a new connection was opened, it
  *                                 will be closed herein; if not, the
  *                                 connection will not be closed by this
  *                                 function unless a fatal error occurs.
  *                                 UPDATE: due to the use of connection
  *                                 caching, we'll avoid closing the
  *                                 connection under all circumstances.
  * @param string  $host            The target FTP hostname.
  * @param int     $port            The port to be used when connecting.
  * @param boolean $use_ssl         Whether or not to use SSL when connecting.
  * @param boolean $passive         Whether or not to use passive mode.
  * @param string  $user            The username to log in with.
  * @param string  $pass            The password to log in with.
  * @param string  $directory       The directory the source file/directory
  *                                 is in.  If empty, no directory change is
  *                                 attempted.  It is acceptable to specify
  *                                 nothing here and give a full path for
  *                                 $source.
  * @param string  $source          The source (remote) FTP file/directory
  *                                 name.
  * @param string  $destination     The destination (remote) FTP
  *                                 file/directory path.
  * @param boolean $allow_overwrite When TRUE, any existing file at
  *                                 $destination will be overwritten,
  *                                 otherwise, an existing file will
  *                                 create an error.
  * @param boolean $quiet           When TRUE, suppresses any error
  *                                 messages and makes this function
  *                                 return NULL without doing anything
  *                                 else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when completed normally, or NULL if an error
  *               occurred (and $quiet is TRUE).
  *
  */
function ssb_ftp_rename($FTP, $host, $port, $use_ssl, $passive, $user, $pass,
                        $directory, $source, $destination, $allow_overwrite,
                        $quiet=FALSE)
{

   $error = '';
   $close_connection = FALSE;
   $cwd = '.';
   if ($source[strlen($source) - 1] == '/'
    || $source[strlen($source) - 1] == '\\')
      $source = substr($source, 0, -1);
   if ($destination[strlen($destination) - 1] == '/'
    || $destination[strlen($destination) - 1] == '\\')
      $destination = substr($destination, 0, -1);
//TODO: path separator below may not work under all O/Ses(?)
   $full_source_path = (empty($directory) ? $source : $directory . '/' . $source);
   $source_directory = dirname($full_source_path);
   $source_file = basename($full_source_path);


   // connect to FTP server
   //
   if (!is_resource($FTP))
   {
      $FTP = ssb_ftp_connect($host, $port, $use_ssl, $passive,
                             $user, $pass, $directory, $quiet);
      $close_connection = TRUE;

      // don't do anything if not connected
      //
      if (is_null($FTP))
         return NULL;
   }


   // if already connected, note directory we started in
   // and then make sure we're in the right directory
   //
   else if (!($cwd = ftp_pwd($FTP)) || (!empty($directory) && !sq_call_function_suppress_errors('ftp_chdir', array($FTP, $directory))))
   {
      if ($quiet) return NULL;
      sq_change_text_domain('server_settings_backend');

      // show connect details when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("Failed to change to directory \"%s\" on FTP host \"%s\""), $directory, $host);
      else
         $error = _("Unable to change directories on FTP server");

      sq_change_text_domain('squirrelmail');
   }


   // override code above that might ask for closing
   // the connection when this function exits...
   // we want to cache all our connections, and closing
   // a connection will b0rk someone else who wants to
   // use the same connection
   //
   $close_connection = FALSE;


   // check if the source file/directory exists in the first place
   //
   if (empty($error)
    && !ftp_path_exists($FTP, $host, $port, $use_ssl, $passive, $user,
                        $pass, $full_source_path, FALSE, $quiet))
   {
      if ($quiet)
      {
         if ($close_connection) ftp_close($FTP);
         else @ftp_chdir($cwd);
         return NULL;
      }
      sq_change_text_domain('server_settings_backend');
      
      // show connect details when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("File \"%s\" not found in directory \"%s\" on FTP host \"%s\""), $source_file, $source_directory, $host);
      else
         $error = _("File not found");
   
      sq_change_text_domain('squirrelmail');
   }


   // if overwriting not allowed, bail if destination already exists
   //
   if (empty($error)
    && !$allow_overwrite
    && ftp_path_exists($FTP, $host, $port, $use_ssl, $passive, $user,
                       $pass, $destination, FALSE, $quiet))
   {
      if ($quiet)
      {
         if ($close_connection) ftp_close($FTP);
         else @ftp_chdir($cwd);
         return NULL;
      }
      sq_change_text_domain('server_settings_backend');

      // show connect details when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("File or directory \"%s\" already exists on FTP host \"%s\"; cannot overwrite"), $destination, $host);
      else
         $error = _("File or directory already exists; cannot overwrite");

      sq_change_text_domain('squirrelmail');
   }


   // now, actually rename the file/directory
   //
   if (empty($error)
    && !sq_call_function_suppress_errors('ftp_rename', array($FTP, $source, $destination)))
   {
      if ($quiet)
      {
         if ($close_connection) ftp_close($FTP);
         else @ftp_chdir($cwd);
         return NULL;
      }
      sq_change_text_domain('server_settings_backend');
     
      // show connect details when in debug mode
      //
      global $sm_debug_mode, $server_settings_backend_debug;
      server_settings_backend_init();
      if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
      if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
         $error = sprintf(_("Could not rename file or directory \"%s\" in directory \"%s\" on FTP host \"%s\" to \"%s\""), $source, $directory, $host, $destination);
      else
         $error = _("FTP error when attempting to rename file or directory");

      sq_change_text_domain('squirrelmail');
   }
   

   // success
   //
   if (empty($error))
   {
      if ($close_connection) ftp_close($FTP);
      else @ftp_chdir($cwd);
      return TRUE;
   }


   // spew error
   //
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   @ftp_close($FTP);
   exit;

}



/**
  * Connects to a FTP server, logs in, sets the right
  * PASV mode, and changes to the needed directory.
  *
  * If a connection has already been made to the same
  * user account on the same FTP server, a cached connection
  * handle will be returned (but the directory change will
  * still be made as requested).
  *
  * @param string  $host      The target FTP hostname.
  * @param int     $port      The port to be used when connecting.
  * @param boolean $use_ssl   Whether or not to use SSL when connecting.
  * @param boolean $passive   Whether or not to use passive mode.
  * @param string  $user      The username to log in with.
  * @param string  $pass      The password to log in with.
  * @param string  $directory The directory to change to after login.
  *                           If empty, no directory change is attempted.
  * @param boolean $quiet     When TRUE, suppresses any error
  *                           messages and makes this function
  *                           return NULL without doing anything
  *                           else (OPTIONAL; default = FALSE).
  *
  * @return mixed The FTP handle if all operations succeeded,
  *               or NULL if an error occurred (and $quiet is TRUE).
  *               
  */
function ssb_ftp_connect($host, $port, $use_ssl, $passive, $user,
                         $pass, $directory, $quiet=FALSE)
{

   // cache connections
   //
   global $ssb_ftp_connections;


   // make a new connection if needed; exit if failure
   //
//$ssb_ftp_connections[$host . $port . $user] = FALSE;  // for debugging
   if (empty($ssb_ftp_connections[$host . $port . $user])
    || !is_resource($ssb_ftp_connections[$host . $port . $user]))
   {

      // connect to server
      //
      if ($use_ssl) $connect_function = 'ftp_ssl_connect';
      else $connect_function = 'ftp_connect';
      if (!($FTP = $connect_function($host, $port)))
      {
         if ($quiet) return NULL;
         sq_change_text_domain('server_settings_backend');

         // show connect details when in debug mode
         //
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Could not connect to FTP host \"%s\" on port \"%s\" (connecting with \"%s\")"), $host, $port, $connect_function);
         else
            $error = _("Unable to connect to FTP server");

         sq_change_text_domain('squirrelmail');
      }


      // login...
      //
      else if (!ftp_login($FTP, $user, $pass))
      {
         if ($quiet)
         {
            @ftp_close($FTP);
            return NULL;
         }
         sq_change_text_domain('server_settings_backend');

         // show user/pwd and connect details when in debug mode
         //
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Failed FTP login to host \"%s\" with username \"%s\" and password \"%s\""), $host, $user, $pass);
         else
            $error = _("Unable to login to FTP server");

         sq_change_text_domain('squirrelmail');
      }


      // set passive mode
      //
      else if (!ftp_pasv($FTP, $passive))
      {
         if ($quiet)
         {
            @ftp_close($FTP);
            return NULL;
         }
         sq_change_text_domain('server_settings_backend');

         // show connect details when in debug mode
         //
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Failed to execute PASV command on FTP host \"%s\" (set to \"%s\")"), $host, $passive);
         else
            $error = _("Unable to call PASV command on FTP server");

         sq_change_text_domain('squirrelmail');
      }


      else $ssb_ftp_connections[$host . $port . $user] = $FTP;

   }


   // change directory if not in correct one
   //
   if (empty($error) && !empty($directory))
   {

      if (!($cwd = ftp_pwd($ssb_ftp_connections[$host . $port . $user]))
       || ($cwd != $directory 
        && !sq_call_function_suppress_errors('ftp_chdir', array($ssb_ftp_connections[$host . $port . $user], $directory))))
      {
         if ($quiet)
         {
            @ftp_close($ssb_ftp_connections[$host . $port . $user]);
            return NULL;
         }
         sq_change_text_domain('server_settings_backend');

         // show connect details when in debug mode
         //
         global $sm_debug_mode, $server_settings_backend_debug;
         server_settings_backend_init();
         if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
         if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
            $error = sprintf(_("Failed to change to directory \"%s\" on FTP host \"%s\" (current directory is \"%s\")"), $directory, $host, $cwd);
         else
            $error = _("Unable to change directories on FTP server");
   
         sq_change_text_domain('squirrelmail');
      }

   }


   // done
   //
   if (empty($error))
      return $ssb_ftp_connections[$host . $port . $user];


   // spit out error
   //
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   if (!empty($ssb_ftp_connections[$host . $port . $user]))
      @ftp_close($ssb_ftp_connections[$host . $port . $user]);
   exit;

}



