<?php

/**
  * SquirrelMail Server Settings Backend Plugin
  * Copyright (c) 2008-2010 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage server_settings_backend
  *
  */



/**
  * Gets a directory listing from the server.
  *
  * This function is only designed for use with backend types
  * that operate on a per-file basis, such as FTP and the
  * local_file backends.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $directory    The path to the desired directory.
  *                              This should typically be a FULL
  *                              (and not relative) directory path.
  * @param array   $backend_info An array of configuration values
  *                              that point to how the directory
  *                              listing should be retrieved.
  * @param boolean $use_lstat    When TRUE, lstat() is used when
  *                              gathering listing details, wherein
  *                              links themselves are stat'ed instead
  *                              of their targets; when FALSE, use
  *                              stat() instead (note that some backends
  *                              (FTP) do not support this option)
  *                              (OPTIONAL; default = TRUE)
  * @param boolean $quiet        When TRUE, suppresses any error
  *                              messages and makes this function
  *                              return NULL without doing anything
  *                              else (OPTIONAL; default = FALSE).
  *
  * @return mixed An array containing the directory listing if it
  *               was retrieved normally or NULL if it was not (and
  *               $quiet is TRUE).  The directory listing will consist
  *               of sub-arrays (one per file/directory) that themselves
  *               contain possibly several different key-value pairs,
  *               each for a different file attribute, but should always
  *               at least contain a "name" key whose value is the file
  *               or directory name.  Other possible attributes are:
  *               "type" ("file", "directory", "link", "pipe", "socket"),
  *               "owner", "group", "size", "permissions", "date" (date
  *               last modified), "link_saki" (link destination),
  *               "numeric_permissions".
  *
  */
function retrieve_directory_listing($directory, $backend_info, $use_lstat=TRUE, $quiet=FALSE)
{

   include_once(SM_PATH . 'plugins/server_settings_backend/init.php');
   include_once(SM_PATH . 'plugins/server_settings_backend/functions.php');

   global $username, $domain;
   $original_text_domain = sq_change_text_domain('server_settings_backend');
   $error = '';


   if (!is_array($backend_info))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to retrieve a directory listing from the server (malformed backend configuration)"), $backend_info['CUSTOM']);
   }


   // for sanity, make sure all array keys are upper case
   //
   else
   {
      $temp_backend_info = $backend_info;
      $backend_info = array();
      foreach ($temp_backend_info as $key => $value)
         $backend_info[strtoupper($key)] = $value;
   }


   // do we have an array of lookups?  we can assume so if
   // there is no VALUE, CUSTOM or BACKEND entry in the array
   //
   // when we have multiple lookups, just use the first one (multiple
   // lookups are only useful for performing multiple save actions)
   //
   if (empty($backend_info['VALUE']) && empty($backend_info['CUSTOM'])
    && empty($backend_info['BACKEND']))
      return retrieve_directory_listing($directory, array_shift($backend_info), $quiet);


   // start by getting the backend type
   //
   $backend = strtolower($backend_info['BACKEND']);
   if (empty($error) && empty($backend)
    || ($backend != 'ftp' && $backend != 'local_file' && $backend != 'sql' && $backend != 'ldap'))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to retrieve a directory listing from the server (\"%s\" lookup unknown)"), $backend);
   }


   // what kind of backend will we be using to retrieve the value?
   //
   if (empty($error)) switch ($backend)
   {

      // ----------------------------------
      //
      case 'sql':
      case 'ldap':
         $error = sprintf(_("Cannot retrieve a directory listing from a non-file-based backend type (\"%s\")"), $backend);
         break;



      // ----------------------------------
      //
      case 'ftp':

         include_once(SM_PATH . 'plugins/server_settings_backend/ftp_functions.php');


         // get host, port, passive mode, if should use SSL, etc
         //
         if (!empty($backend_info['PORT']) && is_array($backend_info['PORT']))
            $port = retrieve_server_setting($backend_info['PORT'], $quiet);
         else $port = 21;
         if (!empty($backend_info['PASSIVE']) && is_array($backend_info['PASSIVE']))
            $passive = retrieve_server_setting($backend_info['PASSIVE'], $quiet);
         else $passive = FALSE;
         if (!empty($backend_info['SSL']) && is_array($backend_info['SSL']))
         {
            $use_ssl = retrieve_server_setting($backend_info['SSL'], $quiet);
            if ($use_ssl && !function_exists('ftp_ssl_connect'))
               $error = _("Misconfiguration encountered when attempting to retrieve a directory listing from the server (SSL-FTP connection not possible when no SSL support is built into PHP)");
         }
         else $use_ssl = FALSE;
         if (!empty($backend_info['HOST']) && is_array($backend_info['HOST']))
            $host = retrieve_server_setting($backend_info['HOST'], $quiet);
         else
         {
            $error = _("Misconfiguration encountered when attempting to retrieve a directory listing from the server (FTP host is misconfigured)");
         }


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // check that the PHP FTP extension is available
         //
         if (!function_exists('ftp_connect'))  // even if using ssl; this is just a test
            $error = _("Misconfiguration encountered when attempting to retrieve a directory listing from the server (no FTP support is built into PHP)");


         // validate we have what we need
         //
         if (empty($error)
          && empty($host))
            $error = _("Misconfiguration encountered when attempting to retrieve a directory listing from the server (missing FTP host)");
         else if (empty($error))
         {

            // get our listing!
            //
            $ret = ftp_get_directory_listing(FALSE, $host, $port, $use_ssl,
                                             $passive, $user, $pass, $directory,
                                             $quiet);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;



      // ----------------------------------
      //
      case 'local_file':

         include_once(SM_PATH . 'plugins/server_settings_backend/local_file_functions.php');


         // get access type, suid info if needed
         //
         if (!empty($backend_info['ACCESS_TYPE']) && is_array($backend_info['ACCESS_TYPE']))
            $access_type = retrieve_server_setting($backend_info['ACCESS_TYPE'], $quiet);
         else
            $error = _("Misconfiguration encountered when attempting to retrieve a directory listing from the server (file access method is misconfigured)");
         if (!empty($backend_info['SUID_LOCATION']) && is_array($backend_info['SUID_LOCATION']))
            $suid_binary = retrieve_server_setting($backend_info['SUID_LOCATION'], $quiet);
         else if ($access_type == 'SUID')
            $error = _("Misconfiguration encountered when attempting to retrieve a directory listing from the server (SUID_LOCATION is misconfigured)");
         else $suid_binary = '';
         if (!empty($backend_info['SUID_DEBUG_OUTPUT']) && is_array($backend_info['SUID_DEBUG_OUTPUT']))
            $suid_debug_file = retrieve_server_setting($backend_info['SUID_DEBUG_OUTPUT'], $quiet);
         else
            $suid_debug_file = '';
         if (!empty($backend_info['MASTER_SECURITY_CHECK']) && is_array($backend_info['MASTER_SECURITY_CHECK']))
            $master_security_check = retrieve_server_setting($backend_info['MASTER_SECURITY_CHECK'], $quiet);
         else
            $master_security_check = FALSE;
         if (!empty($backend_info['MASTER_USERNAME']) && is_array($backend_info['MASTER_USERNAME']))
            $master_username = retrieve_server_setting($backend_info['MASTER_USERNAME'], $quiet);
         else
            $master_username = NULL;


         // if using "master" security check, grab needed values from environment
         //
         if ($master_security_check)
         {
            if (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_PASSWORD', $master_password, SQ_SERVER)
             || $master_password == 'test-password')
               $error = _("Misconfiguration encountered when attempting to retrieve a directory listing from the server (master password is misconfigured)");
            if ($master_username
             && (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_USERNAME', $master_username, SQ_SERVER)
              || $master_username == 'test-username'))
               $error = _("Misconfiguration encountered when attempting to retrieve a directory listing from the server (master username is misconfigured)");
         }
         else
            $master_username = $master_password = NULL;


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // validate we have what we need
         //
         if (empty($error)
          && (empty($access_type)
          || ($access_type == 'SUID' && empty($suid_binary))))
            $error = _("Misconfiguration encountered when attempting to retrieve a directory listing from the server (missing access type)");
         else if (empty($error))
         {

            // get our listing!
            //
            $directory = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $directory);
            $ret = get_local_directory_listing($access_type, $user, $pass, 
                                               $directory, $suid_binary,
                                               $suid_debug_file, $master_username,
                                               $master_password, $quiet, $use_lstat);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;

   }


   // should not get here without an error message
   // unless something is very wrong
   //
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function retrieve_directory_listing()");


   // spit out error and exit
   //
   if ($quiet) return NULL;
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Determines if a certain file or directory exists on the server.
  *
  * This function is only designed for use with backend types
  * that operate on a per-file basis, such as FTP and the
  * local_file backends.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $path         The path to the desired file or
  *                              directory.  This should typically
  *                              be a FULL (and not relative)
  *                              directory path.
  * @param boolean $return_info  When TRUE, return information
  *                              about the file/directory if it
  *                              exists instead of just a boolean
  *                              return value.
  * @param array   $backend_info An array of configuration values
  *                              that point to how the path should
  *                              be accessed.
  * @param boolean $quiet        When TRUE, suppresses any error
  *                              messages and makes this function
  *                              return NULL without doing anything
  *                              else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE if the path exists and $return_info is FALSE,
  *               NULL if the path does not exist or an error occurred
  *               and $quiet is TRUE, or when the path exists and
  *               $return_info is TRUE, an array containing the path's
  *               file/directory attributes is returned.  This array
  *               will contain possibly several different key-value pairs,
  *               each for a different file attribute, but should always
  *               at least contain a "name" key whose value is the file
  *               or directory name.  Other possible attributes are:
  *               "type" ("file", "directory", "link", "pipe", "socket"),
  *               "owner", "group", "size", "permissions", "date" (date
  *               last modified), "link_saki" (link destination),
  *               "numeric_permissions".
  *
  */
function path_exists($path, $return_info, $backend_info, $quiet=FALSE)
{

   include_once(SM_PATH . 'plugins/server_settings_backend/init.php');
   include_once(SM_PATH . 'plugins/server_settings_backend/functions.php');

   global $username, $domain;
   $original_text_domain = sq_change_text_domain('server_settings_backend');
   $error = '';


   if (!is_array($backend_info))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to determine the existence of a path on the server (malformed backend configuration)"), $backend_info['CUSTOM']);
   }


   // for sanity, make sure all array keys are upper case
   //
   else
   {
      $temp_backend_info = $backend_info;
      $backend_info = array();
      foreach ($temp_backend_info as $key => $value)
         $backend_info[strtoupper($key)] = $value;
   }


   // do we have an array of lookups?  we can assume so if
   // there is no VALUE, CUSTOM or BACKEND entry in the array
   //
   // when we have multiple lookups, just use the first one (multiple
   // lookups are only useful for performing multiple save actions)
   //
   if (empty($backend_info['VALUE']) && empty($backend_info['CUSTOM'])
    && empty($backend_info['BACKEND']))
      return path_exists($path, $return_info, array_shift($backend_info), $quiet);


   // start by getting the backend type
   //
   $backend = strtolower($backend_info['BACKEND']);
   if (empty($error) && empty($backend)
    || ($backend != 'ftp' && $backend != 'local_file' && $backend != 'sql' && $backend != 'ldap'))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to determine the existence of a path on the server (\"%s\" lookup unknown)"), $backend);
   }


   // what kind of backend will we be using to retrieve the value?
   //
   if (empty($error)) switch ($backend)
   {

      // ----------------------------------
      //
      case 'sql':
      case 'ldap':
         $error = sprintf(_("Cannot determine the existence of a path on a non-file-based backend type (\"%s\")"), $backend);
         break;



      // ----------------------------------
      //
      case 'ftp':

         include_once(SM_PATH . 'plugins/server_settings_backend/ftp_functions.php');


         // get host, port, passive mode, if should use SSL, etc
         //
         if (!empty($backend_info['PORT']) && is_array($backend_info['PORT']))
            $port = retrieve_server_setting($backend_info['PORT'], $quiet);
         else $port = 21;
         if (!empty($backend_info['PASSIVE']) && is_array($backend_info['PASSIVE']))
            $passive = retrieve_server_setting($backend_info['PASSIVE'], $quiet);
         else $passive = FALSE;
         if (!empty($backend_info['SSL']) && is_array($backend_info['SSL']))
         {
            $use_ssl = retrieve_server_setting($backend_info['SSL'], $quiet);
            if ($use_ssl && !function_exists('ftp_ssl_connect'))
               $error = _("Misconfiguration encountered when attempting to determine the existence of a path on the server (SSL-FTP connection not possible when no SSL support is built into PHP)");
         }
         else $use_ssl = FALSE;
         if (!empty($backend_info['HOST']) && is_array($backend_info['HOST']))
            $host = retrieve_server_setting($backend_info['HOST'], $quiet);
         else
         {
            $error = _("Misconfiguration encountered when attempting to determine the existence of a path on the server (FTP host is misconfigured)");
         }


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // check that the PHP FTP extension is available
         //
         if (!function_exists('ftp_connect'))  // even if using ssl; this is just a test
            $error = _("Misconfiguration encountered when attempting to determine the existence of a path on the server (no FTP support is built into PHP)");


         // validate we have what we need
         //
         if (empty($error)
          && empty($host))
            $error = _("Misconfiguration encountered when attempting to determine the existence of a path on the server (missing FTP host)");
         else if (empty($error))
         {

            // find the path!
            //
            $ret = ftp_path_exists(FALSE, $host, $port, $use_ssl,
                                   $passive, $user, $pass, $path,
                                   $return_info, $quiet);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;



      // ----------------------------------
      //
      case 'local_file':

         include_once(SM_PATH . 'plugins/server_settings_backend/local_file_functions.php');


         // get access type, suid info if needed
         //
         if (!empty($backend_info['ACCESS_TYPE']) && is_array($backend_info['ACCESS_TYPE']))
            $access_type = retrieve_server_setting($backend_info['ACCESS_TYPE'], $quiet);
         else
            $error = _("Misconfiguration encountered when attempting to determine the existence of a path on the server (file access method is misconfigured)");
         if (!empty($backend_info['SUID_LOCATION']) && is_array($backend_info['SUID_LOCATION']))
            $suid_binary = retrieve_server_setting($backend_info['SUID_LOCATION'], $quiet);
         else if ($access_type == 'SUID')
            $error = _("Misconfiguration encountered when attempting to determine the existence of a path on the server (SUID_LOCATION is misconfigured)");
         else $suid_binary = '';
         if (!empty($backend_info['SUID_DEBUG_OUTPUT']) && is_array($backend_info['SUID_DEBUG_OUTPUT']))
            $suid_debug_file = retrieve_server_setting($backend_info['SUID_DEBUG_OUTPUT'], $quiet);
         else
            $suid_debug_file = '';
         if (!empty($backend_info['MASTER_SECURITY_CHECK']) && is_array($backend_info['MASTER_SECURITY_CHECK']))
            $master_security_check = retrieve_server_setting($backend_info['MASTER_SECURITY_CHECK'], $quiet);
         else
            $master_security_check = FALSE;
         if (!empty($backend_info['MASTER_USERNAME']) && is_array($backend_info['MASTER_USERNAME']))
            $master_username = retrieve_server_setting($backend_info['MASTER_USERNAME'], $quiet);
         else
            $master_username = NULL;


         // if using "master" security check, grab needed values from environment
         //
         if ($master_security_check)
         {
            if (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_PASSWORD', $master_password, SQ_SERVER)
             || $master_password == 'test-password')
               $error = _("Misconfiguration encountered when attempting to determine the existence of a path on the server (master password is misconfigured)");
            if ($master_username
             && (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_USERNAME', $master_username, SQ_SERVER)
              || $master_username == 'test-username'))
               $error = _("Misconfiguration encountered when attempting to determine the existence of a path on the server (master username is misconfigured)");
         }
         else
            $master_username = $master_password = NULL;


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // validate we have what we need
         //
         if (empty($error)
          && (empty($access_type)
          || ($access_type == 'SUID' && empty($suid_binary))))
            $error = _("Misconfiguration encountered when attempting to determine the existence of a path on the server (missing access type)");
         else if (empty($error))
         {

            // find the path!
            //
            $path = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $path);
            $ret = local_file_path_exists($access_type, $user, $pass, $path,
                                          $return_info, $suid_binary,
                                          $suid_debug_file, $master_username,
                                          $master_password, $quiet);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;

   }


   // should not get here without an error message
   // unless something is very wrong
   //
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function path_exists()");


   // spit out error and exit
   //
   if ($quiet) return NULL;
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Creates a directory on the server.
//TODO -- THIS IS NOT RECURSIVE -- MAKE A RECURSIVE OPTION?
  *
  * This function is only designed for use with backend types
  * that operate on a per-file basis, such as FTP and the
  * local_file backends.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $directory    The path to the desired directory.
  *                              This should typically be a FULL
  *                              (and not relative) directory path.
  * @param int     $mode         The octal permissions number to
  *                              apply to the newly created directory,
  *                              which may be ignored depending on the
  *                              backend (using 0755 is a common default
  *                              you can use unless more restricted
  *                              access is desired).
  * @param array   $backend_info An array of configuration values
  *                              that point to how the directory
  *                              should be created.
  * @param boolean $quiet        When TRUE, suppresses any error
  *                              messages and makes this function
  *                              return NULL without doing anything
  *                              else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when the directory was created normally, or
  *               NULL if it was not (and $quiet is TRUE).
  *
  */
function create_directory($directory, $mode, $backend_info, $quiet=FALSE)
{

   include_once(SM_PATH . 'plugins/server_settings_backend/init.php');
   include_once(SM_PATH . 'plugins/server_settings_backend/functions.php');

   global $username, $domain;
   $original_text_domain = sq_change_text_domain('server_settings_backend');
   $error = '';


   if (!is_array($backend_info))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to create a directory on the server (malformed backend configuration)"), $backend_info['CUSTOM']);
   }


   // for sanity, make sure all array keys are upper case
   //
   else
   {
      $temp_backend_info = $backend_info;
      $backend_info = array();
      foreach ($temp_backend_info as $key => $value)
         $backend_info[strtoupper($key)] = $value;
   }


   // do we have an array of lookups?  we can assume so if
   // there is no VALUE, CUSTOM or BACKEND entry in the array
   //
   // when we have multiple lookups, just use the first one (multiple
   // lookups are only useful for performing multiple save actions)
   //
   if (empty($backend_info['VALUE']) && empty($backend_info['CUSTOM'])
    && empty($backend_info['BACKEND']))
      return create_directory($directory, $mode, array_shift($backend_info), $quiet);


   // start by getting the backend type
   //
   $backend = strtolower($backend_info['BACKEND']);
   if (empty($error) && empty($backend)
    || ($backend != 'ftp' && $backend != 'local_file' && $backend != 'sql' && $backend != 'ldap'))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to create a directory on the server (\"%s\" lookup unknown)"), $backend);
   }


   // what kind of backend will we be using to retrieve the value?
   //
   if (empty($error)) switch ($backend)
   {

      // ----------------------------------
      //
      case 'sql':
      case 'ldap':
         $error = sprintf(_("Cannot create a directory on a non-file-based backend type (\"%s\")"), $backend);
         break;



      // ----------------------------------
      //
      case 'ftp':

         include_once(SM_PATH . 'plugins/server_settings_backend/ftp_functions.php');


         // get host, port, passive mode, if should use SSL, etc
         //
         if (!empty($backend_info['PORT']) && is_array($backend_info['PORT']))
            $port = retrieve_server_setting($backend_info['PORT'], $quiet);
         else $port = 21;
         if (!empty($backend_info['PASSIVE']) && is_array($backend_info['PASSIVE']))
            $passive = retrieve_server_setting($backend_info['PASSIVE'], $quiet);
         else $passive = FALSE;
         if (!empty($backend_info['SSL']) && is_array($backend_info['SSL']))
         {
            $use_ssl = retrieve_server_setting($backend_info['SSL'], $quiet);
            if ($use_ssl && !function_exists('ftp_ssl_connect'))
               $error = _("Misconfiguration encountered when attempting to create a directory on the server (SSL-FTP connection not possible when no SSL support is built into PHP)");
         }
         else $use_ssl = FALSE;
         if (!empty($backend_info['HOST']) && is_array($backend_info['HOST']))
            $host = retrieve_server_setting($backend_info['HOST'], $quiet);
         else
         {
            $error = _("Misconfiguration encountered when attempting to create a directory on the server (FTP host is misconfigured)");
         }


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // check that the PHP FTP extension is available
         //
         if (!function_exists('ftp_connect'))  // even if using ssl; this is just a test
            $error = _("Misconfiguration encountered when attempting to create a directory on the server (no FTP support is built into PHP)");


         // validate we have what we need
         //
         if (empty($error)
          && empty($host))
            $error = _("Misconfiguration encountered when attempting to create a directory on the server (missing FTP host)");
         else if (empty($error))
         {

            // create directory!
            //
//TODO -- THIS IS NOT RECURSIVE -- MAKE A RECURSIVE OPTION?
            $ret = ssb_ftp_mkdir(FALSE, $host, $port, $use_ssl, $passive,
                                 $user, $pass, $directory, $mode, $quiet);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;



      // ----------------------------------
      //
      case 'local_file':

         include_once(SM_PATH . 'plugins/server_settings_backend/local_file_functions.php');


         // get access type, suid info if needed
         //
         if (!empty($backend_info['ACCESS_TYPE']) && is_array($backend_info['ACCESS_TYPE']))
            $access_type = retrieve_server_setting($backend_info['ACCESS_TYPE'], $quiet);
         else
            $error = _("Misconfiguration encountered when attempting to create a directory on the server (file access method is misconfigured)");
         if (!empty($backend_info['SUID_LOCATION']) && is_array($backend_info['SUID_LOCATION']))
            $suid_binary = retrieve_server_setting($backend_info['SUID_LOCATION'], $quiet);
         else if ($access_type == 'SUID')
            $error = _("Misconfiguration encountered when attempting to create a directory on the server (SUID_LOCATION is misconfigured)");
         else $suid_binary = '';
         if (!empty($backend_info['SUID_DEBUG_OUTPUT']) && is_array($backend_info['SUID_DEBUG_OUTPUT']))
            $suid_debug_file = retrieve_server_setting($backend_info['SUID_DEBUG_OUTPUT'], $quiet);
         else
            $suid_debug_file = '';
         if (!empty($backend_info['MASTER_SECURITY_CHECK']) && is_array($backend_info['MASTER_SECURITY_CHECK']))
            $master_security_check = retrieve_server_setting($backend_info['MASTER_SECURITY_CHECK'], $quiet);
         else
            $master_security_check = FALSE;
         if (!empty($backend_info['MASTER_USERNAME']) && is_array($backend_info['MASTER_USERNAME']))
            $master_username = retrieve_server_setting($backend_info['MASTER_USERNAME'], $quiet);
         else
            $master_username = NULL;


         // if using "master" security check, grab needed values from environment
         //
         if ($master_security_check)
         {
            if (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_PASSWORD', $master_password, SQ_SERVER)
             || $master_password == 'test-password')
               $error = _("Misconfiguration encountered when attempting to create a directory on the server (master password is misconfigured)");
            if ($master_username
             && (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_USERNAME', $master_username, SQ_SERVER)
              || $master_username == 'test-username'))
               $error = _("Misconfiguration encountered when attempting to create a directory on the server (master username is misconfigured)");
         }
         else
            $master_username = $master_password = NULL;


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // validate we have what we need
         //
         if (empty($error)
          && (empty($access_type)
          || ($access_type == 'SUID' && empty($suid_binary))))
            $error = _("Misconfiguration encountered when attempting to create a directory on the server (missing access type)");
         else if (empty($error))
         {

            // create directory!
            //
//TODO -- THIS IS NOT RECURSIVE -- MAKE A RECURSIVE OPTION -- NOTE: SUID backend already has a recursive command we can use, so it won't need special code here like FTP and PHP file access backends will (they need a copy of the SUID backend's mkdir_recursive() function, which is borrowed from the bash shell's source code)
            $directory = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $directory);
            $ret = local_file_mkdir($access_type, $user, $pass, $directory,
                                    $mode, $suid_binary, $suid_debug_file,
                                    $master_username, $master_password, $quiet);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;

   }


   // should not get here without an error message
   // unless something is very wrong
   //
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function create_directory()");


   // spit out error and exit
   //
   if ($quiet) return NULL;
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Changes the permissions of a file or directory on the server.
  *
  * This function is only designed for use with backend types
  * that operate on a per-file basis, such as FTP and the
  * local_file backends.
  *
  * Note that some backend types may not support this
  * functionality, in which case NULL is returned if at
  * all possible, but sometimes FALSE may be returned
  * in this case.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $path         The path to the desired file or
  *                              directory.  This should typically
  *                              be a FULL (and not relative)
  *                              directory path.
  * @param int     $mode         The octal permissions number to
  *                              apply.
  * @param array   $backend_info An array of configuration values
  *                              that point to how the file or
  *                              directory should be accessed.
  * @param boolean $quiet        When TRUE, suppresses any error
  *                              messages and makes this function
  *                              return NULL without doing anything
  *                              else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when the permissions were applied normally, or
  *               NULL or FALSE if they were not (and $quiet is TRUE).
  *
  */
function change_permissions($path, $mode, $backend_info, $quiet=FALSE)
{

   include_once(SM_PATH . 'plugins/server_settings_backend/init.php');
   include_once(SM_PATH . 'plugins/server_settings_backend/functions.php');

   global $username, $domain;
   $original_text_domain = sq_change_text_domain('server_settings_backend');
   $error = '';


   if (!is_array($backend_info))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to change the permissions of a file or directory on the server (malformed backend configuration)"), $backend_info['CUSTOM']);
   }


   // for sanity, make sure all array keys are upper case
   //
   else
   {
      $temp_backend_info = $backend_info;
      $backend_info = array();
      foreach ($temp_backend_info as $key => $value)
         $backend_info[strtoupper($key)] = $value;
   }


   // do we have an array of lookups?  we can assume so if
   // there is no VALUE, CUSTOM or BACKEND entry in the array
   //
   // when we have multiple lookups, just use the first one (multiple
   // lookups are only useful for performing multiple save actions)
   //
   if (empty($backend_info['VALUE']) && empty($backend_info['CUSTOM'])
    && empty($backend_info['BACKEND']))
      return change_permissions($path, $mode, array_shift($backend_info), $quiet);


   // start by getting the backend type
   //
   $backend = strtolower($backend_info['BACKEND']);
   if (empty($error) && empty($backend)
    || ($backend != 'ftp' && $backend != 'local_file' && $backend != 'sql' && $backend != 'ldap'))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to change the permissions of a file or directory on the server (\"%s\" lookup unknown)"), $backend);
   }


   // what kind of backend will we be using to retrieve the value?
   //
   if (empty($error)) switch ($backend)
   {

      // ----------------------------------
      //
      case 'sql':
      case 'ldap':
         $error = sprintf(_("Cannot change the permissions of a file or directory on a non-file-based backend type (\"%s\")"), $backend);
         break;



      // ----------------------------------
      //
      case 'ftp':

         include_once(SM_PATH . 'plugins/server_settings_backend/ftp_functions.php');


         // get host, port, passive mode, if should use SSL, etc
         //
         if (!empty($backend_info['PORT']) && is_array($backend_info['PORT']))
            $port = retrieve_server_setting($backend_info['PORT'], $quiet);
         else $port = 21;
         if (!empty($backend_info['PASSIVE']) && is_array($backend_info['PASSIVE']))
            $passive = retrieve_server_setting($backend_info['PASSIVE'], $quiet);
         else $passive = FALSE;
         if (!empty($backend_info['SSL']) && is_array($backend_info['SSL']))
         {
            $use_ssl = retrieve_server_setting($backend_info['SSL'], $quiet);
            if ($use_ssl && !function_exists('ftp_ssl_connect'))
               $error = _("Misconfiguration encountered when attempting to change the permissions of a file or directory on the server (SSL-FTP connection not possible when no SSL support is built into PHP)");
         }
         else $use_ssl = FALSE;
         if (!empty($backend_info['HOST']) && is_array($backend_info['HOST']))
            $host = retrieve_server_setting($backend_info['HOST'], $quiet);
         else
         {
            $error = _("Misconfiguration encountered when attempting to change the permissions of a file or directory on the server (FTP host is misconfigured)");
         }


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // check that the PHP FTP extension is available
         //
         if (!function_exists('ftp_connect'))  // even if using ssl; this is just a test
            $error = _("Misconfiguration encountered when attempting to change the permissions of a file or directory on the server (no FTP support is built into PHP)");


         // validate we have what we need
         //
         if (empty($error)
          && empty($host))
            $error = _("Misconfiguration encountered when attempting to change the permissions of a file or directory on the server (missing FTP host)");
         else if (empty($error))
         {

            // change permissions!
            //
            $ret = ssb_ftp_chmod(FALSE, $host, $port, $use_ssl, $passive,
                                 $user, $pass, $path, $mode, TRUE);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;



      // ----------------------------------
      //
      case 'local_file':

         include_once(SM_PATH . 'plugins/server_settings_backend/local_file_functions.php');


         // get access type, suid info if needed
         //
         if (!empty($backend_info['ACCESS_TYPE']) && is_array($backend_info['ACCESS_TYPE']))
            $access_type = retrieve_server_setting($backend_info['ACCESS_TYPE'], $quiet);
         else
            $error = _("Misconfiguration encountered when attempting to change the permissions of a file or directory on the server (file access method is misconfigured)");
         if (!empty($backend_info['SUID_LOCATION']) && is_array($backend_info['SUID_LOCATION']))
            $suid_binary = retrieve_server_setting($backend_info['SUID_LOCATION'], $quiet);
         else if ($access_type == 'SUID')
            $error = _("Misconfiguration encountered when attempting to change the permissions of a file or directory on the server (SUID_LOCATION is misconfigured)");
         else $suid_binary = '';
         if (!empty($backend_info['SUID_DEBUG_OUTPUT']) && is_array($backend_info['SUID_DEBUG_OUTPUT']))
            $suid_debug_file = retrieve_server_setting($backend_info['SUID_DEBUG_OUTPUT'], $quiet);
         else
            $suid_debug_file = '';
         if (!empty($backend_info['MASTER_SECURITY_CHECK']) && is_array($backend_info['MASTER_SECURITY_CHECK']))
            $master_security_check = retrieve_server_setting($backend_info['MASTER_SECURITY_CHECK'], $quiet);
         else
            $master_security_check = FALSE;
         if (!empty($backend_info['MASTER_USERNAME']) && is_array($backend_info['MASTER_USERNAME']))
            $master_username = retrieve_server_setting($backend_info['MASTER_USERNAME'], $quiet);
         else
            $master_username = NULL;


         // if using "master" security check, grab needed values from environment
         //
         if ($master_security_check)
         {
            if (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_PASSWORD', $master_password, SQ_SERVER)
             || $master_password == 'test-password')
               $error = _("Misconfiguration encountered when attempting to change the permissions of a file or directory on the server (master password is misconfigured)");
            if ($master_username
             && (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_USERNAME', $master_username, SQ_SERVER)
              || $master_username == 'test-username'))
               $error = _("Misconfiguration encountered when attempting to change the permissions of a file or directory on the server (master username is misconfigured)");
         }
         else
            $master_username = $master_password = NULL;


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // validate we have what we need
         //
         if (empty($error)
          && (empty($access_type)
          || ($access_type == 'SUID' && empty($suid_binary))))
            $error = _("Misconfiguration encountered when attempting to change the permissions of a file or directory on the server (missing access type)");
         else if (empty($error))
         {

            // change permissions!
            //
            $path = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $path);
            $ret = chmod_local_file_or_directory($access_type, $user, $pass,
                                                 $path, $mode, $suid_binary,
                                                 $suid_debug_file, $master_username,
                                                 $master_password, $quiet);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;

   }


   // should not get here without an error message
   // unless something is very wrong
   //
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function change_permissions()");


   // spit out error and exit
   //
   if ($quiet) return NULL;
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Deletes a file or directory from the server.
  *
  * This function is only designed for use with backend types
  * that operate on a per-file basis, such as FTP and the
  * local_file backends.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $path         The path to the desired file or directory.
  *                              This should typically be a FULL
  *                              (and not relative) directory path.
  * @param boolean $is_directory When TRUE, the entry being deleted
  *                              is assumed to be a directory, when
  *                              FALSE it is assumed to be a file.
  *                              Some (most) backends may (will) ignore
  *                              this and test the entry on their own
  *                              and act accordingly unless $autodetect
  *                              is FALSE.
  * @param array   $backend_info An array of configuration values
  *                              that point to how the path is accessed.
  * @param boolean $autodetect   When FALSE, $is_directory will be used
  *                              unquestioningly.  When TRUE, the $path
  *                              will be inspected to try to determine
  *                              if it is a file, directory, etc.
  *                              (OPTIONAL; default = TRUE)
  * @param boolean $quiet        When TRUE, suppresses any error
  *                              messages and makes this function
  *                              return NULL without doing anything
  *                              else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when the entry was deleted normally, or
  *               NULL if it was not (and $quiet is TRUE).
  *
  */
function delete_file_or_directory($path, $is_directory, $backend_info,
                                  $autodetect=TRUE, $quiet=FALSE)
{

   include_once(SM_PATH . 'plugins/server_settings_backend/init.php');
   include_once(SM_PATH . 'plugins/server_settings_backend/functions.php');

   global $username, $domain;
   $original_text_domain = sq_change_text_domain('server_settings_backend');
   $error = '';


   if (!is_array($backend_info))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to delete a file or directory from the server (malformed backend configuration)"), $backend_info['CUSTOM']);
   }


   // for sanity, make sure all array keys are upper case
   //
   else
   {
      $temp_backend_info = $backend_info;
      $backend_info = array();
      foreach ($temp_backend_info as $key => $value)
         $backend_info[strtoupper($key)] = $value;
   }


   // do we have an array of lookups?  we can assume so if
   // there is no VALUE, CUSTOM or BACKEND entry in the array
   //
   // when we have multiple lookups, just use the first one (multiple
   // lookups are only useful for performing multiple save actions)
   //
   if (empty($backend_info['VALUE']) && empty($backend_info['CUSTOM'])
    && empty($backend_info['BACKEND']))
      return delete_file_or_directory($path, $is_directory, array_shift($backend_info), $autodetect, $quiet);


   // start by getting the backend type
   //
   $backend = strtolower($backend_info['BACKEND']);
   if (empty($error) && empty($backend)
    || ($backend != 'ftp' && $backend != 'local_file' && $backend != 'sql' && $backend != 'ldap'))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to delete a file or directory from the server (\"%s\" lookup unknown)"), $backend);
   }


   // what kind of backend will we be using to retrieve the value?
   //
   if (empty($error)) switch ($backend)
   {

      // ----------------------------------
      //
      case 'sql':
      case 'ldap':
         $error = sprintf(_("Cannot delete a file or directory from a non-file-based backend type (\"%s\")"), $backend);
         break;



      // ----------------------------------
      //
      case 'ftp':

         include_once(SM_PATH . 'plugins/server_settings_backend/ftp_functions.php');


         // get host, port, passive mode, if should use SSL, etc
         //
         if (!empty($backend_info['PORT']) && is_array($backend_info['PORT']))
            $port = retrieve_server_setting($backend_info['PORT'], $quiet);
         else $port = 21;
         if (!empty($backend_info['PASSIVE']) && is_array($backend_info['PASSIVE']))
            $passive = retrieve_server_setting($backend_info['PASSIVE'], $quiet);
         else $passive = FALSE;
         if (!empty($backend_info['SSL']) && is_array($backend_info['SSL']))
         {
            $use_ssl = retrieve_server_setting($backend_info['SSL'], $quiet);
            if ($use_ssl && !function_exists('ftp_ssl_connect'))
               $error = _("Misconfiguration encountered when attempting to delete a file or directory from the server (SSL-FTP connection not possible when no SSL support is built into PHP)");
         }
         else $use_ssl = FALSE;
         if (!empty($backend_info['HOST']) && is_array($backend_info['HOST']))
            $host = retrieve_server_setting($backend_info['HOST'], $quiet);
         else
         {
            $error = _("Misconfiguration encountered when attempting to delete a file or directory from the server (FTP host is misconfigured)");
         }


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // check that the PHP FTP extension is available
         //
         if (!function_exists('ftp_connect'))  // even if using ssl; this is just a test
            $error = _("Misconfiguration encountered when attempting to delete a file or directory from the server (no FTP support is built into PHP)");


         // validate we have what we need
         //
         if (empty($error)
          && empty($host))
            $error = _("Misconfiguration encountered when attempting to delete a file or directory from the server (missing FTP host)");
         else if (empty($error))
         {

            // delete it!
            //
            $ret = ssb_ftp_delete(FALSE, $host, $port, $use_ssl,
                                  $passive, $user, $pass, '', $path,
                                  $is_directory, $autodetect, $quiet);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;



      // ----------------------------------
      //
      case 'local_file':

         include_once(SM_PATH . 'plugins/server_settings_backend/local_file_functions.php');


         // get access type, suid info if needed
         //
         if (!empty($backend_info['ACCESS_TYPE']) && is_array($backend_info['ACCESS_TYPE']))
            $access_type = retrieve_server_setting($backend_info['ACCESS_TYPE'], $quiet);
         else
            $error = _("Misconfiguration encountered when attempting to delete a file or directory from the server (file access method is misconfigured)");
         if (!empty($backend_info['SUID_LOCATION']) && is_array($backend_info['SUID_LOCATION']))
            $suid_binary = retrieve_server_setting($backend_info['SUID_LOCATION'], $quiet);
         else if ($access_type == 'SUID')
            $error = _("Misconfiguration encountered when attempting to delete a file or directory from the server (SUID_LOCATION is misconfigured)");
         else $suid_binary = '';
         if (!empty($backend_info['SUID_DEBUG_OUTPUT']) && is_array($backend_info['SUID_DEBUG_OUTPUT']))
            $suid_debug_file = retrieve_server_setting($backend_info['SUID_DEBUG_OUTPUT'], $quiet);
         else
            $suid_debug_file = '';
         if (!empty($backend_info['MASTER_SECURITY_CHECK']) && is_array($backend_info['MASTER_SECURITY_CHECK']))
            $master_security_check = retrieve_server_setting($backend_info['MASTER_SECURITY_CHECK'], $quiet);
         else
            $master_security_check = FALSE;
         if (!empty($backend_info['MASTER_USERNAME']) && is_array($backend_info['MASTER_USERNAME']))
            $master_username = retrieve_server_setting($backend_info['MASTER_USERNAME'], $quiet);
         else
            $master_username = NULL;


         // if using "master" security check, grab needed values from environment
         //
         if ($master_security_check)
         {
            if (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_PASSWORD', $master_password, SQ_SERVER)
             || $master_password == 'test-password')
               $error = _("Misconfiguration encountered when attempting to delete a file or directory from the server (master password is misconfigured)");
            if ($master_username
             && (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_USERNAME', $master_username, SQ_SERVER)
              || $master_username == 'test-username'))
               $error = _("Misconfiguration encountered when attempting to delete a file or directory from the server (master username is misconfigured)");
         }
         else
            $master_username = $master_password = NULL;


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // validate we have what we need
         //
         if (empty($error)
          && (empty($access_type)
          || ($access_type == 'SUID' && empty($suid_binary))))
            $error = _("Misconfiguration encountered when attempting to delete a file or directory from the server (missing access type)");
         else if (empty($error))
         {

            // delete it!
            //
            $path = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $path);
            $ret = local_file_delete($access_type, $user, $pass, $path, $is_directory,
                                     $autodetect, $suid_binary, $suid_debug_file,
                                     $master_username, $master_password, $quiet);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;

   }


   // should not get here without an error message
   // unless something is very wrong
   //
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function delete_file_or_directory()");


   // spit out error and exit
   //
   if ($quiet) return NULL;
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Creates a file on the server.
  *
  * This function is only designed for use with backend types
  * that operate on a per-file basis, such as FTP and the
  * local_file backends.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $path            The path to the desired file.
  *                                 This should typically be a FULL
  *                                 (and not relative) directory path.
  * @param string  $contents        What to put in the new file, if
  *                                 anything.
  * @param int     $mode            The octal permissions number to
  *                                 apply to the newly created file,
  *                                 which may be ignored depending on
  *                                 the backend (using 0644 is a
  *                                 common default you can use unless
  *                                 more restricted access is desired).
  * @param boolean $allow_overwrite When TRUE, any existing file at
  *                                 $path will be overwritten with
  *                                 $contents, otherwise, an existing
  *                                 file will create an error.
  * @param array   $backend_info    An array of configuration values
  *                                 that point to how the file should
  *                                 be created.
  * @param boolean $quiet           When TRUE, suppresses any error
  *                                 messages and makes this function
  *                                 return NULL without doing anything
  *                                 else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when the file was created normally, or
  *               NULL if it was not (and $quiet is TRUE).
  *
  */
function create_file($path, $contents, $mode, $allow_overwrite,
                     $backend_info, $quiet=FALSE)
{

   include_once(SM_PATH . 'plugins/server_settings_backend/init.php');
   include_once(SM_PATH . 'plugins/server_settings_backend/functions.php');

   global $username, $domain;
   $original_text_domain = sq_change_text_domain('server_settings_backend');
   $error = '';


   if (!is_array($backend_info))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to create a file on the server (malformed backend configuration)"), $backend_info['CUSTOM']);
   }


   // for sanity, make sure all array keys are upper case
   //
   else
   {
      $temp_backend_info = $backend_info;
      $backend_info = array();
      foreach ($temp_backend_info as $key => $value)
         $backend_info[strtoupper($key)] = $value;
   }


   // do we have an array of lookups?  we can assume so if
   // there is no VALUE, CUSTOM or BACKEND entry in the array
   //
   // when we have multiple lookups, just use the first one (multiple
   // lookups are only useful for performing multiple save actions)
   //
   if (empty($backend_info['VALUE']) && empty($backend_info['CUSTOM'])
    && empty($backend_info['BACKEND']))
      return create_file($path, $contents, $mode, $allow_overwrite,
                         array_shift($backend_info), $quiet);


   // start by getting the backend type
   //
   $backend = strtolower($backend_info['BACKEND']);
   if (empty($error) && empty($backend)
    || ($backend != 'ftp' && $backend != 'local_file' && $backend != 'sql' && $backend != 'ldap'))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to create a file on the server (\"%s\" lookup unknown)"), $backend);
   }


   // what kind of backend will we be using to retrieve the value?
   //
   if (empty($error)) switch ($backend)
   {

      // ----------------------------------
      //
      case 'sql':
      case 'ldap':
         $error = sprintf(_("Cannot create a file on a non-file-based backend type (\"%s\")"), $backend);
         break;



      // ----------------------------------
      //
      case 'ftp':

         include_once(SM_PATH . 'plugins/server_settings_backend/ftp_functions.php');


         // get host, port, ASCII/Binary, passive mode, if should use SSL, etc
         //
         if (!empty($backend_info['PORT']) && is_array($backend_info['PORT']))
            $port = retrieve_server_setting($backend_info['PORT'], $quiet);
         else $port = 21;
         if (!empty($backend_info['MODE']) && is_array($backend_info['MODE']))
            $transfer_mode = strtoupper(retrieve_server_setting($backend_info['MODE'], $quiet));
         else $transfer_mode = 'BINARY';
         if (!empty($backend_info['PASSIVE']) && is_array($backend_info['PASSIVE']))
            $passive = retrieve_server_setting($backend_info['PASSIVE'], $quiet);
         else $passive = FALSE;
         if (!empty($backend_info['SSL']) && is_array($backend_info['SSL']))
         {
            $use_ssl = retrieve_server_setting($backend_info['SSL'], $quiet);
            if ($use_ssl && !function_exists('ftp_ssl_connect'))
               $error = _("Misconfiguration encountered when attempting to create a file on the server (SSL-FTP connection not possible when no SSL support is built into PHP)");
         }
         else $use_ssl = FALSE;
         if (!empty($backend_info['HOST']) && is_array($backend_info['HOST']))
            $host = retrieve_server_setting($backend_info['HOST'], $quiet);
         else
         {
            $error = _("Misconfiguration encountered when attempting to create a file on the server (FTP host is misconfigured)");
         }


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // check that the PHP FTP extension is available
         //
         if (!function_exists('ftp_connect'))  // even if using ssl; this is just a test
            $error = _("Misconfiguration encountered when attempting to create a file on the server (no FTP support is built into PHP)");


         // validate we have what we need
         //
         if (empty($error)
          && (empty($host)
           || ($transfer_mode != 'BINARY' && $transfer_mode != 'ASCII')))
            $error = _("Misconfiguration encountered when attempting to create a file on the server (missing FTP host or mode is unknown)");
         else if (empty($error))
         {

            // create file!
            //
            $ret = ftp_put_file(FALSE, $contents, $allow_overwrite,
                                $host, $port, $use_ssl, $passive,
                                $transfer_mode, $user, $pass, '',
                                $path, FALSE, $quiet);
            if ($ret)
               // ignore return value since some FTP servers don't support CHMOD
               ssb_ftp_chmod(FALSE, $host, $port, $use_ssl, $passive,
                             $user, $pass, $path, $mode, TRUE);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;



      // ----------------------------------
      //
      case 'local_file':

         include_once(SM_PATH . 'plugins/server_settings_backend/local_file_functions.php');


         // get access type, suid info if needed
         //
         if (!empty($backend_info['ACCESS_TYPE']) && is_array($backend_info['ACCESS_TYPE']))
            $access_type = retrieve_server_setting($backend_info['ACCESS_TYPE'], $quiet);
         else
            $error = _("Misconfiguration encountered when attempting to create a file on the server (file access method is misconfigured)");
         if (!empty($backend_info['SUID_LOCATION']) && is_array($backend_info['SUID_LOCATION']))
            $suid_binary = retrieve_server_setting($backend_info['SUID_LOCATION'], $quiet);
         else if ($access_type == 'SUID')
            $error = _("Misconfiguration encountered when attempting to create a file on the server (SUID_LOCATION is misconfigured)");
         else $suid_binary = '';
         if (!empty($backend_info['SUID_DEBUG_OUTPUT']) && is_array($backend_info['SUID_DEBUG_OUTPUT']))
            $suid_debug_file = retrieve_server_setting($backend_info['SUID_DEBUG_OUTPUT'], $quiet);
         else
            $suid_debug_file = '';
         if (!empty($backend_info['MASTER_SECURITY_CHECK']) && is_array($backend_info['MASTER_SECURITY_CHECK']))
            $master_security_check = retrieve_server_setting($backend_info['MASTER_SECURITY_CHECK'], $quiet);
         else
            $master_security_check = FALSE;
         if (!empty($backend_info['MASTER_USERNAME']) && is_array($backend_info['MASTER_USERNAME']))
            $master_username = retrieve_server_setting($backend_info['MASTER_USERNAME'], $quiet);
         else
            $master_username = NULL;


         // if using "master" security check, grab needed values from environment
         //
         if ($master_security_check)
         {
            if (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_PASSWORD', $master_password, SQ_SERVER)
             || $master_password == 'test-password')
               $error = _("Misconfiguration encountered when attempting to create a file on the server (master password is misconfigured)");
            if ($master_username
             && (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_USERNAME', $master_username, SQ_SERVER)
              || $master_username == 'test-username'))
               $error = _("Misconfiguration encountered when attempting to create a file on the server (master username is misconfigured)");
         }
         else
            $master_username = $master_password = NULL;


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // validate we have what we need
         //
         if (empty($error)
          && (empty($access_type)
          || ($access_type == 'SUID' && empty($suid_binary))))
            $error = _("Misconfiguration encountered when attempting to create a file on the server (missing access type)");
         else if (empty($error))
         {

            // create file!
            //
            $path = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $path);
            $ret = put_local_file($access_type, $contents, $allow_overwrite,
                                  $user, $pass, $path, FALSE, $suid_binary,
                                  $suid_debug_file, $master_username,
                                  $master_password, $quiet);
            if ($ret)
               $ret = chmod_local_file_or_directory($access_type, $user, $pass,
                                                    $path, $mode, $suid_binary,
                                                    $suid_debug_file, $master_username,
                                                    $master_password, $quiet);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;

   }


   // should not get here without an error message
   // unless something is very wrong
   //
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function create_file()");


   // spit out error and exit
   //
   if ($quiet) return NULL;
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Gets the contents of a file on the server.
  *
  * This function is only designed for use with backend types
  * that operate on a per-file basis, such as FTP and the
  * local_file backends.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $path         The path to the desired file.
  *                              This should typically be a FULL
  *                              (and not relative) directory path.
  * @param array   $backend_info An array of configuration values
  *                              that point to how the file should
  *                              be created.
  * @param boolean $quiet        When TRUE, suppresses any error
  *                              messages and makes this function
  *                              return NULL without doing anything
  *                              else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when the file was created normally, or
  *               NULL if it was not (and $quiet is TRUE).
  *
  */
function get_file($path, $backend_info, $quiet=FALSE)
{

   include_once(SM_PATH . 'plugins/server_settings_backend/init.php');
   include_once(SM_PATH . 'plugins/server_settings_backend/functions.php');

   global $username, $domain;
   $original_text_domain = sq_change_text_domain('server_settings_backend');
   $error = '';


   if (!is_array($backend_info))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to retrieve a file from the server (malformed backend configuration)"), $backend_info['CUSTOM']);
   }


   // for sanity, make sure all array keys are upper case
   //
   else
   {
      $temp_backend_info = $backend_info;
      $backend_info = array();
      foreach ($temp_backend_info as $key => $value)
         $backend_info[strtoupper($key)] = $value;
   }


   // do we have an array of lookups?  we can assume so if
   // there is no VALUE, CUSTOM or BACKEND entry in the array
   //
   // when we have multiple lookups, just use the first one (multiple
   // lookups are only useful for performing multiple save actions)
   //
   if (empty($backend_info['VALUE']) && empty($backend_info['CUSTOM'])
    && empty($backend_info['BACKEND']))
      return get_file($path, array_shift($backend_info), $quiet);


   // start by getting the backend type
   //
   $backend = strtolower($backend_info['BACKEND']);
   if (empty($error) && empty($backend)
    || ($backend != 'ftp' && $backend != 'local_file' && $backend != 'sql' && $backend != 'ldap'))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to retrieve a file from the server (\"%s\" lookup unknown)"), $backend);
   }


   // what kind of backend will we be using to retrieve the value?
   //
   if (empty($error)) switch ($backend)
   {

      // ----------------------------------
      //
      case 'sql':
      case 'ldap':
         $error = sprintf(_("Cannot retrieve a file from a non-file-based backend type (\"%s\")"), $backend);
         break;



      // ----------------------------------
      //
      case 'ftp':

         include_once(SM_PATH . 'plugins/server_settings_backend/ftp_functions.php');


         // get host, port, ASCII/Binary, passive mode, if should use SSL, etc
         //
         if (!empty($backend_info['PORT']) && is_array($backend_info['PORT']))
            $port = retrieve_server_setting($backend_info['PORT'], $quiet);
         else $port = 21;
         if (!empty($backend_info['MODE']) && is_array($backend_info['MODE']))
            $mode = strtoupper(retrieve_server_setting($backend_info['MODE'], $quiet));
         else $mode = 'BINARY';
         if (!empty($backend_info['PASSIVE']) && is_array($backend_info['PASSIVE']))
            $passive = retrieve_server_setting($backend_info['PASSIVE'], $quiet);
         else $passive = FALSE;
         if (!empty($backend_info['SSL']) && is_array($backend_info['SSL']))
         {
            $use_ssl = retrieve_server_setting($backend_info['SSL'], $quiet);
            if ($use_ssl && !function_exists('ftp_ssl_connect'))
               $error = _("Misconfiguration encountered when attempting to retrieve a file from the server (SSL-FTP connection not possible when no SSL support is built into PHP)");
         }
         else $use_ssl = FALSE;
         if (!empty($backend_info['HOST']) && is_array($backend_info['HOST']))
            $host = retrieve_server_setting($backend_info['HOST'], $quiet);
         else
         {
            $error = _("Misconfiguration encountered when attempting to retrieve a file from the server (FTP host is misconfigured)");
         }


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // check that the PHP FTP extension is available
         //
         if (!function_exists('ftp_connect'))  // even if using ssl; this is just a test
            $error = _("Misconfiguration encountered when attempting to retrieve a file from the server (no FTP support is built into PHP)");


         // validate we have what we need
         //
         if (empty($error)
          && (empty($host)
           || ($mode != 'BINARY' && $mode != 'ASCII')))
            $error = _("Misconfiguration encountered when attempting to retrieve a file from the server (missing FTP host or mode is unknown)");
         else if (empty($error))
         {

            // retrieve file!
            //
            $local_file = ftp_get_file(FALSE, $host, $port, $use_ssl, $passive,
                                       $mode, $user, $pass, '', $path, FALSE,
                                       $quiet);

            // assume that unless $local_file is NULL,
            // the file exists and all is normal,
            // if not, return empty value
            //
            if (is_null($local_file)) $ret = NULL;
            else if (!($contents = @file($local_file))) $ret = '';
            else $ret = implode('', $contents);
            @unlink($local_file);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;



      // ----------------------------------
      //
      case 'local_file':

         include_once(SM_PATH . 'plugins/server_settings_backend/local_file_functions.php');


         // get access type, suid info if needed
         //
         if (!empty($backend_info['ACCESS_TYPE']) && is_array($backend_info['ACCESS_TYPE']))
            $access_type = retrieve_server_setting($backend_info['ACCESS_TYPE'], $quiet);
         else
            $error = _("Misconfiguration encountered when attempting to retrieve a file from the server (file access method is misconfigured)");
         if (!empty($backend_info['SUID_LOCATION']) && is_array($backend_info['SUID_LOCATION']))
            $suid_binary = retrieve_server_setting($backend_info['SUID_LOCATION'], $quiet);
         else if ($access_type == 'SUID')
            $error = _("Misconfiguration encountered when attempting to retrieve a file from the server (SUID_LOCATION is misconfigured)");
         else $suid_binary = '';
         if (!empty($backend_info['SUID_DEBUG_OUTPUT']) && is_array($backend_info['SUID_DEBUG_OUTPUT']))
            $suid_debug_file = retrieve_server_setting($backend_info['SUID_DEBUG_OUTPUT'], $quiet);
         else
            $suid_debug_file = '';
         if (!empty($backend_info['MASTER_SECURITY_CHECK']) && is_array($backend_info['MASTER_SECURITY_CHECK']))
            $master_security_check = retrieve_server_setting($backend_info['MASTER_SECURITY_CHECK'], $quiet);
         else
            $master_security_check = FALSE;
         if (!empty($backend_info['MASTER_USERNAME']) && is_array($backend_info['MASTER_USERNAME']))
            $master_username = retrieve_server_setting($backend_info['MASTER_USERNAME'], $quiet);
         else
            $master_username = NULL;


         // if using "master" security check, grab needed values from environment
         //
         if ($master_security_check)
         {
            if (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_PASSWORD', $master_password, SQ_SERVER)
             || $master_password == 'test-password')
               $error = _("Misconfiguration encountered when attempting to retrieve a file from the server (master password is misconfigured)");
            if ($master_username
             && (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_USERNAME', $master_username, SQ_SERVER)
              || $master_username == 'test-username'))
               $error = _("Misconfiguration encountered when attempting to retrieve a file from the server (master username is misconfigured)");
         }
         else
            $master_username = $master_password = NULL;


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // validate we have what we need
         //
         if (empty($error)
          && (empty($access_type)
          || ($access_type == 'SUID' && empty($suid_binary))))
            $error = _("Misconfiguration encountered when attempting to retrieve a file from the server (missing access type)");
         else if (empty($error))
         {

            // retrieve file!
            //
            $path = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $path);
            $local_file = get_local_file($access_type, $user, $pass, $path, FALSE,
                                         $suid_binary, $suid_debug_file,
                                         $master_username, $master_password, $quiet);

            // assume that unless $local_file is NULL,
            // the file exists and all is normal,
            // if not, return empty value
            //
            if (is_null($local_file)) $ret = NULL;
            else if (!($contents = @file($local_file))) $ret = '';
            else $ret = implode('', $contents);
            if ($access_type != 'PHP') @unlink($local_file);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;

   }


   // should not get here without an error message
   // unless something is very wrong
   //
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function get_file()");


   // spit out error and exit
   //
   if ($quiet) return NULL;
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Moves/renames a file or directory on the server.
  *
  * This function is only designed for use with backend types
  * that operate on a per-file basis, such as FTP and the
  * local_file backends.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $source_path      The path to the desired file or
  *                                  directory before moving/renaming.
  *                                  This should typically be a FULL
  *                                  (and not relative) directory path.
  * @param string  $destination_path The desired new file/directory
  *                                  path and name.  This should
  *                                  typically be a FULL (and not
  *                                  relative) directory path.  NOTE
  *                                  that any existing file at the
  *                                  destination location will be
  *                                  overwritten!
  * @param boolean $allow_overwrite  When TRUE, any existing file at
  *                                  $destination_path will be
  *                                  overwritten, otherwise an existing 
  *                                  file will create an error.
  * @param array   $backend_info     An array of configuration values
  *                                  that point to how the files/
  *                                  directories are accessed.
  * @param boolean $quiet            When TRUE, suppresses any error
  *                                  messages and makes this function
  *                                  return NULL without doing anything
  *                                  else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when the file or directory was renamed/moved
  *               normally, or NULL if it was not (and $quiet is TRUE).
  *
  */
function move_file_or_directory($source_path, $destination_path,
                                $backend_info, $allow_overwrite,
                                $quiet=FALSE)
{

   include_once(SM_PATH . 'plugins/server_settings_backend/init.php');
   include_once(SM_PATH . 'plugins/server_settings_backend/functions.php');

   global $username, $domain;
   $original_text_domain = sq_change_text_domain('server_settings_backend');
   $error = '';


   if (!is_array($backend_info))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to move or rename a file or directory on the server (malformed backend configuration)"), $backend_info['CUSTOM']);
   }


   // for sanity, make sure all array keys are upper case
   //
   else
   {
      $temp_backend_info = $backend_info;
      $backend_info = array();
      foreach ($temp_backend_info as $key => $value)
         $backend_info[strtoupper($key)] = $value;
   }


   // do we have an array of lookups?  we can assume so if
   // there is no VALUE, CUSTOM or BACKEND entry in the array
   //
   // when we have multiple lookups, just use the first one (multiple
   // lookups are only useful for performing multiple save actions)
   //
   if (empty($backend_info['VALUE']) && empty($backend_info['CUSTOM'])
    && empty($backend_info['BACKEND']))
      return move_file_or_directory($source_path, $destination_path,
                                    $allow_overwrite,
                                    array_shift($backend_info), $quiet);


   // start by getting the backend type
   //
   $backend = strtolower($backend_info['BACKEND']);
   if (empty($error) && empty($backend)
    || ($backend != 'ftp' && $backend != 'local_file' && $backend != 'sql' && $backend != 'ldap'))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to move or rename a file or directory on the server (\"%s\" lookup unknown)"), $backend);
   }


   // what kind of backend will we be using to retrieve the value?
   //
   if (empty($error)) switch ($backend)
   {

      // ----------------------------------
      //
      case 'sql':
      case 'ldap':
         $error = sprintf(_("Cannot move or rename a file or directory on a non-file-based backend type (\"%s\")"), $backend);
         break;



      // ----------------------------------
      //
      case 'ftp':

         include_once(SM_PATH . 'plugins/server_settings_backend/ftp_functions.php');


         // get host, port, passive mode, if should use SSL, etc
         //
         if (!empty($backend_info['PORT']) && is_array($backend_info['PORT']))
            $port = retrieve_server_setting($backend_info['PORT'], $quiet);
         else $port = 21;
         if (!empty($backend_info['PASSIVE']) && is_array($backend_info['PASSIVE']))
            $passive = retrieve_server_setting($backend_info['PASSIVE'], $quiet);
         else $passive = FALSE;
         if (!empty($backend_info['SSL']) && is_array($backend_info['SSL']))
         {
            $use_ssl = retrieve_server_setting($backend_info['SSL'], $quiet);
            if ($use_ssl && !function_exists('ftp_ssl_connect'))
               $error = _("Misconfiguration encountered when attempting to move or rename a file or directory on the server (SSL-FTP connection not possible when no SSL support is built into PHP)");
         }
         else $use_ssl = FALSE;
         if (!empty($backend_info['HOST']) && is_array($backend_info['HOST']))
            $host = retrieve_server_setting($backend_info['HOST'], $quiet);
         else
         {
            $error = _("Misconfiguration encountered when attempting to move or rename a file or directory on the server (FTP host is misconfigured)");
         }


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // check that the PHP FTP extension is available
         //
         if (!function_exists('ftp_connect'))  // even if using ssl; this is just a test
            $error = _("Misconfiguration encountered when attempting to move or rename a file or directory on the server (no FTP support is built into PHP)");


         // validate we have what we need
         //
         if (empty($error)
          && empty($host))
            $error = _("Misconfiguration encountered when attempting to move or rename a file or directory on the server (missing FTP host)");
         else if (empty($error))
         {

            // rename it!
            //
            $ret = ssb_ftp_rename(FALSE, $host, $port, $use_ssl, $passive,
                                  $user, $pass, '', $source_path,
                                  $destination_path, $allow_overwrite, $quiet);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;



      // ----------------------------------
      //
      case 'local_file':

         include_once(SM_PATH . 'plugins/server_settings_backend/local_file_functions.php');


         // get access type, suid info if needed
         //
         if (!empty($backend_info['ACCESS_TYPE']) && is_array($backend_info['ACCESS_TYPE']))
            $access_type = retrieve_server_setting($backend_info['ACCESS_TYPE'], $quiet);
         else
            $error = _("Misconfiguration encountered when attempting to move or rename a file or directory on the server (file access method is misconfigured)");
         if (!empty($backend_info['SUID_LOCATION']) && is_array($backend_info['SUID_LOCATION']))
            $suid_binary = retrieve_server_setting($backend_info['SUID_LOCATION'], $quiet);
         else if ($access_type == 'SUID')
            $error = _("Misconfiguration encountered when attempting to move or rename a file or directory on the server (SUID_LOCATION is misconfigured)");
         else $suid_binary = '';
         if (!empty($backend_info['SUID_DEBUG_OUTPUT']) && is_array($backend_info['SUID_DEBUG_OUTPUT']))
            $suid_debug_file = retrieve_server_setting($backend_info['SUID_DEBUG_OUTPUT'], $quiet);
         else
            $suid_debug_file = '';
         if (!empty($backend_info['MASTER_SECURITY_CHECK']) && is_array($backend_info['MASTER_SECURITY_CHECK']))
            $master_security_check = retrieve_server_setting($backend_info['MASTER_SECURITY_CHECK'], $quiet);
         else
            $master_security_check = FALSE;
         if (!empty($backend_info['MASTER_USERNAME']) && is_array($backend_info['MASTER_USERNAME']))
            $master_username = retrieve_server_setting($backend_info['MASTER_USERNAME'], $quiet);
         else
            $master_username = NULL;


         // if using "master" security check, grab needed values from environment
         //
         if ($master_security_check)
         {
            if (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_PASSWORD', $master_password, SQ_SERVER)
             || $master_password == 'test-password')
               $error = _("Misconfiguration encountered when attempting to move or rename a file or directory on the server (master password is misconfigured)");
            if ($master_username
             && (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_USERNAME', $master_username, SQ_SERVER)
              || $master_username == 'test-username'))
               $error = _("Misconfiguration encountered when attempting to move or rename a file or directory on the server (master username is misconfigured)");
         }
         else
            $master_username = $master_password = NULL;


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // validate we have what we need
         //
         if (empty($error)
          && (empty($access_type)
          || ($access_type == 'SUID' && empty($suid_binary))))
            $error = _("Misconfiguration encountered when attempting to move or rename a file or directory on the server (missing access type)");
         else if (empty($error))
         {

            // rename it!
            //
            $source_path = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $source_path);
            $destination_path = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $destination_path);
            $ret = local_file_rename($access_type, $user, $pass, $source_path,
                                     $destination_path, $allow_overwrite,
                                     $suid_binary, $suid_debug_file,
                                     $master_username, $master_password, $quiet);


            sq_change_text_domain($original_text_domain);
            return $ret;

         }
         break;

   }


   // should not get here without an error message
   // unless something is very wrong
   //
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function move_file_or_directory()");


   // spit out error and exit
   //
   if ($quiet) return NULL;
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Copies a file or directory on the server.
  *
  * This function does not copy links - they are ignored.
  *
  * This function is only designed for use with backend types
  * that operate on a per-file basis, such as FTP and the
  * local_file backends.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $source_path      The path to the desired file or
  *                                  directory that will be copied.
  *                                  This should typically be a FULL
  *                                  (and not relative) directory path.
  * @param string  $destination_path The desired new file/directory
  *                                  path and name.  This should
  *                                  typically be a FULL (and not
  *                                  relative) directory path.  NOTE
  *                                  that any existing file at the
  *                                  destination location will be
  *                                  overwritten!
  * @param boolean $is_directory     When TRUE, the entry being copied
  *                                  is assumed to be a directory, when
  *                                  FALSE it is assumed to be a file.
  *                                  Some (most) backends may (will) ignore
  *                                  this and test the entry on their own
  *                                  and act accordingly unless $autodetect
  *                                  is FALSE.
  * @param boolean $recursive        When TRUE, and a directory is being
  *                                  copied, any subdirectories and their
  *                                  contents are recursively copied as
  *                                  well; otherwise only the immediate
  *                                  contents of the directory (first
  *                                  level) are copied - no subdirectories
  *                                  will be included.
  * @param boolean $allow_overwrite  When TRUE, any existing file or
  *                                  directory at $destination_path will
  *                                  be overwritten, otherwise an existing 
  *                                  file/directory will create an error.
  * @param array   $backend_info     An array of configuration values
  *                                  that point to how the files/
  *                                  directories are accessed.
  * @param boolean $autodetect       When FALSE, $is_directory will be used
  *                                  unquestioningly.  When TRUE, the $path
  *                                  will be inspected to try to determine
  *                                  if it is a file, directory, etc.
  *                                  (OPTIONAL; default = TRUE)
  * @param boolean $quiet            When TRUE, suppresses any error
  *                                  messages and makes this function
  *                                  return NULL without doing anything
  *                                  else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE when the file or directory was copied normally,
  *               or NULL if it was not (and $quiet is TRUE).
  *
  */
function copy_file_or_directory($source_path, $destination_path,
                                $is_directory, $recursive, $allow_overwrite,
                                $backend_info, $autodetect=TRUE, $quiet=FALSE)
{

   include_once(SM_PATH . 'plugins/server_settings_backend/init.php');
   include_once(SM_PATH . 'plugins/server_settings_backend/functions.php');

   global $username, $domain;
   $original_text_domain = sq_change_text_domain('server_settings_backend');
   $error = '';


   if (!is_array($backend_info))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to copy a file or directory on the server (malformed backend configuration)"), $backend_info['CUSTOM']);
   }


   // for sanity, make sure all array keys are upper case
   //
   else
   {
      $temp_backend_info = $backend_info;
      $backend_info = array();
      foreach ($temp_backend_info as $key => $value)
         $backend_info[strtoupper($key)] = $value;
   }


   // do we have an array of lookups?  we can assume so if
   // there is no VALUE, CUSTOM or BACKEND entry in the array
   //
   // when we have multiple lookups, just use the first one (multiple
   // lookups are only useful for performing multiple save actions)
   //
   if (empty($backend_info['VALUE']) && empty($backend_info['CUSTOM'])
    && empty($backend_info['BACKEND']))
      return copy_file_or_directory($source_path, $destination_path,
                                    $is_directory, $recursive, $allow_overwrite,
                                    array_shift($backend_info), $autodetect, $quiet);


   // start by getting the backend type
   //
   $backend = strtolower($backend_info['BACKEND']);
   if (empty($error) && empty($backend)
    || ($backend != 'ftp' && $backend != 'local_file' && $backend != 'sql' && $backend != 'ldap'))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to copy a file or directory on the server (\"%s\" lookup unknown)"), $backend);
   }


   // what kind of backend will we be using to retrieve the value?
   //
   if (empty($error)) switch ($backend)
   {

      // ----------------------------------
      //
      case 'sql':
      case 'ldap':
         $error = sprintf(_("Cannot copy a file or directory on a non-file-based backend type (\"%s\")"), $backend);
         break;



      // ----------------------------------
      //
      case 'ftp':

         include_once(SM_PATH . 'plugins/server_settings_backend/ftp_functions.php');


         // get host, port, passive mode, ASCII/Binary, if should use SSL, etc
         //
         if (!empty($backend_info['PORT']) && is_array($backend_info['PORT']))
            $port = retrieve_server_setting($backend_info['PORT'], $quiet);
         else $port = 21;
         if (!empty($backend_info['PASSIVE']) && is_array($backend_info['PASSIVE']))
            $passive = retrieve_server_setting($backend_info['PASSIVE'], $quiet);
         else $passive = FALSE;
         if (!empty($backend_info['MODE']) && is_array($backend_info['MODE']))
            $mode = strtoupper(retrieve_server_setting($backend_info['MODE'], $quiet));
         else $mode = 'BINARY';
         if (!empty($backend_info['SSL']) && is_array($backend_info['SSL']))
         {
            $use_ssl = retrieve_server_setting($backend_info['SSL'], $quiet);
            if ($use_ssl && !function_exists('ftp_ssl_connect'))
               $error = _("Misconfiguration encountered when attempting to copy a file or directory on the server (SSL-FTP connection not possible when no SSL support is built into PHP)");
         }
         else $use_ssl = FALSE;
         if (!empty($backend_info['HOST']) && is_array($backend_info['HOST']))
            $host = retrieve_server_setting($backend_info['HOST'], $quiet);
         else
         {
            $error = _("Misconfiguration encountered when attempting to copy a file or directory on the server (FTP host is misconfigured)");
         }


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // check that the PHP FTP extension is available
         //
         if (!function_exists('ftp_connect'))  // even if using ssl; this is just a test
            $error = _("Misconfiguration encountered when attempting to copy a file or directory on the server (no FTP support is built into PHP)");


         // validate we have what we need
         //
         if (empty($error)
          && (empty($host)
           || ($mode != 'BINARY' && $mode != 'ASCII')))
            $error = _("Misconfiguration encountered when attempting to copy a file or directory on the server (missing FTP host or mode is unknown)");
         else if (empty($error))
         {

            // first inspect destination
            //
            // if not allowed to overwrite and target already exists, error out
            // if overwrite is allowed, DELETE the target first - CAREFUL!
            // although note that because only empty dirs are delete-able, in
            // many cases where a target dir already exists it will just error
            // out trying to delete if it's not empty
            //
            $ret = TRUE;
            if (($file_info = ftp_path_exists(FALSE, $host, $port, $use_ssl,
                                              $passive, $user, $pass,
                                              $destination_path, TRUE, $quiet)))
            {
               if (!$allow_overwrite)
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("File or directory \"%s\" already exists; cannot overwrite"), $destination_path);
                  else
                     $error = _("File or directory already exists; cannot overwrite");
               }
               else
               {
                  // watch out!
                  // 
                  if (!empty($file_info['type']) && $file_info['type'] == 'directory')
                     $del_dir = TRUE;
                  else
                     $del_dir = FALSE;
                  $ret = ssb_ftp_delete(FALSE, $host, $port, $use_ssl,
                                        $passive, $user, $pass, '',
                                        $destination_path, $del_dir, TRUE, $quiet);
               }
            }


            // grab information about source
            //
            if ($ret)
               $source_file_info = ftp_path_exists(FALSE, $host, $port, $use_ssl,
                                                   $passive, $user, $pass,
                                                   $source_path, TRUE, $quiet);


            // if possible, try to determine $is_directory on our own
            //
            if ($autodetect)
            {
               if (!empty($source_file_info['type']) && $source_file_info['type'] == 'directory')
                  $is_directory = TRUE;
               else if (!empty($source_file_info['type']) && $source_file_info['type'] == 'file')
                  $is_directory = FALSE;

               // don't support copying anything except files/dirs
               //
               else if (!empty($source_file_info['type']))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("\"%s\" is not a file or a directory; cannot copy"), $source_path);
                  else
                     $error = _("Cannot copy entities other than files or directories");
               }
            }


            // copy just a file?
            //
            if ($ret && !$is_directory)
            {
               $local_file = ftp_get_file(FALSE, $host, $port, $use_ssl, $passive,
                                          $mode, $user, $pass, '', $source_path,
                                          FALSE, $quiet);

               // assume that unless $local_file is NULL,
               // the file exists and all is normal,
               // if not, use an empty value
               //
               if (is_null($local_file))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("Error attempting to read \"%s\""), $source_path);
                  else
                     $error = _("Error attempting to read file");
               }
               else
               {
                  if (!($contents = @file($local_file))) $contents = '';
                  else $contents = implode('', $contents);
                  @unlink($local_file);

                  $ret = ftp_put_file(FALSE, $contents, $allow_overwrite,
                                      $host, $port, $use_ssl, $passive,
                                      $mode, $user, $pass, '',
                                      $destination_path, FALSE, $quiet);

                  // attempt to replicate same file permissions
                  //
                  if ($ret
                   && !empty($source_file_info['numeric_permissions']))
                     // intentionally ignore return value in case functionality not supported
                     ssb_ftp_chmod(FALSE, $host, $port, $use_ssl, $passive,
                                   $user, $pass, $destination_path,
                                   $source_file_info['numeric_permissions'], TRUE);
               }
            }


            // copy directory?
            //
            else if ($ret)  // && $is_directory)
            {

               // get list of things to copy
               //
               $ret = ftp_get_directory_listing(FALSE, $host, $port,
                                                $use_ssl, $passive, $user,
                                                $pass, $source_path, $quiet);
               if (is_null($ret))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("Error attempting to obtain directory listing: \"%s\""), $source_path);
                  else
                     $error = _("Error attempting to obtain directory listing");
               }

               // create target directory
               //
               if (!is_null($ret)
                && !ssb_ftp_mkdir(FALSE, $host, $port, $use_ssl, $passive,
                                  $user, $pass, $destination_path, 0700, $quiet))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("Could not create target directory \"%s\""), $destination_path);
                  else
                     $error = _("Could not create target directory");
               }

               // attempt to replicate same directory permissions
               //
               if (!is_null($ret)
                && !empty($source_file_info['numeric_permissions']))
                  // intentionally ignore return value in case functionality not supported
                  ssb_ftp_chmod(FALSE, $host, $port, $use_ssl, $passive,
                                $user, $pass, $destination_path,
                                $source_file_info['numeric_permissions'], TRUE);

               // copy each directory entry one at a time
               //
               if (!is_null($ret)) foreach ($ret as $entry)
               {

                  if (empty($entry['name'])) // assume path_exists otherwise
                     continue;

                  if ($entry['name'] == '.' || $entry['name'] == '..')
                     continue;

                  // copy subdirectories?
                  //
                  if (!empty($entry['type']) && $entry['type'] == 'directory')
                  {
                     if ($recursive)
                     {
//TODO: path separator below may not work under all O/Ses(?)
                        $result = copy_file_or_directory($source_path . '/' . $entry['name'], $destination_path . '/' . $entry['name'], TRUE, TRUE, $allow_overwrite, $backend_info, $quiet);

                        if (!$result)
                        {
                           if ($quiet)
                           {
                              sq_change_text_domain($original_text_domain);
                              return NULL;
                           }
                           $ret = NULL;
                           global $sm_debug_mode, $server_settings_backend_debug;
                           server_settings_backend_init();
                           if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                           if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
//TODO: path separator below may not work under all O/Ses(?)
                              $error = sprintf(_("Error occurred copying subdirectory \"%s\""), $source_path . '/' . $entry['name']);
                           else
                              $error = _("Error occurred copying subdirectory");
                           break;
                        }
                     }
                  }
                  else if (!empty($entry['type']) && $entry['type'] == 'file')
                  {
                     $local_file = ftp_get_file(FALSE, $host, $port, $use_ssl, $passive,
                                                $mode, $user, $pass, $source_path,
                                                $entry['name'], FALSE, $quiet);

                     // assume that unless $local_file is NULL,
                     // the file exists and all is normal,
                     // if not, use an empty value
                     //
                     if (is_null($local_file))
                     {
                        if ($quiet)
                        {
                           sq_change_text_domain($original_text_domain);
                           return NULL;
                        }
                        $ret = NULL;
                        global $sm_debug_mode, $server_settings_backend_debug;
                        server_settings_backend_init();
                        if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                        if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
//TODO: path separator below may not work under all O/Ses(?)
                           $error = sprintf(_("Error attempting to read \"%s\""), $source_path . '/' . $entry['name']);
                        else
                           $error = _("Error attempting to read file");
                        break;
                     }
                     else
                     {
                        if (!($contents = @file($local_file))) $contents = '';
                        else $contents = implode('', $contents);
                        @unlink($local_file);

                        $result = ftp_put_file(FALSE, $contents, $allow_overwrite,
                                               $host, $port, $use_ssl, $passive,
                                               $mode, $user, $pass, $destination_path,
                                               $entry['name'], FALSE, $quiet);

                        // attempt to replicate same file permissions
                        //
                        if ($result
                         && !empty($entry['numeric_permissions']))
                           // intentionally ignore return value in case functionality not supported
                           ssb_ftp_chmod(FALSE, $host, $port, $use_ssl, $passive,
                                         $user, $pass,
//TODO: path separator below may not work under all O/Ses(?)
                                         $destination_path . '/' . $entry['name'],
                                         $entry['numeric_permissions'], TRUE);

                        if (!$result)
                        {
                           if ($quiet)
                           {
                              sq_change_text_domain($original_text_domain);
                              return NULL;
                           }
                           $ret = NULL;
                           global $sm_debug_mode, $server_settings_backend_debug;
                           server_settings_backend_init();
                           if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                           if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
//TODO: path separator below may not work under all O/Ses(?)
                              $error = sprintf(_("Error attempting to write \"%s\""), $destination_path . '/' . $entry['name']);
                           else
                              $error = _("Error attempting to write file");
                           break;
                        }
                     }
                  }

               }

               if (!is_null($ret)) $ret = TRUE; // sometimes $ret still contains file dir listing

            }


            sq_change_text_domain($original_text_domain);
            if (empty($error)) return $ret;

         }
         break;



      // ----------------------------------
      //
      case 'local_file':

         include_once(SM_PATH . 'plugins/server_settings_backend/local_file_functions.php');


         // get access type, suid info if needed
         //
         if (!empty($backend_info['ACCESS_TYPE']) && is_array($backend_info['ACCESS_TYPE']))
            $access_type = retrieve_server_setting($backend_info['ACCESS_TYPE'], $quiet);
         else
            $error = _("Misconfiguration encountered when attempting to copy a file or directory on the server (file access method is misconfigured)");
         if (!empty($backend_info['SUID_LOCATION']) && is_array($backend_info['SUID_LOCATION']))
            $suid_binary = retrieve_server_setting($backend_info['SUID_LOCATION'], $quiet);
         else if ($access_type == 'SUID')
            $error = _("Misconfiguration encountered when attempting to copy a file or directory on the server (SUID_LOCATION is misconfigured)");
         else $suid_binary = '';
         if (!empty($backend_info['SUID_DEBUG_OUTPUT']) && is_array($backend_info['SUID_DEBUG_OUTPUT']))
            $suid_debug_file = retrieve_server_setting($backend_info['SUID_DEBUG_OUTPUT'], $quiet);
         else
            $suid_debug_file = '';
         if (!empty($backend_info['MASTER_SECURITY_CHECK']) && is_array($backend_info['MASTER_SECURITY_CHECK']))
            $master_security_check = retrieve_server_setting($backend_info['MASTER_SECURITY_CHECK'], $quiet);
         else
            $master_security_check = FALSE;
         if (!empty($backend_info['MASTER_USERNAME']) && is_array($backend_info['MASTER_USERNAME']))
            $master_username = retrieve_server_setting($backend_info['MASTER_USERNAME'], $quiet);
         else
            $master_username = NULL;


         // if using "master" security check, grab needed values from environment
         //
         if ($master_security_check)
         {
            if (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_PASSWORD', $master_password, SQ_SERVER)
             || $master_password == 'test-password')
               $error = _("Misconfiguration encountered when attempting to copy a file or directory on the server (master password is misconfigured)");
            if ($master_username
             && (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_USERNAME', $master_username, SQ_SERVER)
              || $master_username == 'test-username'))
               $error = _("Misconfiguration encountered when attempting to copy a file or directory on the server (master username is misconfigured)");
         }
         else
            $master_username = $master_password = NULL;


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // validate we have what we need
         //
         if (empty($error)
          && (empty($access_type)
          || ($access_type == 'SUID' && empty($suid_binary))))
            $error = _("Misconfiguration encountered when attempting to copy a file or directory on the server (missing access type)");
         else if (empty($error))
         {

            $source_path = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $source_path);
            $destination_path = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $destination_path);


            // first inspect destination
            //
            // if not allowed to overwrite and target already exists, error out
            // if overwrite is allowed, DELETE the target first - CAREFUL!
            // although note that because only empty dirs are delete-able, in
            // many cases where a target dir already exists it will just error
            // out trying to delete if it's not empty
            //
            $ret = TRUE;
            if (($file_info = local_file_path_exists($access_type, $user, $pass,
                                                     $destination_path, TRUE,
                                                     $suid_binary, $suid_debug_file,
                                                     $master_username,
                                                     $master_password, $quiet)))
            {
               if (!$allow_overwrite)
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("File or directory \"%s\" already exists; cannot overwrite"), $destination_path);
                  else
                     $error = _("File or directory already exists; cannot overwrite");
               }
               else
               {
                  // watch out!
                  // 
                  if (!empty($file_info['type']) && $file_info['type'] == 'directory')
                     $del_dir = TRUE;
                  else
                     $del_dir = FALSE;
                  $ret = local_file_delete($access_type, $user, $pass,
                                           $destination_path, $del_dir, TRUE,
                                           $suid_binary, $suid_debug_file,
                                           $master_username, $master_password, quiet);
               }
            }


            // grab information about source (note NOT using lstat())
            //
            if ($ret)
               $source_file_info = local_file_path_exists($access_type, $user, $pass,
                                                          $source_path, TRUE,
                                                          $suid_binary, $suid_debug_file,
                                                          $master_username,
                                                          $master_password, $quiet, FALSE);


            // if possible, try to determine $is_directory on our own
            //
            if ($autodetect)
            {
               if (!empty($source_file_info['type']) && $source_file_info['type'] == 'directory')
                  $is_directory = TRUE;
               else if (!empty($source_file_info['type']) && $source_file_info['type'] == 'file')
                  $is_directory = FALSE;

               // don't support copying anything except files/dirs
               //
               else if (!empty($source_file_info['type']))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("\"%s\" is not a file or a directory; cannot copy"), $source_path);
                  else
                     $error = _("Cannot copy entities other than files or directories");
               }
            }


            // copy just a file?
            //
            if ($ret && !$is_directory)
            {
               $local_file = get_local_file($access_type, $user, $pass, $source_path,
                                            FALSE, $suid_binary, $suid_debug_file,
                                            $master_username, $master_password, $quiet);

               // assume that unless $local_file is NULL,
               // the file exists and all is normal,
               // if not, use an empty value
               //
               if (is_null($local_file))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("Error attempting to read \"%s\""), $source_path);
                  else
                     $error = _("Error attempting to read file");
               }
               else
               {
                  if (!($contents = @file($local_file))) $contents = '';
                  else $contents = implode('', $contents);
                  if ($access_type != 'PHP') @unlink($local_file);

                  $ret = put_local_file($access_type, $contents, $allow_overwrite,
                                        $user, $pass, $destination_path,
                                        FALSE, $suid_binary, $suid_debug_file,
                                        $master_username, $master_password, $quiet);

                  // attempt to replicate same file permissions
                  //
                  if ($ret
                   && !empty($source_file_info['numeric_permissions']))
                     $ret = chmod_local_file_or_directory($access_type, $user, $pass,
                                                          $destination_path,
                                                          octdec($source_file_info['numeric_permissions']),
                                                          $suid_binary, $suid_debug_file,
                                                          $master_username, $master_password,
                                                          $quiet);
               }
            }


            // copy directory?
            //
            else if ($ret)  // && $is_directory)
            {

               // get list of things to copy (note NOT using lstat)
               //
               $ret = get_local_directory_listing($access_type, $user, $pass, $source_path,
                                                  $suid_binary, $suid_debug_file,
                                                  $master_username, $master_password,
                                                  $quiet, FALSE);
               if (is_null($ret))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("Error attempting to obtain directory listing: \"%s\""), $source_path);
                  else
                     $error = _("Error attempting to obtain directory listing");
               }

               $directory_listing = $ret;

               // create target directory
               //
               if (!is_null($ret)
                && !local_file_mkdir($access_type, $user, $pass, $destination_path,
                                     0700, $suid_binary, $suid_debug_file,
                                     $master_username, $master_password, $quiet))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("Could not create target directory \"%s\""), $destination_path);
                  else
                     $error = _("Could not create target directory");
               }

               // attempt to replicate same directory permissions
               //
               if (!is_null($ret)
                && !empty($source_file_info['numeric_permissions']))
                  $ret = chmod_local_file_or_directory($access_type, $user, $pass,
                                                       $destination_path,
                                                       octdec($source_file_info['numeric_permissions']),
                                                       $suid_binary, $suid_debug_file,
                                                       $master_username, $master_password,
                                                       $quiet);

               // copy each directory entry one at a time
               //
               if (!is_null($ret)) foreach ($directory_listing as $entry)
               {

                  if (empty($entry['name'])) // assume path_exists otherwise
                     continue;

                  if ($entry['name'] == '.' || $entry['name'] == '..')
                     continue;

                  // copy subdirectories?
                  //
                  if (!empty($entry['type']) && $entry['type'] == 'directory')
                  {
                     if ($recursive)
                     {
//TODO: path separator below may not work under all O/Ses(?)
                        $result = copy_file_or_directory($source_path . '/' . $entry['name'], $destination_path . '/' . $entry['name'], TRUE, TRUE, $allow_overwrite, $backend_info, $autodetect, $quiet);

                        if (!$result)
                        {
                           if ($quiet)
                           {
                              sq_change_text_domain($original_text_domain);
                              return NULL;
                           }
                           $ret = NULL;
                           global $sm_debug_mode, $server_settings_backend_debug;
                           server_settings_backend_init();
                           if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                           if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
//TODO: path separator below may not work under all O/Ses(?)
                              $error = sprintf(_("Error occurred copying subdirectory \"%s\""), $source_path . '/' . $entry['name']);
                           else
                              $error = _("Error occurred copying subdirectory");
                           break;
                        }
                     }
                  }
                  else if (!empty($entry['type']) && $entry['type'] == 'file')
                  {
                     $local_file = get_local_file($access_type, $user, $pass,
//TODO: path separator below may not work under all O/Ses(?)
                                                  $source_path . '/' . $entry['name'],
                                                  FALSE, $suid_binary, $suid_debug_file,
                                                  $master_username, $master_password,
                                                  $quiet);

                     // assume that unless $local_file is NULL,
                     // the file exists and all is normal,
                     // if not, use an empty value
                     //
                     if (is_null($local_file))
                     {
                        if ($quiet)
                        {
                           sq_change_text_domain($original_text_domain);
                           return NULL;
                        }
                        $ret = NULL;
                        global $sm_debug_mode, $server_settings_backend_debug;
                        server_settings_backend_init();
                        if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                        if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
//TODO: path separator below may not work under all O/Ses(?)
                           $error = sprintf(_("Error attempting to read \"%s\""), $source_path . '/' . $entry['name']);
                        else
                           $error = _("Error attempting to read file");
                        break;
                     }
                     else
                     {
                        if (!($contents = @file($local_file))) $contents = '';
                        else $contents = implode('', $contents);
                        if ($access_type != 'PHP') @unlink($local_file);

                        $result = put_local_file($access_type, $contents, $allow_overwrite,
                                                 $user, $pass,
//TODO: path separator below may not work under all O/Ses(?)
                                                 $destination_path . '/' . $entry['name'],
                                                 FALSE, $suid_binary, $suid_debug_file,
                                                 $master_username, $master_password, $quiet);

                        // attempt to replicate same file permissions
                        //
                        if ($result
                         && !empty($entry['numeric_permissions']))
                           $result = chmod_local_file_or_directory($access_type, $user, $pass,
//TODO: path separator below may not work under all O/Ses(?)
                                                                   $destination_path . '/' . $entry['name'],
                                                                   octdec($entry['numeric_permissions']),
                                                                   $suid_binary,
                                                                   $suid_debug_file,
                                                                   $master_username,
                                                                   $master_password,
                                                                   $quiet);

                        if (!$result)
                        {
                           if ($quiet)
                           {
                              sq_change_text_domain($original_text_domain);
                              return NULL;
                           }
                           $ret = NULL;
                           global $sm_debug_mode, $server_settings_backend_debug;
                           server_settings_backend_init();
                           if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                           if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
//TODO: path separator below may not work under all O/Ses(?)
                              $error = sprintf(_("Error attempting to write \"%s\""), $destination_path . '/' . $entry['name']);
                           else
                              $error = _("Error attempting to write file");
                           break;
                        }
                     }
                  }

               }

               if (!is_null($ret)) $ret = TRUE; // sometimes $ret still contains file dir listing

            }


            sq_change_text_domain($original_text_domain);
            if (empty($error)) return $ret;

         }
         break;

   }


   // should not get here without an error message
   // unless something is very wrong
   //
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function copy_file_or_directory()");


   // spit out error and exit
   //
   if ($quiet) return NULL;
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Compresses a directory and its contents (or just a file)
  * on the server into a zip, tar, or gzipped tar archive file.
  *
  * Note that links' targets are included, not the links themselves.
  *
  * This function is only designed for use with backend types
  * that operate on a per-file basis, such as FTP and the
  * local_file backends.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $path             The path to the desired file or
  *                                  directory that will be compressed.
  *                                  This should typically be a FULL
  *                                  (and not relative) directory path.
  * @param string  $archive_type     "ZIP", "TAR" or "TGZ" (for gzipped
  *                                  tar files, which are only available
  *                                  if your installation of PHP has
  *                                  zlib support compiled in (--with-zlib
  *                                  - see http://us.php.net/manual/en/zlib.installation.php))
  * @param boolean $use_path         When TRUE, the directory portion
  *                                  of the path is used when adding
  *                                  files to the archive; otherwise
  *                                  the parent directory is stripped
  *                                  off. (only the *base path* is affected;
  *                                  subdirectories will *always* be created)
  * @param boolean $is_directory     When TRUE, the entry being archived
  *                                  is assumed to be a directory, when
  *                                  FALSE it is assumed to be a file.
  *                                  Some (most) backends may (will) ignore
  *                                  this and test the entry on their own
  *                                  and act accordingly unless $autodetect
  *                                  is FALSE.
  * @param boolean $recursive        When TRUE, any subdirectories and
  *                                  their contents are recursively
  *                                  included in the archive file;
  *                                  otherwise only the immediate
  *                                  contents of the directory (first
  *                                  level) are included - no
  *                                  subdirectories will be included.
  *                                  This has no effect when just a
  *                                  file is being archived.
  * @param array   $backend_info     An array of configuration values
  *                                  that point to how the directory
  *                                  or file is accessed.
  * @param boolean $quiet            When TRUE, suppresses any error
  *                                  messages and makes this function
  *                                  return NULL without doing anything
  *                                  else (OPTIONAL; default = FALSE).
  * @param boolean $autodetect       When FALSE, $is_directory will be used
  *                                  unquestioningly.  When TRUE, the $path
  *                                  will be inspected to try to determine
  *                                  if it is a file, directory, etc.
  *                                  (OPTIONAL; default = TRUE)
  * @param boolean $gnu_extended_tar When $archive_type is "TAR" or "TGZ",
  *                                  this indicates whether or not the GNU
  *                                  extensions to the TAR standard should
  *                                  be used to handle long filenames
  *                                  (TRUE; recommended), or if the USTAR/
  *                                  POSIX extension should be used instead
  *                                  (FALSE; file paths still limited to
  *                                  256 characters and will be truncated
  *                                  after that, possibly resulting in loss
  *                                  of some files and/or path information
  *                                  if duplication arises as a result of
  *                                  truncation) (OPTIONAL; default = TRUE)
  * @param object $archive_file      This is only for use internally,
  *                                  when this function calls itself
  *                                  recursively in order to add
  *                                  subdirectories to the archive file;
  *                                  the original caller of this function
  *                                  should usually not specify anything
  *                                  here.  Note this parameter is passed
  *                                  by reference.  It should be an object
  *                                  of class "zipfile" or class "tar",
  *                                  located in zip.php or tar.class.php
  *                                  respectively.
  * @param int    $depth             Only used for recursive calls to
  *                                  this function in conjunction with
  *                                  working with path information;
  *                                  this should typically not be specified
  *                                  by the original caller.
  *
  * @return mixed A string containing the full file path to the locally
  *               created zip archive file (typically in the SquirrelMail
  *               attachments directory) if it was created normally and
  *               $archive_file was not specified.  If $archive_file was
  *               specified and all operations completed successfully,
  *               just TRUE is returned.  NULL is returned if an error
  *               occurred and $quiet is TRUE.
  *
  */
function create_archive($path, $archive_type, $use_path, $is_directory,
                        $recursive, $backend_info, $quiet=FALSE,
                        $autodetect=TRUE, $gnu_extended_tar=TRUE,
                        &$archive_file=NULL, $depth=0)
{

   $error = '';

   include_once(SM_PATH . 'plugins/server_settings_backend/init.php');
   include_once(SM_PATH . 'plugins/server_settings_backend/functions.php');


   // what kind of archive file?
   //
   $archive_type = strtoupper($archive_type);
   if ($archive_type == 'ZIP')
      include_once(SM_PATH . 'plugins/server_settings_backend/zip.php');
   else if ($archive_type == 'TAR')
      include_once(SM_PATH . 'plugins/server_settings_backend/tar.class.php');
   else if ($archive_type == 'TGZ')
   {
      if (!function_exists('gzopen'))
         $error = _("Misconfiguration encountered when attempting to archive a file or directory on the server (no ZLIB support is built into PHP)");
      else
         include_once(SM_PATH . 'plugins/server_settings_backend/tar.class.php');
   }
   else
      $error = _("Unknown archive type");


   global $username, $domain;
   $original_text_domain = sq_change_text_domain('server_settings_backend');


   if (!is_array($backend_info))
      $error = sprintf(_("Misconfiguration encountered when attempting to archive a file or directory on the server (malformed backend configuration)"), $backend_info['CUSTOM']);


   // for sanity, make sure all array keys are upper case
   //
   else
   {
      $temp_backend_info = $backend_info;
      $backend_info = array();
      foreach ($temp_backend_info as $key => $value)
         $backend_info[strtoupper($key)] = $value;
   }


   // do we have an array of lookups?  we can assume so if
   // there is no VALUE, CUSTOM or BACKEND entry in the array
   //
   // when we have multiple lookups, just use the first one (multiple
   // lookups are only useful for performing multiple save actions)
   //
   if (empty($backend_info['VALUE']) && empty($backend_info['CUSTOM'])
    && empty($backend_info['BACKEND']))
      return create_archive($path, $archive_type, $use_path, $is_directory,
                            $recursive, array_shift($backend_info), $quiet,
                            $autodetect, $gnu_extended_tar, $archive_file, $depth);


   // start by getting the backend type
   //
   $backend = strtolower($backend_info['BACKEND']);
   if (empty($error) && empty($backend)
    || ($backend != 'ftp' && $backend != 'local_file' && $backend != 'sql' && $backend != 'ldap'))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to archive a file or directory on the server (\"%s\" lookup unknown)"), $backend);
   }


   // do some setup:
   //
   // remove trailing slashes from path, see how many
   // parent directories are in path and set depth
   // accordingly if first time in, determine if
   // archive file should be physically created or
   // not, etc.
   //
   while (strlen($path) > 1
    && ($path[strlen($path) - 1] == '/'
     || $path[strlen($path) - 1] == '\\'))
      $path = substr($path, 0, -1);
   $path_elements = preg_split('/[\/\\\]/', $path);
   if (is_null($archive_file))
   {
      $return_success_as_boolean = FALSE;

      if ($archive_type == 'ZIP')
         $archive_file = new zipfile();
      else // TAR or TGZ (unknown type already caught at top of fxn)
         $archive_file = new tar();

      // when we are at the beginning, we need to
      // set depth to the number of original path elements
      //
      $depth = sizeof($path_elements);
   }
   else
      $return_success_as_boolean = TRUE;

   // strip off $depth elements off front of path
   // to create trimmed "short path"
   //
   for ($i = 0; $i < $depth; $i++)
      array_shift($path_elements);
//TODO: path separator below may not work under all O/Ses(?)
   $short_path = implode('/', $path_elements);


   // what kind of backend will we be using to retrieve the value?
   //
   if (empty($error)) switch ($backend)
   {

      // ----------------------------------
      //
      case 'sql':
      case 'ldap':
         $error = sprintf(_("Cannot archive a file or directory on a non-file-based backend type (\"%s\")"), $backend);
         break;



      // ----------------------------------
      //
      case 'ftp':

         include_once(SM_PATH . 'plugins/server_settings_backend/ftp_functions.php');


         // get host, port, passive mode, ASCII/Binary, if should use SSL, etc
         //
         if (!empty($backend_info['PORT']) && is_array($backend_info['PORT']))
            $port = retrieve_server_setting($backend_info['PORT'], $quiet);
         else $port = 21;
         if (!empty($backend_info['PASSIVE']) && is_array($backend_info['PASSIVE']))
            $passive = retrieve_server_setting($backend_info['PASSIVE'], $quiet);
         else $passive = FALSE;
         if (!empty($backend_info['MODE']) && is_array($backend_info['MODE']))
            $mode = strtoupper(retrieve_server_setting($backend_info['MODE'], $quiet));
         else $mode = 'BINARY';
         if (!empty($backend_info['SSL']) && is_array($backend_info['SSL']))
         {
            $use_ssl = retrieve_server_setting($backend_info['SSL'], $quiet);
            if ($use_ssl && !function_exists('ftp_ssl_connect'))
               $error = _("Misconfiguration encountered when attempting to archive a file or directory on the server (SSL-FTP connection not possible when no SSL support is built into PHP)");
         }
         else $use_ssl = FALSE;
         if (!empty($backend_info['HOST']) && is_array($backend_info['HOST']))
            $host = retrieve_server_setting($backend_info['HOST'], $quiet);
         else
         {
            $error = _("Misconfiguration encountered when attempting to archive a file or directory on the server (FTP host is misconfigured)");
         }


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // check that the PHP FTP extension is available
         //
         if (!function_exists('ftp_connect'))  // even if using ssl; this is just a test
            $error = _("Misconfiguration encountered when attempting to archive a file or directory on the server (no FTP support is built into PHP)");


         // validate we have what we need
         //
         if (empty($error)
          && (empty($host)
           || ($mode != 'BINARY' && $mode != 'ASCII')))
            $error = _("Misconfiguration encountered when attempting to archive a file or directory on the server (missing FTP host or mode is unknown)");
         else if (empty($error))
         {

            $ret = TRUE;


            // grab information about source
            //
            $file_info = ftp_path_exists(FALSE, $host, $port, $use_ssl,
                                         $passive, $user, $pass,
                                         $path, TRUE, $quiet);

            // if possible, try to determine $is_directory on our own
            //
            if ($autodetect)
            {
               if (!empty($file_info['type']) && $file_info['type'] == 'directory')
                  $is_directory = TRUE;
               else if (!empty($file_info['type']) && $file_info['type'] == 'file')
                  $is_directory = FALSE;

               // don't support archiving anything except files/dirs
               //
               else if (!empty($file_info['type']))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("\"%s\" is not a file or a directory; cannot archive"), $path);
                  else
                     $error = _("Cannot archive entities other than files or directories");
               }
            }


            // archive just a file?
            //
            if ($ret && !$is_directory)
            {
               $local_file = ftp_get_file(FALSE, $host, $port, $use_ssl, $passive,
                                          $mode, $user, $pass, '', $path,
                                          FALSE, $quiet);

               // assume that unless $local_file is NULL,
               // the file exists and all is normal,
               // if not, use an empty value
               //
               if (is_null($local_file))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("Error attempting to read \"%s\""), $path);
                  else
                     $error = _("Error attempting to read file");
               }
               else
               {
                  if (!($contents = @file($local_file))) $contents = '';
                  else $contents = implode('', $contents);
                  @unlink($local_file);

                  if ($use_path) $file_path = $path;
                  else $file_path = basename($path);
                  if (!empty($file_info['date_modified']))
                     $file_time = $file_info['date_modified'];
                  else
                     $file_time = time();

                  if ($archive_type == 'ZIP')
                     $archive_file->addFile($contents, $file_path, $file_time);
                  else // TAR or TGZ (unknown type already caught at top of fxn)
                  {
                     if (!empty($file_info['numeric_permissions']))
                        $file_mode = $file_info['numeric_permissions'];
                     else
                        $file_mode = 0440;  //TODO: is there a better default?

                     if (!empty($file_info['uid']))
                        $file_uid = $file_info['uid'];
                     else
                        $file_uid = 0;

                     if (!empty($file_info['gid']))
                        $file_gid = $file_info['gid'];
                     else
                        $file_gid = 0;

                     if (!empty($file_info['size']))
                        $file_size = $file_info['size'];
                     else
                        $file_size = strlen($contents); // assuming no mbstring overload

                     $archive_file->addVirtualFile($contents, $file_path, $file_time, $file_mode, $file_uid, $file_gid, $file_size);
                  }
               }
            }


            // copy directory?
            //
            else if ($ret)  // && $is_directory)
            {

               // get list of directory contents
               //
               $ret = ftp_get_directory_listing(FALSE, $host, $port,
                                                $use_ssl, $passive, $user,
                                                $pass, $path, $quiet);

               // archive each directory entry one at a time
               //
               if (!is_null($ret)) foreach ($ret as $entry)
               {

                  if (empty($entry['name'])) // assume path_exists otherwise
                     continue;

                  if ($entry['name'] == '.' || $entry['name'] == '..')
                     continue;

                  // copy subdirectories?
                  //
                  if (!empty($entry['type']) && $entry['type'] == 'directory')
                  {
                     if ($recursive)
                     {
//TODO: path separator below may not work under all O/Ses(?)
                        $result = create_archive($path . '/' . $entry['name'], $archive_type, $use_path, TRUE, TRUE, $backend_info, $quiet, $autodetect, $gnu_extended_tar, $archive_file, $depth);

                        if (!$result)
                        {
                           if ($quiet)
                           {
                              sq_change_text_domain($original_text_domain);
                              return NULL;
                           }
                           $ret = NULL;
                           global $sm_debug_mode, $server_settings_backend_debug;
                           server_settings_backend_init();
                           if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                           if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
//TODO: path separator below may not work under all O/Ses(?)
                              $error = sprintf(_("Error occurred archiving subdirectory \"%s\""), $path . '/' . $entry['name']);
                           else
                              $error = _("Error occurred archiving subdirectory");
                           break;
                        }
                     }
                  }
                  else if (!empty($entry['type']) && $entry['type'] == 'file')
                  {
                     $local_file = ftp_get_file(FALSE, $host, $port, $use_ssl, $passive,
                                                $mode, $user, $pass, $path,
                                                $entry['name'], FALSE, $quiet);

                     // assume that unless $local_file is NULL,
                     // the file exists and all is normal,
                     // if not, use an empty value
                     //
                     if (is_null($local_file))
                     {
                        if ($quiet)
                        {
                           sq_change_text_domain($original_text_domain);
                           return NULL;
                        }
                        $ret = NULL;
                        global $sm_debug_mode, $server_settings_backend_debug;
                        server_settings_backend_init();
                        if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                        if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
//TODO: path separator below may not work under all O/Ses(?)
                           $error = sprintf(_("Error attempting to read \"%s\""), $path . '/' . $entry['name']);
                        else
                           $error = _("Error attempting to read file");
                        break;
                     }
                     else
                     {
                        if (!($contents = @file($local_file))) $contents = '';
                        else $contents = implode('', $contents);
                        @unlink($local_file);

                        if ($use_path)
                           $file_path = $path;
                        else
                           $file_path = $short_path;

                        if (!empty($entry['date_modified']))
                           $file_time = $entry['date_modified'];
                        else
                           $file_time = time();

                        if ($archive_type == 'ZIP')
//TODO: path separator below may not work under all O/Ses(?)
                           $archive_file->addFile($contents, $file_path
                                                . (empty($file_path) ? '' : '/')
                                                . $entry['name'], $file_time);
                        else // TAR or TGZ (unknown type already caught at top of fxn)
                        {
                           if (!empty($entry['numeric_permissions']))
                              $file_mode = $entry['numeric_permissions'];
                           else
                              $file_mode = 0440;  //TODO: is there a better default?

                           if (!empty($entry['uid']))
                              $file_uid = $entry['uid'];
                           else
                              $file_uid = 0;

                           if (!empty($entry['gid']))
                              $file_gid = $entry['gid'];
                           else
                              $file_gid = 0;

                           if (!empty($entry['size']))
                              $file_size = $entry['size'];
                           else
                              $file_size = strlen($contents); // assuming no mbstring overload

                           $archive_file->addVirtualFile($contents, $file_path
                                                . (empty($file_path) ? '' : '/')
                                                . $entry['name'], $file_time,
                                           $file_mode, $file_uid, $file_gid, $file_size);
                        }
                     }
                  }

               }
               else if (is_null($ret))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("Error attempting to obtain directory listing: \"%s\""), $path);
                  else
                     $error = _("Error attempting to obtain directory listing");
               }

               if (!is_null($ret)) $ret = TRUE; // sometimes $ret still contains file dir listing

            }


            if ($ret && empty($error))
            {
 
               // we were just adding files to a given archive,
               // so just return TRUE
               //
               if ($return_success_as_boolean)
               {
                  sq_change_text_domain($original_text_domain);
                  return TRUE;
               }


               // what extension do we use?
               //
               if ($archive_type == 'ZIP')
                  $extension = 'zip';
               else if ($archive_type == 'TAR')
                  $extension = 'tar';
               else // TGZ (unknown type already caught at top of fxn)
                  $extension = 'tar.gz';


               // otherwise, we're ready to write the archive file
               // to a local file 
               //
               $temp = ssb_get_temp_file($quiet, basename($path), $extension);
               if (is_null($temp)) // implied $quiet is TRUE
               {
                  $error = _("Error obtaining temporary archive file");
                  $ret = NULL;
               }
               else
                  list($archive_file_path, $ARCHIVE) = $temp;

               if ($ret)
               {
                  if ($archive_type == 'ZIP')
                  {
                     if (!sq_call_function_suppress_errors('fwrite', array($ARCHIVE, $archive_file->file())))
                     {
                        $error = _("Error writing to archive file");
                        $ret = NULL;
                        @fclose($ARCHIVE);
                        @unlink($archive_file_path);
                     }
                     else
                        @fclose($ARCHIVE);
                  }

                  else // TAR or TGZ (unknown type already caught at top of fxn)
                  {
                     @fclose($ARCHIVE);
                     if (!$archive_file->toTar($archive_file_path,
                                               $archive_type == 'TAR' ? FALSE : TRUE,
                                               $gnu_extended_tar))
                     {
                        $error = _("Error writing to archive file");
                        $ret = NULL;
                        @unlink($archive_file_path);
                     }
                  }
               }

               if (empty($error) && $ret)
               {
                  sq_change_text_domain($original_text_domain);
                  return $archive_file_path;
               }
            }

         }
         break;



      // ----------------------------------
      //
      case 'local_file':
                           
         include_once(SM_PATH . 'plugins/server_settings_backend/local_file_functions.php');
                        
                           
         // get access type, suid info if needed
         //
         if (!empty($backend_info['ACCESS_TYPE']) && is_array($backend_info['ACCESS_TYPE']))
            $access_type = retrieve_server_setting($backend_info['ACCESS_TYPE'], $quiet);
         else
            $error = _("Misconfiguration encountered when attempting to archive a file or directory on the server (file access method is misconfigured)");
         if (!empty($backend_info['SUID_LOCATION']) && is_array($backend_info['SUID_LOCATION']))
            $suid_binary = retrieve_server_setting($backend_info['SUID_LOCATION'], $quiet);
         else if ($access_type == 'SUID')
            $error = _("Misconfiguration encountered when attempting to archive a file or directory on the server (SUID_LOCATION is misconfigured)");
         else $suid_binary = '';
         if (!empty($backend_info['SUID_DEBUG_OUTPUT']) && is_array($backend_info['SUID_DEBUG_OUTPUT']))
            $suid_debug_file = retrieve_server_setting($backend_info['SUID_DEBUG_OUTPUT'], $quiet);
         else
            $suid_debug_file = '';
         if (!empty($backend_info['MASTER_SECURITY_CHECK']) && is_array($backend_info['MASTER_SECURITY_CHECK']))
            $master_security_check = retrieve_server_setting($backend_info['MASTER_SECURITY_CHECK'], $quiet);
         else
            $master_security_check = FALSE;
         if (!empty($backend_info['MASTER_USERNAME']) && is_array($backend_info['MASTER_USERNAME']))
            $master_username = retrieve_server_setting($backend_info['MASTER_USERNAME'], $quiet);
         else
            $master_username = NULL;


         // if using "master" security check, grab needed values from environment
         //
         if ($master_security_check)
         {
            if (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_PASSWORD', $master_password, SQ_SERVER)
             || $master_password == 'test-password')
               $error = _("Misconfiguration encountered when attempting to archive a file or directory on the server (master password is misconfigured)");
            if ($master_username
             && (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_USERNAME', $master_username, SQ_SERVER)
              || $master_username == 'test-username'))
               $error = _("Misconfiguration encountered when attempting to archive a file or directory on the server (master username is misconfigured)");
         }
         else
            $master_username = $master_password = NULL;


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // validate we have what we need
         //
         if (empty($error)
          && (empty($access_type)
          || ($access_type == 'SUID' && empty($suid_binary))))
            $error = _("Misconfiguration encountered when attempting to archive a file or directory on the server (missing access type)");
         else if (empty($error))
         {

            $ret = TRUE;
            $path = str_replace(array('%1', '%2'), array(ssb_str_esc($user), ssb_str_esc($dom)), $path);


            // grab information about source (note NOT using lstat())
            //
            $file_info = local_file_path_exists($access_type, $user, $pass,
                                                $path, TRUE, $suid_binary,
                                                $suid_debug_file, $master_username,
                                                $master_password, $quiet, FALSE);

            // if possible, try to determine $is_directory on our own
            //
            if ($autodetect)
            {
               if (!empty($file_info['type']) && $file_info['type'] == 'directory')
                  $is_directory = TRUE;
               else if (!empty($file_info['type']) && $file_info['type'] == 'file')
                  $is_directory = FALSE;

               // don't support archiving anything except files/dirs
               //
               else if (!empty($file_info['type']))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("\"%s\" is not a file or a directory; cannot archive"), $path);
                  else
                     $error = _("Cannot archive entities other than files or directories");
               }
            }


            // archive just a file?
            //
            if ($ret && !$is_directory)
            {
               $local_file = get_local_file($access_type, $user, $pass, $path,
                                            FALSE, $suid_binary, $suid_debug_file,
                                            $master_username, $master_password, $quiet);

               // assume that unless $local_file is NULL,
               // the file exists and all is normal,
               // if not, use an empty value
               //
               if (is_null($local_file))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("Error attempting to read \"%s\""), $path);
                  else
                     $error = _("Error attempting to read file");
               }
               else
               {
                  if (!($contents = @file($local_file))) $contents = '';
                  else $contents = implode('', $contents);
                  if ($access_type != 'PHP') @unlink($local_file);

                  if ($use_path) $file_path = $path;
                  else $file_path = basename($path);
                  if (!empty($file_info['date_modified']))
                     $file_time = $file_info['date_modified'];
                  else
                     $file_time = time();

                  if ($archive_type == 'ZIP')
                     $archive_file->addFile($contents, $file_path, $file_time);
                  else // TAR or TGZ (unknown type already caught at top of fxn)
                  {
                     if (!empty($file_info['raw_mode']))
                        $file_mode = $file_info['raw_mode'];
                     else if (!empty($file_info['numeric_permissions']))
                        $file_mode = octdec($file_info['numeric_permissions']);
                     else
                        $file_mode = 0440;  //TODO: is there a better default?

                     if (!empty($file_info['uid']))
                        $file_uid = $file_info['uid'];
                     else
                        $file_uid = 0;

                     if (!empty($file_info['gid']))
                        $file_gid = $file_info['gid'];
                     else
                        $file_gid = 0;

                     if (!empty($file_info['owner']))
                        $file_owner = $file_info['owner'];
                     else
                        $file_owner = '';

                     if (!empty($file_info['group']))
                        $file_group = $file_info['group'];
                     else
                        $file_group = '';

                     if (!empty($file_info['size']))
                        $file_size = $file_info['size'];
                     else
                        $file_size = strlen($contents); // assuming no mbstring overload

                     $archive_file->addVirtualFile($contents, $file_path, $file_time, $file_mode, $file_uid, $file_gid, $file_size, $file_owner, $file_group);
                  }
               }
            }


            // copy directory?
            //
            else if ($ret)  // && $is_directory)
            {

               // get list of directory contents (note NOT using lstat)
               //
               $ret = get_local_directory_listing($access_type, $user, $pass, $path,
                                                  $suid_binary, $suid_debug_file,
                                                  $master_username, $master_password,
                                                  $quiet, FALSE);

               // archive each directory entry one at a time
               //
               if (!is_null($ret)) foreach ($ret as $entry)
               {

                  if (empty($entry['name'])) // assume path_exists otherwise
                     continue;

                  if ($entry['name'] == '.' || $entry['name'] == '..')
                     continue;

                  // copy subdirectories?
                  //
                  if (!empty($entry['type']) && $entry['type'] == 'directory')
                  {
                     if ($recursive)
                     {
//TODO: path separator below may not work under all O/Ses(?)
                        $result = create_archive($path . '/' . $entry['name'], $archive_type, $use_path, TRUE, TRUE, $backend_info, $quiet, $autodetect, $gnu_extended_tar, $archive_file, $depth);

                        if (!$result)
                        {
                           if ($quiet)
                           {
                              sq_change_text_domain($original_text_domain);
                              return NULL;
                           }
                           $ret = NULL;
                           global $sm_debug_mode, $server_settings_backend_debug;
                           server_settings_backend_init();
                           if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                           if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
//TODO: path separator below may not work under all O/Ses(?)
                              $error = sprintf(_("Error occurred archiving subdirectory \"%s\""), $path . '/' . $entry['name']);
                           else
                              $error = _("Error occurred archiving subdirectory");
                           break;
                        }
                     }
                  }
                  else if (!empty($entry['type']) && $entry['type'] == 'file')
                  {
                     $local_file = get_local_file($access_type, $user, $pass,
//TODO: path separator below may not work under all O/Ses(?)
                                                  $path . '/' . $entry['name'],
                                                  FALSE, $suid_binary, $suid_debug_file,
                                                  $master_username, $master_password,
                                                  $quiet);

                     // assume that unless $local_file is NULL,
                     // the file exists and all is normal,
                     // if not, use an empty value
                     //
                     if (is_null($local_file))
                     {
                        if ($quiet)
                        {
                           sq_change_text_domain($original_text_domain);
                           return NULL;
                        }
                        $ret = NULL;
                        global $sm_debug_mode, $server_settings_backend_debug;
                        server_settings_backend_init();
                        if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                        if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
//TODO: path separator below may not work under all O/Ses(?)
                           $error = sprintf(_("Error attempting to read \"%s\""), $path . '/' . $entry['name']);
                        else
                           $error = _("Error attempting to read file");
                        break;
                     }
                     else
                     {
                        if (!($contents = @file($local_file))) $contents = '';
                        else $contents = implode('', $contents);
                        if ($access_type != 'PHP') @unlink($local_file);

                        if ($use_path)
                           $file_path = $path;
                        else
                           $file_path = $short_path;

                        if (!empty($entry['date_modified']))
                           $file_time = $entry['date_modified'];
                        else
                           $file_time = time();

                        if ($archive_type == 'ZIP')
//TODO: path separator below may not work under all O/Ses(?)
                           $archive_file->addFile($contents, $file_path
                                                . (empty($file_path) ? '' : '/')
                                                . $entry['name'], $file_time);
                        else // TAR or TGZ (unknown type already caught at top of fxn)
                        {
                           if (!empty($entry['raw_mode']))
                              $file_mode = $entry['raw_mode'];
                           else if (!empty($entry['numeric_permissions']))
                              $file_mode = octdec($entry['numeric_permissions']);
                           else
                              $file_mode = 0440;  //TODO: is there a better default?

                           if (!empty($entry['uid']))
                              $file_uid = $entry['uid'];
                           else
                              $file_uid = 0;

                           if (!empty($entry['gid']))
                              $file_gid = $entry['gid'];
                           else
                              $file_gid = 0;

                           if (!empty($entry['owner']))
                              $file_owner = $entry['owner'];
                           else
                              $file_owner = '';

                           if (!empty($entry['group']))
                              $file_group = $entry['group'];
                           else
                              $file_group = '';

                           if (!empty($entry['size']))
                              $file_size = $entry['size'];
                           else
                              $file_size = strlen($contents); // assuming no mbstring overload

                           $archive_file->addVirtualFile($contents, $file_path
                                                . (empty($file_path) ? '' : '/')
                                                . $entry['name'], $file_time,
                                           $file_mode, $file_uid, $file_gid,
                                           $file_size, $file_owner, $file_group);
                        }
                     }
                  }

               }
               else if (is_null($ret))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("Error attempting to obtain directory listing: \"%s\""), $path);
                  else
                     $error = _("Error attempting to obtain directory listing");
               }

               if (!is_null($ret)) $ret = TRUE; // sometimes $ret still contains file dir listing

            }


            if ($ret && empty($error))
            {
 
               // we were just adding files to a given archive,
               // so just return TRUE
               //
               if ($return_success_as_boolean)
               {
                  sq_change_text_domain($original_text_domain);
                  return TRUE;
               }


               // what extension do we use?
               //
               if ($archive_type == 'ZIP')
                  $extension = 'zip';
               else if ($archive_type == 'TAR')
                  $extension = 'tar';
               else // TGZ (unknown type already caught at top of fxn)
                  $extension = 'tar.gz';


               // otherwise, we're ready to write the archive file
               // to a local file 
               //
               $temp = ssb_get_temp_file($quiet, basename($path), $extension);
               if (is_null($temp)) // implied $quiet is TRUE
               {
                  $error = _("Error obtaining temporary archive file");
                  $ret = NULL;
               }
               else
                  list($archive_file_path, $ARCHIVE) = $temp;

               if ($ret)
               {
                  if ($archive_type == 'ZIP')
                  {
                     if (!sq_call_function_suppress_errors('fwrite', array($ARCHIVE, $archive_file->file())))
                     {
                        $error = _("Error writing to archive file");
                        $ret = NULL;
                        @fclose($ARCHIVE);
                        @unlink($archive_file_path);
                     }
                     else
                        @fclose($ARCHIVE);
                  }

                  else // TAR or TGZ (unknown type already caught at top of fxn)
                  {
                     @fclose($ARCHIVE);
                     if (!$archive_file->toTar($archive_file_path,
                                               $archive_type == 'TAR' ? FALSE : TRUE,
                                               $gnu_extended_tar))
                     {
                        $error = _("Error writing to archive file");
                        $ret = NULL;
                        @unlink($archive_file_path);
                     }
                  }
               }

               if (empty($error) && $ret)
               {
                  sq_change_text_domain($original_text_domain);
                  return $archive_file_path;
               }
            }

         }
         break;

   }


   // should not get here without an error message
   // unless something is very wrong
   //
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function create_archive()");


   // spit out error and exit
   //
   if ($quiet) return NULL;
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Calculates the approximate size of the given file or
  * directory on the server.
  *
  * If any links are found in the location being calculated,
  * the link targets are used to make the final calculation.
  *
  * This function is only designed for use with backend types
  * that operate on a per-file basis, such as FTP and the
  * local_file backends.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param string  $path             The path to the desired file or
  *                                  directory that will be calculated.
  *                                  This should typically be a FULL
  *                                  (and not relative) directory path.
  * @param boolean $is_directory     When TRUE, the entry being calculated
  *                                  is assumed to be a directory, when
  *                                  FALSE it is assumed to be a file.
  *                                  Some (most) backends may (will) ignore
  *                                  this and test the entry on their own
  *                                  and act accordingly unless $autodetect
  *                                  is FALSE.
  * @param boolean $recursive        When TRUE, any subdirectories and
  *                                  their contents are recursively
  *                                  included in the size calculation;
  *                                  otherwise only the immediate
  *                                  contents of the directory (first
  *                                  level) are included - no
  *                                  subdirectories will be included.
  *                                  This has no effect when just a
  *                                  file is being calculated.
  * @param array   $backend_info     An array of configuration values
  *                                  that point to how the directory
  *                                  or file is accessed.
  * @param boolean $autodetect       When FALSE, $is_directory will be used
  *                                  unquestioningly.  When TRUE, the $path
  *                                  will be inspected to try to determine
  *                                  if it is a file, directory, etc.
  *                                  (OPTIONAL; default = TRUE)
  * @param boolean $quiet            When TRUE, suppresses any error
  *                                  messages and makes this function
  *                                  return NULL without doing anything
  *                                  else (OPTIONAL; default = FALSE).
  *
  * @return mixed An integer representing the total size in bytes or
  *               NULL if an error occurred and $quiet is TRUE.
  *
  */
function calculate_size($path, $is_directory, $recursive, $backend_info,
                        $autodetect=TRUE, $quiet=FALSE)
{

   include_once(SM_PATH . 'plugins/server_settings_backend/init.php');
   include_once(SM_PATH . 'plugins/server_settings_backend/functions.php');

   global $username, $domain;
   $original_text_domain = sq_change_text_domain('server_settings_backend');
   $error = '';


   if (!is_array($backend_info))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to calculate the size of a file or directory on the server (malformed backend configuration)"), $backend_info['CUSTOM']);
   }


   // for sanity, make sure all array keys are upper case
   //
   else
   {
      $temp_backend_info = $backend_info;
      $backend_info = array();
      foreach ($temp_backend_info as $key => $value)
         $backend_info[strtoupper($key)] = $value;
   }


   // do we have an array of lookups?  we can assume so if
   // there is no VALUE, CUSTOM or BACKEND entry in the array
   //
   // when we have multiple lookups, just use the first one (multiple
   // lookups are only useful for performing multiple save actions)
   //
   if (empty($backend_info['VALUE']) && empty($backend_info['CUSTOM'])
    && empty($backend_info['BACKEND']))
      return calculate_size($path, $is_directory, $recursive,
                            array_shift($backend_info), $autodetect, $quiet);


   // start by getting the backend type
   //
   $backend = strtolower($backend_info['BACKEND']);
   if (empty($error) && empty($backend)
    || ($backend != 'ftp' && $backend != 'local_file' && $backend != 'sql' && $backend != 'ldap'))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to calculate the size of a file or directory on the server (\"%s\" lookup unknown)"), $backend);
   }


   // what kind of backend will we be using to retrieve the value?
   //
   if (empty($error)) switch ($backend)
   {

      // ----------------------------------
      //
      case 'sql':
      case 'ldap':
         $error = sprintf(_("Cannot calculate the size of a file or directory on a non-file-based backend type (\"%s\")"), $backend);
         break;



      // ----------------------------------
      //
      case 'ftp':

         include_once(SM_PATH . 'plugins/server_settings_backend/ftp_functions.php');


         // get host, port, passive mode, if should use SSL, etc
         //
         if (!empty($backend_info['PORT']) && is_array($backend_info['PORT']))
            $port = retrieve_server_setting($backend_info['PORT'], $quiet);
         else $port = 21;
         if (!empty($backend_info['PASSIVE']) && is_array($backend_info['PASSIVE']))
            $passive = retrieve_server_setting($backend_info['PASSIVE'], $quiet);
         else $passive = FALSE;
         if (!empty($backend_info['MODE']) && is_array($backend_info['MODE']))
            $mode = strtoupper(retrieve_server_setting($backend_info['MODE'], $quiet));
         else $mode = 'BINARY';
         if (!empty($backend_info['SSL']) && is_array($backend_info['SSL']))
         {
            $use_ssl = retrieve_server_setting($backend_info['SSL'], $quiet);
            if ($use_ssl && !function_exists('ftp_ssl_connect'))
               $error = _("Misconfiguration encountered when attempting to calculate the size of a file or directory on the server (SSL-FTP connection not possible when no SSL support is built into PHP)");
         }
         else $use_ssl = FALSE;
         if (!empty($backend_info['HOST']) && is_array($backend_info['HOST']))
            $host = retrieve_server_setting($backend_info['HOST'], $quiet);
         else
         {
            $error = _("Misconfiguration encountered when attempting to calculate the size of a file or directory on the server (FTP host is misconfigured)");
         }


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // check that the PHP FTP extension is available
         //
         if (!function_exists('ftp_connect'))  // even if using ssl; this is just a test
            $error = _("Misconfiguration encountered when attempting to calculate the size of a file or directory on the server (no FTP support is built into PHP)");


         // validate we have what we need
         //
         if (empty($error)
          && (empty($host)
           || ($mode != 'BINARY' && $mode != 'ASCII')))
            $error = _("Misconfiguration encountered when attempting to calculate the size of a file or directory on the server (missing FTP host or mode is unknown)");
         else if (empty($error))
         {

            // grab information about source
            //
            $ret = 0;
            $file_info = ftp_path_exists(FALSE, $host, $port, $use_ssl,
                                         $passive, $user, $pass,
                                         $path, TRUE, $quiet);

            // if possible, try to determine $is_directory on our own
            //
            if ($autodetect)
            {
               if (!empty($file_info['type']) && $file_info['type'] == 'directory')
                  $is_directory = TRUE;
               else if (!empty($file_info['type']) && $file_info['type'] == 'file')
                  $is_directory = FALSE;
               else if (!empty($file_info['type']))
                  $is_directory = FALSE;  // treat other types as regular file
            }


            // just a file?
            //
            if (!is_null($ret) && !$is_directory)
            {
               if (isset($file_info['size']))
                  $ret = $file_info['size'];

               else
               {
                  $local_file = ftp_get_file(FALSE, $host, $port, $use_ssl, $passive,
                                             $mode, $user, $pass, '', $path,
                                             FALSE, $quiet);

                  // assume that unless $local_file is NULL,
                  // the file exists and all is normal,
                  // if not, use an empty value
                  //
                  if (is_null($local_file))
                  {
                     if ($quiet)
                     {
                        sq_change_text_domain($original_text_domain);
                        return NULL;
                     }
                     $ret = NULL;
                     global $sm_debug_mode, $server_settings_backend_debug;
                     server_settings_backend_init();
                     if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                     if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                        $error = sprintf(_("Error attempting to read \"%s\""), $path);
                     else
                        $error = _("Error attempting to read file");
                  }
                  else
                  {
                     if (!($ret = @filesize($local_file)))
                        $ret = 0;
                     @unlink($local_file);
                  }
               }
            }


            // calculate directory?
            //
            else if (!is_null($ret))  // && $is_directory)
            {

               // get list of directory contents
               //
               $listing = ftp_get_directory_listing(FALSE, $host, $port,
                                                    $use_ssl, $passive, $user,
                                                    $pass, $path, $quiet);

               // add up each directory entry one at a time
               //
               if (!is_null($listing)) foreach ($listing as $entry)
               {

                  if ($entry['name'] == '.' || $entry['name'] == '..')
                     continue;

                  // include subdirectories?
                  //
                  if (!empty($entry['type']) && $entry['type'] == 'directory'
                   && !empty($entry['name']))
                  {
                     if ($recursive)
                     {
//TODO: path separator below may not work under all O/Ses(?)
                        $size = calculate_size($path . '/' . $entry['name'], TRUE, TRUE, $backend_info, $quiet);

                        if (is_null($size))
                        {
                           if ($quiet)
                           {
                              sq_change_text_domain($original_text_domain);
                              return NULL;
                           }
                           $ret = NULL;
                           global $sm_debug_mode, $server_settings_backend_debug;
                           server_settings_backend_init();
                           if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                           if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
//TODO: path separator below may not work under all O/Ses(?)
                              $error = sprintf(_("Error occurred calculating size of subdirectory \"%s\""), $path . '/' . $entry['name']);
                           else
                              $error = _("Error occurred calculating size of subdirectory");
                           break;
                        }
                        else $ret += $size;
                     }
                  }
                  else if (!empty($entry['type']) && $entry['type'] == 'file')
                  {
                     if (isset($entry['size']))
                        $ret += $entry['size'];

                     else if (!empty($entry['name']))
                     {
                        $local_file = ftp_get_file(FALSE, $host, $port, $use_ssl,
                                                   $passive, $mode, $user, $pass,
                                                   $path, $entry['name'], FALSE,
                                                   $quiet);

                        // assume that unless $local_file is NULL,
                        // the file exists and all is normal,
                        // if not, use an empty value
                        //
                        if (is_null($local_file))
                        {
                           if ($quiet)
                           {
                              sq_change_text_domain($original_text_domain);
                              return NULL;
                           }
                           $ret = NULL;
                           global $sm_debug_mode, $server_settings_backend_debug;
                           server_settings_backend_init();
                           if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                           if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
//TODO: path separator below may not work under all O/Ses(?)
                              $error = sprintf(_("Error attempting to read \"%s\""), $path . '/' . $entry['name']);
                           else
                              $error = _("Error attempting to read file");
                           break;
                        }
                        else
                        {
                           if (!($size = @filesize($local_file)))
                              $size = 0;
                           $ret += $size;
                           @unlink($local_file);
                        }
                     }
                  }

               }
               else if (is_null($listing))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("Error attempting to obtain directory listing: \"%s\""), $path);
                  else
                     $error = _("Error attempting to obtain directory listing");
               }

            }


            if (!is_null($ret))
            {
               sq_change_text_domain($original_text_domain);
               return $ret;
            }

         }
         break;



      // ----------------------------------
      //
      case 'local_file':
                           
         include_once(SM_PATH . 'plugins/server_settings_backend/local_file_functions.php');
                        
                           
         // get access type, suid info if needed
         //
         if (!empty($backend_info['ACCESS_TYPE']) && is_array($backend_info['ACCESS_TYPE']))
            $access_type = retrieve_server_setting($backend_info['ACCESS_TYPE'], $quiet);
         else
            $error = _("Misconfiguration encountered when attempting to calculate the size of a file or directory on the server (file access method is misconfigured)");
         if (!empty($backend_info['SUID_LOCATION']) && is_array($backend_info['SUID_LOCATION']))
            $suid_binary = retrieve_server_setting($backend_info['SUID_LOCATION'], $quiet);
         else if ($access_type == 'SUID')
            $error = _("Misconfiguration encountered when attempting to calculate the size of a file or directory on the server (SUID_LOCATION is misconfigured)");
         else $suid_binary = '';
         if (!empty($backend_info['SUID_DEBUG_OUTPUT']) && is_array($backend_info['SUID_DEBUG_OUTPUT']))
            $suid_debug_file = retrieve_server_setting($backend_info['SUID_DEBUG_OUTPUT'], $quiet);
         else
            $suid_debug_file = '';
         if (!empty($backend_info['MASTER_SECURITY_CHECK']) && is_array($backend_info['MASTER_SECURITY_CHECK']))
            $master_security_check = retrieve_server_setting($backend_info['MASTER_SECURITY_CHECK'], $quiet);
         else
            $master_security_check = FALSE;
         if (!empty($backend_info['MASTER_USERNAME']) && is_array($backend_info['MASTER_USERNAME']))
            $master_username = retrieve_server_setting($backend_info['MASTER_USERNAME'], $quiet);
         else
            $master_username = NULL;


         // if using "master" security check, grab needed values from environment
         //
         if ($master_security_check)
         {
            if (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_PASSWORD', $master_password, SQ_SERVER)
             || $master_password == 'test-password')
               $error = _("Misconfiguration encountered when attempting to calculate the size of a file or directory on the server (master password is misconfigured)");
            if ($master_username
             && (!sqGetGlobalVar('SQUIRRELMAIL_MASTER_USERNAME', $master_username, SQ_SERVER)
              || $master_username == 'test-username'))
               $error = _("Misconfiguration encountered when attempting to calculate the size of a file or directory on the server (master username is misconfigured)");
         }
         else
            $master_username = $master_password = NULL;


         // manipulate username, domain and password if needed
         //
         list($user, $dom, $pass) = get_credentials($backend_info,
                                                    $username,
                                                    $domain,
                                                    sqauth_read_password(),
                                                    $quiet);


         // when all three are null, an error occured and $quiet
         // is turned on, so we can just fake an error
         //
         if ($quiet && is_null($user) && is_null($dom) && is_null($pass))
            $error = 'dummy';


         // validate we have what we need
         //
         if (empty($error)
          && (empty($access_type)
          || ($access_type == 'SUID' && empty($suid_binary))))
            $error = _("Misconfiguration encountered when attempting to calculate the size of a file or directory on the server (missing access type)");
         else if (empty($error))
         {

            // grab information about source (note NOT using lstat())
            //
            $ret = 0;
            $file_info = local_file_path_exists($access_type, $user, $pass,
                                                $path, TRUE, $suid_binary,
                                                $suid_debug_file, $master_username,
                                                $master_password, $quiet, FALSE);

            // if possible, try to determine $is_directory on our own
            //
            if ($autodetect)
            {
               if (!empty($file_info['type']) && $file_info['type'] == 'directory')
                  $is_directory = TRUE;
               else if (!empty($file_info['type']) && $file_info['type'] == 'file')
                  $is_directory = FALSE;
               else if (!empty($file_info['type']))
                  $is_directory = FALSE;  // treat other types as regular file
            }


            // just a file?
            //
            if (!is_null($ret) && !$is_directory)
            {
               if (isset($file_info['size']))
                  $ret = $file_info['size'];

               else
               {
                  $local_file = get_local_file($access_type, $user, $pass, $path,
                                               FALSE, $suid_binary, $suid_debug_file,
                                               $master_username, $master_password,
                                               $quiet);

                  // assume that unless $local_file is NULL,
                  // the file exists and all is normal,
                  // if not, use an empty value
                  //
                  if (is_null($local_file))
                  {
                     if ($quiet)
                     {
                        sq_change_text_domain($original_text_domain);
                        return NULL;
                     }
                     $ret = NULL;
                     global $sm_debug_mode, $server_settings_backend_debug;
                     server_settings_backend_init();
                     if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                     if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                        $error = sprintf(_("Error attempting to read \"%s\""), $path);
                     else
                        $error = _("Error attempting to read file");
                  }
                  else
                  {
                     if (!($ret = @filesize($local_file)))
                        $ret = 0;
                     if ($access_type != 'PHP') @unlink($local_file);
                  }
               }
            }


            // calculate directory?
            //
            else if (!is_null($ret))  // && $is_directory)
            {

               // get list of directory contents (note NOT using lstat)
               //
               $listing = get_local_directory_listing($access_type, $user, $pass, $path,
                                                      $suid_binary, $suid_debug_file,
                                                      $master_username, $master_password,
                                                      $quiet, FALSE);

               // add up each directory entry one at a time
               //
               if (!is_null($listing)) foreach ($listing as $entry)
               {

                  if ($entry['name'] == '.' || $entry['name'] == '..')
                     continue;

                  // include subdirectories?
                  //
                  if (!empty($entry['type']) && $entry['type'] == 'directory'
                   && !empty($entry['name']))
                  {
                     if ($recursive)
                     {
//TODO: path separator below may not work under all O/Ses(?)
                        $size = calculate_size($path . '/' . $entry['name'], TRUE, TRUE, $backend_info, $quiet);

                        if (is_null($size))
                        {
                           if ($quiet)
                           {
                              sq_change_text_domain($original_text_domain);
                              return NULL;
                           }
                           $ret = NULL;
                           global $sm_debug_mode, $server_settings_backend_debug;
                           server_settings_backend_init();
                           if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                           if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
//TODO: path separator below may not work under all O/Ses(?)
                              $error = sprintf(_("Error occurred calculating size of subdirectory \"%s\""), $path . '/' . $entry['name']);
                           else
                              $error = _("Error occurred calculating size of subdirectory");
                           break;
                        }
                        else $ret += $size;
                     }
                  }
                  else if (!empty($entry['type']) && $entry['type'] == 'file')
                  {
                     if (isset($entry['size']))
                        $ret += $entry['size'];

                     else if (!empty($entry['name']))
                     {
                        $local_file = get_local_file($access_type, $user, $pass,
//TODO: path separator below may not work under all O/Ses(?)
                                                     $path . '/' . $entry['name'],
                                                     FALSE, $suid_binary, $suid_debug_file,
                                                     $master_username, $master_password,
                                                     $quiet);

                        // assume that unless $local_file is NULL,
                        // the file exists and all is normal,
                        // if not, use an empty value
                        //
                        if (is_null($local_file))
                        {
                           if ($quiet)
                           {
                              sq_change_text_domain($original_text_domain);
                              return NULL;
                           }
                           $ret = NULL;
                           global $sm_debug_mode, $server_settings_backend_debug;
                           server_settings_backend_init();
                           if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                           if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
//TODO: path separator below may not work under all O/Ses(?)
                              $error = sprintf(_("Error attempting to read \"%s\""), $path . '/' . $entry['name']);
                           else
                              $error = _("Error attempting to read file");
                           break;
                        }
                        else
                        {
                           if (!($size = @filesize($local_file)))
                              $size = 0;
                           $ret += $size;
                           if ($access_type != 'PHP') @unlink($local_file);
                        }
                     }
                  }

               }
               else if (is_null($listing))
               {
                  if ($quiet)
                  {
                     sq_change_text_domain($original_text_domain);
                     return NULL;
                  }
                  $ret = NULL;
                  global $sm_debug_mode, $server_settings_backend_debug;
                  server_settings_backend_init();
                  if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
                  if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
                     $error = sprintf(_("Error attempting to obtain directory listing: \"%s\""), $path);
                  else
                     $error = _("Error attempting to obtain directory listing");
               }

            }


            if (!is_null($ret))
            {
               sq_change_text_domain($original_text_domain);
               return $ret;
            }

         }
         break;

   }


   // should not get here without an error message
   // unless something is very wrong
   //
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function calculate_size()");


   // spit out error and exit
   //
   if ($quiet) return NULL;
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



