<?php

/**
  * SquirrelMail Server Settings Backend Plugin
  * Copyright (c) 2008-2009 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage server_settings_backend
  *
  */



/**
  * Tests a setting value on the server.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * If the requested setting consists of multiple values,
  * the test value is deemed to be a match if it matches
  * any one of the values inside the setting.
  *
  * @param mixed   $test_value   The value to test against.
  * @param array   $backend_info An array of configuration values
  *                              that point to how the setting value
  *                              should be retrieved.
  * @param boolean $quiet        When TRUE, suppresses any error
  *                              messages and makes this function
  *                              return NULL without doing anything
  *                              else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE if the setting value matches the test value,
  *               FALSE if it does not or NULL if some problem/error
  *               occurred (and $quiet is TRUE).
  *
  */
function test_server_setting($test_value, $backend_info, $quiet=FALSE)
{

   include_once(SM_PATH . 'plugins/server_settings_backend/init.php');

   $value = retrieve_server_setting($backend_info, $quiet, $test_value);
   if (is_null($value)) return NULL;

   if (is_array($value)) return in_array($test_value, $value);
   else return $test_value == $value;

}



/**
  * Retrieves a setting from the server.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * If the requested setting consists of multiple values,
  * every effort is made to return them in an array, but
  * specifying the correct value of "MULTIPLE" in the
  * storage configuration settings helps ensure this will
  * work correctly.  If "MULTIPLE" is not found, default
  * behavior is to return multiple values in an array.
  *
  * @param array   $backend_info An array of configuration values
  *                              that point to how the setting value
  *                              should be retrieved.
  * @param boolean $quiet        When TRUE, suppresses any error
  *                              messages and makes this function
  *                              return NULL without doing anything
  *                              else (OPTIONAL; default = FALSE).
  * @param mixed $test_value     When given, this indicates that the
  *                              setting is being retrieved
  *                              specifically for testing per the
  *                              test_server_setting() function (or
  *                              similar), and so if there is a
  *                              different (more efficient kind of
  *                              lookup for this action, it can be
  *                              enabled.  When given, its value must
  *                              be the value being tested for
  *                              (OPTIONAL; default = NULL).
  * @param boolean $return_file_contents For backend types that work
  *                                      on a file scale, when this
  *                                      parameter is TRUE, the normal
  *                                      return value is placed inside
  *                                      an array to which the file
  *                                      contents are added as the
  *                                      second and last array element,
  *                                      and this two-element array is
  *                                      then returned.  This parameter
  *                                      has no effect for backends that
  *                                      do not work with files, such as
  *                                      LDAP and SQL (OPTIONAL;
  *                                      default = FALSE).
  *
  * @return mixed The setting value if it was retrieved normally
  *               or NULL if it was not (and $quiet is TRUE) (or
  *               a two-element array containing the setting value
  *               or NULL when errors ocurred and the file contents
  *               if $return_file_contents is TRUE).
  *
  */
function retrieve_server_setting($backend_info, $quiet=FALSE, $test_value=NULL,
                                 $return_file_contents=FALSE)
{

   include_once(SM_PATH . 'plugins/server_settings_backend/init.php');

   $original_text_domain = sq_change_text_domain('server_settings_backend');
   $error = '';


   if (!is_array($backend_info))
   {
      $error = _("Misconfiguration encountered when attempting to retrieve a setting from the server (malformed backend configuration)");
   }


   // for sanity, make sure all array keys are upper case
   //
   else
   {
      $temp_backend_info = $backend_info;
      $backend_info = array();
      foreach ($temp_backend_info as $key => $value)
         $backend_info[strtoupper($key)] = $value;
   }


   // did the user provide a value for us?
   // if so, use it, forget the rest of this function
   //
   if (isset($backend_info['VALUE']))
   {
      return $backend_info['VALUE'];
   }


   // did the user provide a custom function?
   // if so, use it, forget the rest of this function
   //
   if (!empty($backend_info['CUSTOM']))
   {
      if (!function_exists($backend_info['CUSTOM']))
         $error = sprintf(_("Misconfiguration encountered when attempting to retrieve a setting from the server (\"%s\" custom backend function unknown)"), $backend_info['CUSTOM']);
      else
         return $backend_info['CUSTOM']($backend_info);
   }


   // do we have an array of lookups?  we can assume so if
   // there is no VALUE, CUSTOM or BACKEND entry in the array
   // (at this point, VALUE and CUSTOM are already dealt with)
   //
   // when we have multiple lookups, just use the first one (multiple
   // lookups are only useful for performing multiple save actions)
   //
   if (empty($backend_info['BACKEND']))
      return retrieve_server_setting(array_shift($backend_info), $quiet, $test_value, $return_file_contents);


   // normal backend lookup... start by getting the backend type
   //
   $backend = strtolower($backend_info['BACKEND']);
   if (empty($error) && empty($backend)
    || ($backend != 'ftp' && $backend != 'local_file' && $backend != 'sql' && $backend != 'ldap'))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to retrieve a setting from the server (\"%s\" lookup unknown)"), $backend);
   }
   

   // what kind of backend will we be using to retrieve the value?
   //
   if (empty($error)) switch ($backend)
   {

      // ----------------------------------
      //
      case 'ftp':

         sq_change_text_domain($original_text_domain);
         include_once(SM_PATH . 'plugins/server_settings_backend/ftp_backend.php');
         return retrieve_server_setting_ftp($backend_info, $quiet,
                                            $test_value, $return_file_contents);
         break;



      // ----------------------------------
      //
      case 'local_file':
//TODO
         $error = "LOCAL_FILE STORAGE BACKEND FOR SERVER SETTINGS PLUGIN IS AS OF YET UNIMPLEMENTED";
         break;



      // ----------------------------------
      //
      case 'sql':

         sq_change_text_domain($original_text_domain);
         include_once(SM_PATH . 'plugins/server_settings_backend/sql_backend.php');
         return retrieve_server_setting_sql($backend_info, $quiet, $test_value);
         break;



      // ----------------------------------
      //
      case 'ldap':
//TODO
         $error = "LDAP STORAGE BACKEND FOR SERVER SETTINGS PLUGIN IS AS OF YET UNIMPLEMENTED";
         break;

   }


   // should not get here without an error message 
   // unless something is very wrong
   //
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function retrieve_server_setting()");


   // spit out error and exit
   //
   if ($quiet)
   {
      sq_change_text_domain($original_text_domain);
      return NULL;
   }
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }
   exit;

}



/**
  * Stores a setting on the server.
  *
  * If some fatal error occurs due to misconfiguration or
  * backend failure, an error message is printed and execution
  * stops immediately, unless $quiet is TRUE.
  *
  * @param mixed   $new_value    The value to be stored.
  * @param array   $backend_info An array of configuration values
  *                              that point to how the setting value
  *                              should be stored.
  * @param boolean $add          When TRUE, and if the setting being
  *                              updated consists of multiple individual
  *                              values, $new_value is added to them
  *                              and the other values are left as is.
  *                              If the setting's configuration does
  *                              not include a "MULTIPLE" item, then 
  *                              this flag has no effect and the setting
  *                              is updated as if it were turned off
  *                              (OPTIONAL; default = FALSE).
  * @param boolean $remove       When TRUE, and if the setting being
  *                              updated consists of multiple individual
  *                              values, $new_value is removed from
  *                              those values (if found within) and the
  *                              other values are left as is.  If the
  *                              setting's configuration does not include
  *                              a "MULTIPLE" item, then this flag has no
  *                              effect and the setting is updated as if
  *                              it were turned off (OPTIONAL; default
  *                              = FALSE).
  * @param boolean $quiet        When TRUE, suppresses any error
  *                              messages and makes this function
  *                              return NULL without doing anything
  *                              else (OPTIONAL; default = FALSE).
  *
  * @return mixed TRUE if the setting was stored normally or
  *               NULL if it was not (and $quiet is TRUE).
  *
  */
function put_server_setting($new_value, $backend_info,
                            $add=FALSE, $remove=FALSE, $quiet=FALSE)
{

   include_once(SM_PATH . 'plugins/server_settings_backend/init.php');

   $original_text_domain = sq_change_text_domain('server_settings_backend');
   $error = '';


   if (!is_array($backend_info))
   {
      $error = _("Misconfiguration encountered when attempting to retrieve a setting from the server (malformed backend configuration)");
   }


   // for sanity, make sure all array keys are upper case
   //
   else
   {
      $temp_backend_info = $backend_info;
      $backend_info = array();
      foreach ($temp_backend_info as $key => $value)
         $backend_info[strtoupper($key)] = $value;
   }


   // do we have an array of lookups?  we can assume so if
   // there is no VALUE, CUSTOM or BACKEND entry in the array
   //
   // when we have multiple lookups, more than one save action
   // is to be performed - just recursively call ourselves
   //
   if (empty($backend_info['VALUE']) && empty($backend_info['CUSTOM'])
    && empty($backend_info['BACKEND']))
   {
      $return_OK = TRUE;
      foreach ($backend_info as $key => $value)
         if (!put_server_setting($new_value, $value, $add, $remove, $quiet))
            $return_OK = FALSE;
      if (!$return_OK)
         return NULL;
      else
         return TRUE;
   }


   // normal backend storage... start by getting the backend type
   //
   $backend = '';
   if (!empty($backend_info['BACKEND']))
      $backend = strtolower($backend_info['BACKEND']);
   if (empty($error) && empty($backend)
    || ($backend != 'ftp' && $backend != 'local_file' && $backend != 'sql' && $backend != 'ldap'))
   {
      $error = sprintf(_("Misconfiguration encountered when attempting to store a setting on the server (\"%s\" lookup unknown)"), $backend);
   }


   // what kind of backend will we be using to store the value
   //
   if (empty($error)) switch ($backend)
   {

      // ----------------------------------
      //
      case 'ftp':

         include_once(SM_PATH . 'plugins/server_settings_backend/ftp_backend.php');
         return put_server_setting_ftp($new_value, $backend_info, $add, $remove, $quiet);
         break;



      // ----------------------------------
      //
      case 'local_file':
//TODO
         $error = "LOCAL_FILE STORAGE BACKEND FOR SERVER SETTINGS PLUGIN IS AS OF YET UNIMPLEMENTED";
         break;



      // ----------------------------------
      //
      case 'sql':

         include_once(SM_PATH . 'plugins/server_settings_backend/sql_backend.php');
         return put_server_setting_sql($new_value, $backend_info, $add, $remove, $quiet);
         break;
   
   
   
      // ----------------------------------
      //
      case 'ldap':
//TODO
         $error = "LDAP STORAGE BACKEND FOR SERVER SETTINGS PLUGIN IS AS OF YET UNIMPLEMENTED";
         break;
   
   }
   
      
   // should not get here without an error message
   // unless something is very wrong
   //    
   if (empty($error))
      $error = _("Unknown error in Server Settings Backend plugin, function put_server_setting()");

         
   // spit out error and exit
   //    
   if ($quiet) return NULL;
   sq_change_text_domain('squirrelmail');  // NOT $original_text_domain
   global $color;
   $ret = plain_error_message($error, $color);
   if (check_sm_version (1, 5, 2))
   {        
      echo $ret;
      global $oTemplate;
      $oTemplate->display('footer.tpl');
   }  
   exit;

}



/**
  * Based on the given configuration and default
  * values, this function possibly modifies the
  * username, domain and/or password and returns
  * them.
  *
  * @param array  $config_info An array of settings that dictate
  *                            if and how the defaults will be
  *                            modified.
  * @param string $user        The default username.
  * @param string $dom         The default domain name.
  * @param string $pass        The default password.
  * @param boolean $quiet      When TRUE, suppresses any error
  *                            messages and makes this function
  *                            return NULL without doing anything
  *                            else (OPTIONAL; default = FALSE).
  *
  * @return array A three-element array consisting of the
  *               resultant username, domain name and password,
  *               or three NULLs if an error occurred and $quiet is TRUE.
  *
  */
function get_credentials($backend_info, $user, $dom, $pass, $quiet=FALSE)
{

   $final_user = $user;
   $final_dom  = $dom;
   $final_pass = $pass;
   $delimiter  = '@';


   // get domain manipulation configuration/manipulation rules
   //
   if (!empty($backend_info['DOMAIN']) && is_array($backend_info['DOMAIN']))
   {

      $dom_rules = $backend_info['DOMAIN'];


      // for sanity, make sure all array keys are upper case
      //
      $temp_dom_rules = $dom_rules;
      $dom_rules = array();
      foreach ($temp_dom_rules as $key => $value)
         $dom_rules[strtoupper($key)] = $value;


      // did the user provide a value for us?
      // if so, use it, forget the rest of these rules
      //
      if (isset($dom_rules['VALUE']))
      {
         $final_dom = $dom_rules['VALUE'];
      }


      // did the user provide a custom function?
      // or another backend lookup?
      // if so, use it, forget the rest of these rules
      //
      else if (!empty($dom_rules['CUSTOM'])
       || !empty($dom_rules['BACKEND']))
      {
         $final_dom = retrieve_server_setting($dom_rules, $quiet);
         if ($quiet && is_null($final_dom))
            return array(NULL, NULL, NULL);
      }


      if (!empty($dom_rules['REGULAR_EXPRESSION_PATTERN'])
       && !empty($dom_rules['REGULAR_EXPRESSION_REPLACEMENT']))
      {
         $final_dom = preg_replace($dom_rules['REGULAR_EXPRESSION_PATTERN'],
                                   $dom_rules['REGULAR_EXPRESSION_REPLACEMENT'],
                                   $final_dom);
      }

   }


   // get username manipulation configuration/manipulation rules
   //
   if (!empty($backend_info['USERNAME']) && is_array($backend_info['USERNAME']))
   {

      $user_rules = $backend_info['USERNAME'];


      // for sanity, make sure all array keys are upper case
      //
      $temp_user_rules = $user_rules;
      $user_rules = array();
      foreach ($temp_user_rules as $key => $value)
         $user_rules[strtoupper($key)] = $value;


      // did the user provide a value for us?
      // if so, use it, forget the rest of these rules
      //
      if (isset($user_rules['VALUE']))
      {
         $final_user = $user_rules['VALUE'];
      }


      // did the user provide a custom function?
      // or another backend lookup?
      // if so, use it, forget the rest of these rules
      //
      else if (!empty($user_rules['CUSTOM'])
       || !empty($user_rules['BACKEND']))
      {
         $final_user = retrieve_server_setting($user_rules, $quiet);
         if ($quiet && is_null($final_user))
            return array(NULL, NULL, NULL);
      }


      if (!empty($user_rules['DELIMITER']))
         $delimiter = $user_rules['DELIMITER'];


      if (!empty($user_rules['STRIP_DOMAIN'])
       && strpos($final_user, $delimiter) !== FALSE)
         $final_user = substr($final_user, 0, strpos($final_user, $delimiter));


      if (!empty($user_rules['ADD_DOMAIN'])
       && $user_rules['ADD_DOMAIN'] == 'ORIGINAL'
       && strpos($final_user, $delimiter) === FALSE)
         $final_user = $final_user . $delimiter . $dom;


      if (!empty($user_rules['ADD_DOMAIN'])
       && $user_rules['ADD_DOMAIN'] == 'MODIFIED'
       && strpos($final_user, $delimiter) === FALSE)
         $final_user = $final_user . $delimiter . $final_dom;


      if (!empty($user_rules['REGULAR_EXPRESSION_PATTERN'])
       && !empty($user_rules['REGULAR_EXPRESSION_REPLACEMENT']))
      {
         $final_user = preg_replace($user_rules['REGULAR_EXPRESSION_PATTERN'],
                                    $user_rules['REGULAR_EXPRESSION_REPLACEMENT'],
                                    $final_user);
      }

   }


   // get password manipulation configuration/manipulation rules
   //
   if (!empty($backend_info['PASSWORD']) && is_array($backend_info['PASSWORD']))
   {

      $pass_rules = $backend_info['PASSWORD'];


      // for sanity, make sure all array keys are upper case
      //
      $temp_pass_rules = $pass_rules;
      $pass_rules = array();
      foreach ($temp_pass_rules as $key => $value)
         $pass_rules[strtoupper($key)] = $value;


      // did the user provide a value for us?
      // if so, use it, forget the rest of these rules
      //
      if (isset($pass_rules['VALUE']))
      {
         $final_pass = $pass_rules['VALUE'];
      }


      // did the user provide a custom function?
      // or another backend lookup?
      // if so, use it, forget the rest of these rules
      //
      else if (!empty($pass_rules['CUSTOM'])
       || !empty($pass_rules['BACKEND']))
      {
         $final_pass = retrieve_server_setting($pass_rules, $quiet);
         if ($quiet && is_null($final_pass))
            return array(NULL, NULL, NULL);
      }


      if (!empty($pass_rules['REGULAR_EXPRESSION_PATTERN'])
       && !empty($pass_rules['REGULAR_EXPRESSION_REPLACEMENT']))
      {
         $final_pass = preg_replace($pass_rules['REGULAR_EXPRESSION_PATTERN'],
                                    $pass_rules['REGULAR_EXPRESSION_REPLACEMENT'],
                                    $final_pass);
      }

   }

   return array($final_user, $final_dom, $final_pass);

}



/**
  * Escapes single quotes and backslashes in a string.
  *
  * @param string $string The string to be escaped.
  *
  * @return string The escaped string.
  *
  */
function ssb_str_esc($string)
{
   return str_replace(array('\\', '\''), array('\\\\', '\\\''), $string);
}



/**
  * Generates a unique temporary local file name in
  * the directory of choice.
  *
  * NOTE that the returned file handle will point to
  * an OPEN file in WRITE mode.
  *
  * @param boolean $quiet     When TRUE, suppresses any error
  *                           messages and makes this function
  *                           return NULL without doing anything
  *                           else (OPTIONAL; default = FALSE).
  * @param string  $name      The desired file name, which
  *                           will be used if available (OPTIONAL;
  *                           default not used).
  * @param string  $extension The desired file extension, which
  *                           will be used if available (OPTIONAL;
  *                           default not used).
  *
  * @return mixed A two-element array containing the
  *               file handle to the (open for writing!)
  *               temporary file and the full path to
  *               the file (in that order), or NULL if
  *               an error occurred (and $quiet is TRUE).
  *
  */
function ssb_get_temp_file($quiet=FALSE, $name='', $extension='')
{

   // give up after 1000 tries
   //
   $maximum_tries = 1000;


   // generate random local filename, use SM attach dir
   //
   global $username, $attachment_dir;
   include_once(SM_PATH . 'functions/prefs.php');
   include_once(SM_PATH . 'functions/strings.php');
   $hashed_attachment_dir = getHashedDir($username, $attachment_dir);
   $FILE = FALSE;
   $full_local_file = '';


   // using PHP >= 4.3.2 we can be truly atomic here
   //
   $filemods = check_php_version(4, 3, 2) ? 'x' : 'w';

   // try pure file name if asked for
   //
   if (!empty($name))
   {
      $full_local_file = "$hashed_attachment_dir/$name"
                       . (!empty($extension) ? '.' . $extension : '');
      if (!file_exists($full_local_file))
      {
         // try to open for (binary) writing
         //
         $FILE = @fopen($full_local_file, $filemods);
      }
   }


   // iterate through other possibilities
   //
   if (!is_resource($FILE)) for ($try=0; $try < $maximum_tries; ++$try)
   {

      $local_file = GenerateRandomString(32, '', 7);
      $full_local_file = "$hashed_attachment_dir/"
                       . (!empty($name) ? $name . '-' : '')
                       . $local_file
                       . (!empty($extension) ? '.' . $extension : '');

      // filename collision; try again
      //
      if (file_exists($full_local_file))
         continue;

      // try to open for (binary) writing
      //
      $FILE = @fopen($full_local_file, $filemods);

      // success!
      //
      if ($FILE !== FALSE)
         break;

   }


   // catch error
   //
   if (!is_resource($FILE))
   {
      if ($quiet) return NULL;

      sq_change_text_domain('server_settings_backend');
      $error = _("Could not open temporary local file");
      sq_change_text_domain('squirrelmail');

      global $color;
      $ret = plain_error_message($error, $color);
      if (check_sm_version (1, 5, 2))
      {
         echo $ret;
         global $oTemplate;
         $oTemplate->display('footer.tpl');
      }
      exit;
   }

   return array($FILE, $full_local_file);

}



/**
  * Parse out target setting value from a file
  *
  * @param string  $file_path     The path to the file from which
  *                               the setting value is to be parsed.
  * @param string  $parse_pattern The regular expression pattern
  *                               needed to parse the setting value
  *                               out of the file.
  * @param string  $pattern_group_number The group number of the regular
  *                                      expression parsing pattern that
  *                                      contains ONLY the actual value
  *                                      (and not, for example, also the
  *                                      setting name).
  * @param boolean $quiet         When TRUE, suppresses any error
  *                               messages and makes this function
  *                               return NULL without doing anything
  *                               else (OPTIONAL; default = FALSE).
  * @param boolean $return_file_contents When TRUE, the normal return
  *                                      value is placed inside an array
  *                                      to which the file contents are
  *                                      added as the second and last
  *                                      array element, and this two-
  *                                      element array is then returned
  *                                      (OPTIONAL; default = FALSE).
  *
  * @return mixed An array containing the setting value and its offset in
  *               the target file (in that order) if it was retrieved
  *               normally and was only found once in the file (scalar
  *               value), or an array of arrays, where the sub-arrays
  *               contain each setting value and its offset in the target
  *               file (in that order) if the setting is found more than
  *               once in the file (the caller should check if the return
  *               value's first array element is an array itslef to
  *               determine if the result is non-scalar), or NULL if the
  *               setting is not found in the file or if an error occurred
  *               (and $quiet is TRUE).  Note that if $return_file_contents
  *               is TRUE, the normal return value described above will be
  *               placed inside a two-element array, and the file contents
  *               placed as the second array element, and that is returned.
  *
  */
function parse_value_from_file($file_path, $parse_pattern,
                               $pattern_group_number, $quiet=FALSE,
                               $return_file_contents=FALSE)
{

   // get full file contents into memory
   //
   if (check_php_version(4, 3, 2))
      $file_contents = file_get_contents($file_path);
   else
   {
      $file_contents = file($file_path);
      $file_contents = implode('', $file_contents);
   }

   $ret = parse_value($file_contents, $parse_pattern,
                      $pattern_group_number, $quiet);

   if ($return_file_contents)
      $ret = array($ret, $file_contents);

   return $ret;

}



/**
  * Parse out target setting value from a string
  *
  * @param string  $haystack      The string from which the
  *                               setting value is to be parsed.
  * @param string  $parse_pattern The regular expression pattern
  *                               needed to parse the setting value
  *                               out of the string.
  * @param string  $pattern_group_number The group number of the regular
  *                                      expression parsing pattern that
  *                                      contains ONLY the actual value
  *                                      (and not, for example, also the
  *                                      setting name).
  * @param boolean $quiet         When TRUE, suppresses any error
  *                               messages and makes this function
  *                               return NULL without doing anything
  *                               else (OPTIONAL; default = FALSE).
  *
  * @return mixed An array containing the setting value and its offset in
  *               the original string (in that order) if it was retrieved
  *               normally and was only found once in the string (scalar
  *               value), or an array of arrays, where the sub-arrays
  *               contain each setting value and its offset in the original
  *               string (in that order) if the setting is found more than
  *               once in the string (the caller should check if the return
  *               value's first array element is an array itslef to
  *               determine if the result is non-scalar), or NULL if the
  *               setting is not found in the string or if an error occurred
  *               (and $quiet is TRUE).
  *
  */
function parse_value($haystack, $parse_pattern, $pattern_group_number, $quiet=FALSE)
{

   // $matches[$pattern_group_number] will be an
   // array of any and all matches, assuming it
   // is correctly configured
   //
   preg_match_all($parse_pattern, $haystack, $matches, PREG_OFFSET_CAPTURE);


   // dump matches when in debug mode so admin can configure
   // pattern and pattern group number correctly
   //
   global $sm_debug_mode, $server_settings_backend_debug;
   include_once(SM_PATH . 'plugins/server_settings_backend/init.php');
   server_settings_backend_init();
   if ($server_settings_backend_debug) $sm_debug_mode |= SM_DEBUG_MODE_SIMPLE;
   if ($sm_debug_mode & SM_DEBUG_MODE_SIMPLE)
   {
      echo '<hr />SERVER SETTINGS BACKEND MATCHES (PATTERN = "'
         . $parse_pattern . '") ARE:';
      sm_print_r($matches);
      echo '<hr />CHOSEN PATTERN GROUP (' . $pattern_group_number . ') IS:';
      sm_print_r($matches[$pattern_group_number]);
      echo "<hr />";
   }


   // nothing found?
   //
   if (empty($matches[$pattern_group_number]))
      return NULL;


   // if only one value found, return as a scalar
   //
   else if (sizeof($matches[$pattern_group_number]) == 1)
      $matches[$pattern_group_number] = $matches[$pattern_group_number][0];


   return $matches[$pattern_group_number];

}



/**
  * Edit a string of file contents, updating a setting
  * with new value(s) to be saved
  *
  * All value updates, replacements and additions make the following
  * string substitutions before the update/replacement/addition is
  * performed:
  *
  *   %1 is replaced with the new value
  *   %u is replaced with the (pre-parsed) username
  *      (parsed according to the rules in the $backend_info's USERNAME entry)
  *   %d is replaced with the (pre-parsed) domain name
  *      (parsed according to the rules in the $backend_info's DOMAIN entry)
  *
  * The arguments subject to these substitutions include
  * $new_setting_template, $replacement_pattern, $update_search_pattern,
  * $update_replace_pattern and $delete_keep_pattern.
  *
  * @param string  $file_contents The file represented as a string
  * @param array   $backend_info  An array of configuration values
  *                               that point to how the setting value
  *                               should be stored.
  * @param string  $user          The pre-parsed username corresponding
  *                               to the user who is logged in (parsed
  *                               according to the rules specified in the
  *                               USERNAME entry in the storage backend
  *                               array).
  * @param string  $dom           The pre-parsed domain name corresponding
  *                               to the user who is logged in (parsed
  *                               according to the rules specified in the
  *                               DOMAIN entry in the storage backend
  *                               array).
  * @param mixed   $value         The new value(s) to be added/removed
  *                               to/from the file (can be scalar or
  *                               an array)
  * @param string  $current_value The current value of the target
  *                               setting.  Should be an array of
  *                               values (arrays) (even if only
  *                               one/scalar value therein) as when
  *                               retrieving the setting's current
  *                               value in the file per parse_value()
  *                               with possible offsets indicating
  *                               the current location of the setting
  *                               in the file string.
  * @param string  $parse_pattern The pattern used to parse the setting
  *                               out of the file string.
  * @param string  $delete_keep_pattern The replacement pattern used
  *                                     when $parse_pattern is used
  *                                     to remove the setting from
  *                                     the file string.
  * @param string  $new_setting_template The template string used to
  *                                      add the setting newly to the
  *                                      end of the file.
  * @param string  $replacement_pattern A pattern used to make the setting
  *                                     replacement in the file, which, if
  *                                     given, renders all other functionality
  *                                     herein moot - no parameters other
  *                                     than $file_contents, $value and
  *                                     $parse_pattern are used at all
  *                                     (OPTIONAL; default empty - not used).
  * @param string  $update_search_pattern A pattern that, if given along with
  *                                       $update_replace_pattern, will be
  *                                       used in a regular expression to
  *                                       replace any non-MULTIPLE/single
  *                                       value settings.  When used, all
  *                                       other arguments herein are not
  *                                       used except $user, $dom, $value,
  *                                       $backend_info and $file_contents.
  * @param string  $update_replace_pattern A pattern that, if given along with
  *                                        $update_replace_pattern, will be
  *                                        used in a regular expression to
  *                                        replace any non-MULTIPLE/single
  *                                        value settings.  When used, all
  *                                        other arguments herein are not
  *                                        used except $user, $dom, $value,
  *                                        $backend_info and $file_contents.
  * @param boolean $add_to_top    When TRUE, indicates that new values
  *                               or multiple values being added to the
  *                               file should be added to the top instead
  *                               of appended to the end (OPTIONAL; default
  *                               is 0 - add to end of file).
  * @param boolean $add           When TRUE, and if the setting being
  *                               updated consists of multiple individual
  *                               values, $value is added to them and
  *                               the other values are left as is.  If
  *                               the setting's configuration does not
  *                               include a "MULTIPLE" item, then this
  *                               flag has no effect and the setting is
  *                               updated as if it were turned off
  *                               (OPTIONAL; default = FALSE).
  * @param boolean $remove        When TRUE, and if the setting being
  *                               updated consists of multiple individual
  *                               values, $value is removed from those
  *                               values (if found within) and the other
  *                               values are left as is.  If the setting's
  *                               configuration does not include a
  *                               "MULTIPLE" item, then this flag has no
  *                               effect and the setting is updated as if
  *                               it were turned off (OPTIONAL; default
  *                               = FALSE).
  * @param boolean $quiet         When TRUE, suppresses any error
  *                               messages and makes this function
  *                               return NULL without doing anything
  *                               else (OPTIONAL; default = FALSE).
  *
  * @return string The newly updated file contents.
  *
  */
function edit_file_update_setting($file_contents, $backend_info, $user, $dom,
                                  $value, $current_value, $parse_pattern,
                                  $delete_keep_pattern, $new_setting_template,
                                  $replacement_pattern='',
                                  $update_search_pattern='',
                                  $update_replace_pattern='', $add_to_top=0,
                                  $add=FALSE, $remove=FALSE, $quiet=FALSE)
{

   // make new value(s) an array, even if only have one/scalar value
   //
   if (is_null($value) || $value === '')
      $new_values = array();
   else if (!is_array($value))
      $new_values = array($value);
   else
      $new_values = $value;


   // if we are using a custom search/replace with a single value
   // setting, use it and then get outta here
   //
   if ((empty($backend_info['MULTIPLE']) || $backend_info['MULTIPLE'] != 'MULTIPLE')
    && !empty($update_search_pattern) && !empty($update_replace_pattern))
   {
      $new_value_replacement = '';
//TODO: will there ever be new values here that fall through this test?
      if (is_array($new_values) && isset($new_values[0]))
         $new_value_replacement = $new_values[0];

      $update_replace_pattern = str_replace(array('%1', '%u', '%d'),
                                            array($new_value_replacement,
                                                  $user, $dom),
                                            $update_replace_pattern);
      $update_search_pattern = str_replace(array('%1', '%u', '%d'),
                                           array($new_value_replacement,
                                                 $user, $dom),
                                           $update_search_pattern);

      if (preg_match($update_search_pattern, $file_contents))
         return preg_replace($update_search_pattern, $update_replace_pattern, $file_contents);
   }


   // if we are replacing one value with a regular
   // expression, let's do that and bail
   //
   if ((empty($backend_info['MULTIPLE']) || $backend_info['MULTIPLE'] != 'MULTIPLE')
    && !empty($replacement_pattern))
   {
      $new_value_replacement = '';
//TODO: will there ever be new values here that fall through this test?
      if (is_array($new_values) && isset($new_values[0]))
         $new_value_replacement = $new_values[0];

      $replacement_pattern = str_replace(array('%1', '%u', '%d'),
                                         array($new_value_replacement,
                                               $user, $dom),
                                         $replacement_pattern);

      if (preg_match($parse_pattern, $file_contents))
         return preg_replace($parse_pattern, $replacement_pattern, $file_contents);
   }


   // if adding a value to a multi-value setting...
   //
   if ($add && !empty($backend_info['MULTIPLE']))
   {

      // add new values and current values
      //
      $new_values = array_merge($current_value, $new_values);

   }


   // if removing a value from a multi-value setting...
   //
   if ($remove && !empty($backend_info['MULTIPLE']))
   {

      // subtract the new value from current
      //
      $temp_new_values = $current_value;
      foreach ($current_value as $cur_key => $cur_val)
         foreach ($new_values as $new_key => $new_val)
            if (isset($cur_val[0]) && $cur_val[0] == $new_val)
               unset($temp_new_values[$cur_key]);
      $new_values = $temp_new_values;

   }


   // if multiple values are kept in one field serialized
   // or imploded, just build that value
   //
   if (!empty($backend_info['MULTIPLE']) && $backend_info['MULTIPLE'] == 'SERIALIZE')
   {

      $temp = array();
      foreach ($new_values as $key => $value)
         if (is_array($value) && isset($value[0]))
            $temp[] = $value[0];
         else if (!is_array($value))
            $temp[] = $value;
      $new_values = array(serialize($temp));

   }
   else if (!empty($backend_info['MULTIPLE']) && $backend_info['MULTIPLE'] != 'MULTIPLE')
   {

      $temp = array();
      foreach ($new_values as $key => $value)
         if (is_array($value) && isset($value[0]))
            $temp[] = $value[0];
         else if (!is_array($value))
            $temp[] = $value;
      $new_values = array(implode($backend_info['MULTIPLE'], $temp));

   }


   // if we are just replacing one value, let's do that
   //
   if ((empty($backend_info['MULTIPLE']) || $backend_info['MULTIPLE'] != 'MULTIPLE')
    && is_array($current_value[0]) && isset($current_value[0][1]))  // if $new_values (or I suppose $current_value) has more than one element here, something really went wrong; for now, we won't test that condition
   {

      // current position and length of value is known - replace it
      //
      $new_file_contents = substr($file_contents, 0, $current_value[0][1]);
      if (empty($new_values))
         $new_file_contents .= '';
      else if (is_array($new_values[0]))
      {
         if (isset($new_values[0][0]))
            $new_file_contents .= $new_values[0][0];
         // (no need; like a no-op) else $new_file_contents .= '';
      }
      else
         $new_file_contents .= $new_values[0];
      $new_file_contents .= substr($file_contents,
                                   $current_value[0][1] + strlen($current_value[0][0]));

   }


   // for scalar saves that are new additions (not replacements)
   // or when the setting occurs multiple times in the same file
   //
   else
   {

      // first, weed current ones out (with scalar saves, there
      // may not be one, but check to be sure)
      //
      $delete_keep_pattern = str_replace(array('%u', '%d'),
                                         array($user, $dom),
                                         $delete_keep_pattern);
      $new_file_contents = preg_replace($parse_pattern, $delete_keep_pattern, $file_contents);


      // next, re-add whatever values we need to
      //
      foreach ($new_values as $new_value)
      {
         $addition = '';
         if (is_array($new_value) && isset($new_value[0]))
            $addition = str_replace(array('%1', '%u', '%d'),
                                    array($new_value[0], $user, $dom),
                                    $new_setting_template);
         else if (!is_array($new_value))
            $addition = str_replace(array('%1', '%u', '%d'),
                                    array($new_value, $user, $dom),
                                    $new_setting_template);
         if ($add_to_top)
            $new_file_contents = $addition . $new_file_contents;
         else
            $new_file_contents .= $addition;
      }

   }


   // strip out extra blank lines if needed
   //
   if (!empty($backend_info['MAX_SEQUENTIAL_EMPTY_LINES']))
   {
      $allowed_newlines = '';

      // we intentionally add one more than MAX_SEQUENTIAL_EMPTY_LINES below
      // because newlines and blank lines are not the same thing
      //
      $new_file_contents = preg_replace("/(\n{" . ($backend_info['MAX_SEQUENTIAL_EMPTY_LINES'] + 1) . "})\n+" . '/',
                                        '$1',
                                        $new_file_contents);
   }


   return $new_file_contents;

}



/**
  * Parses permissions from octal number to 
  * displayable format and back again.
  *
  * @param mixed $permissions Either an octal (integer)
  *                           permissions number to be converted
  *                           into a displayable string or the
  *                           string that will be converted into
  *                           an octal permissions number.
  *
  * @return mixed Either the converted octal permissions number
  *               or displayable string.
  *
  */
function parse_permissions($permissions)
{

   // convert from octal to string
   //
   if (is_int($permissions))
   {

      if (($permissions & 0xC000) == 0xC000) {
         // Socket
         $mode = 's';
      } elseif (($permissions & 0xA000) == 0xA000) {
         // Symbolic Link
         $mode = 'l';
      } elseif (($permissions & 0x8000) == 0x8000) {
         // Regular
         $mode = '-';
      } elseif (($permissions & 0x6000) == 0x6000) {
         // Block special
         $mode = 'b';
      } elseif (($permissions & 0x4000) == 0x4000) {
         // Directory
         $mode = 'd';
      } elseif (($permissions & 0x2000) == 0x2000) {
         // Character special
         $mode = 'c';
      } elseif (($permissions & 0x1000) == 0x1000) {
         // FIFO pipe
         $mode = 'p';
      } else {
         // Unknown
         //$mode = 'u';
         $mode = '';
      }

      // Owner
      $mode .= (($permissions & 0x0100) ? 'r' : '-');
      $mode .= (($permissions & 0x0080) ? 'w' : '-');
      $mode .= (($permissions & 0x0040) ?
               (($permissions & 0x0800) ? 's' : 'x' ) :
               (($permissions & 0x0800) ? 'S' : '-'));

      // Group
      $mode .= (($permissions & 0x0020) ? 'r' : '-');
      $mode .= (($permissions & 0x0010) ? 'w' : '-');
      $mode .= (($permissions & 0x0008) ?
               (($permissions & 0x0400) ? 's' : 'x' ) :
               (($permissions & 0x0400) ? 'S' : '-'));

      // World
      $mode .= (($permissions & 0x0004) ? 'r' : '-');
      $mode .= (($permissions & 0x0002) ? 'w' : '-');
      $mode .= (($permissions & 0x0001) ?
               (($permissions & 0x0200) ? 't' : 'x' ) :
               (($permissions & 0x0200) ? 'T' : '-'));

      return $mode;

   }

   // convert from string to octal
   //
   else
   {

      $permissions = trim($permissions);

      // TODO: is this assumption alright to make?
      //
      if (strlen($permissions) == 9) $permissions = '-' . $permissions;

      // TODO: should we generate or return some kind of error instead?
      //
      if (strlen($permissions) != 10) return 0;

      $mode = 0;

      if ($permissions[1] == 'r') $mode += 0400;
      if ($permissions[2] == 'w') $mode += 0200;
      if ($permissions[3] == 'x') $mode += 0100;
      else if ($permissions[3] == 's') $mode += 04100;
      else if ($permissions[3] == 'S') $mode += 04000;

      if ($permissions[4] == 'r') $mode += 040;
      if ($permissions[5] == 'w') $mode += 020;
      if ($permissions[6] == 'x') $mode += 010;
      else if ($permissions[6] == 's') $mode += 02010;
      else if ($permissions[6] == 'S') $mode += 02000;

      if ($permissions[7] == 'r') $mode += 04;
      if ($permissions[8] == 'w') $mode += 02;
      if ($permissions[9] == 'x') $mode += 01;
      else if ($permissions[9] == 't') $mode += 01001;
      else if ($permissions[9] == 'T') $mode += 01000;

      return $mode;

   }

}



