<?php

/**
  * SquirrelMail Server Settings Plugin
  * Copyright (c) 2008-2010 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage server_settings
  *
  */

global $server_settings_option_pages;



// This is where the options pages that let users configure
// various parts of your backend systems are defined.
//
// You may define as many options pages as you'd like.  Each
// one is defined in an array that consists of the following
// elements (as the corresponding array keys - see the examples
// below):
//
//    TITLE          This text will be shown both on the SquirrelMail
//                   Options page (where all other options pages are
//                   listed) as well as at the top of the actual page
//                   you are creating.
//
//    DESCRIPTION    This text will be shown on the SquirrelMail
//                   Options page (where all other options pages are
//                   listed), and should give a short (one or two
//                   sentence) explanation of the settings that are
//                   controlled on the page you are creating.
//
//    NO_OPTION_PAGE If present (filled with any non-empty value),
//                   this indicates that no link on the SquirrelMail
//                   Options page for this group of settings.  This
//                   is most commonly used in conjunction with the
//                   "MENU_LINK" element.  Please ensure that this
//                   element *does not* exist when an option link
//                   is to be shown for this group of settings.
//
//    MENU_LINK      When present, this is the text of a link that
//                   will be added to the menu that is shown across
//                   the top of all SquirrelMail content pages
//                   (the menu that typically starts with "Compose",
//                   "Addresses", "Folders", etc.).  The link will
//                   take the user to the options page for this
//                   settings group.
//
//    RETURN_TO_SELF When present (filled with any non-empty value),
//                   the option page will simply reload when the user
//                   clicks the submit button (instead of returning
//                   to the SquirrelMail main Options page).
//
//    WIDGETS        This is an array of the controls/settings that
//                   will be contained in the options page.  This
//                   array should contain all the typical elements
//                   that help define option widgets per the normal
//                   SquirrelMail options API explained here (NAME,
//                   CAPTION, TYPE, etc.):
//
//                      http://squirrelmail.org/docs/devel/devel-4.html#ss4.7
//
//                   Additionally, you may specify three more special
//                   elements in the widget's configuration:
//
//                      SECTION        This defines the subsection on
//                                     the options page under which this
//                                     widget should be placed.  It is
//                                     optional, and if not given, it will
//                                     be grouped in a generic unlabelled
//                                     section.
//
//                      SETUP_LOCATION This indicates where the initial
//                                     value for the widget should be
//                                     placed in the widget information
//                                     array.  This is NOT a required
//                                     element, and if not given, the
//                                     value will be placed in
//                                     "initial_value".  Sometimes this
//                                     needs to be changed, such as for
//                                     options of type SM_OPT_EDIT_LIST,
//                                     which need to be set as "posvals".
//
//                      MULTIPLE_VALUE_UPDATES_SEPERATELY
//                                     If present (filled with any non-empty
//                                     value), this indicates that when saving
//                                     widgets consisting of multiple values,
//                                     adds and removals should be done
//                                     seperately (as opposed to reconstructing
//                                     the whole list and re-saving it all at
//                                     once).  This is useful for backends
//                                     such as SQL where unchanged items in the
//                                     list should be left as is.
//
//                   If the setting should be stored by SquirrelMail
//                   in the normal user preferences, then this is all
//                   that is needed, however, if the setting corresponds
//                   to a configuration value stored elsewhere on the
//                   server, you must also define elements here that
//                   indicate how to retrieve and set the value:
//
//                      STORAGE  This is a subarray of information explaining
//                               how to look up the setting's value and put
//                               a new value in its place.  The elements herein
//                               and their possible values are defined in the
//                               Server Settings Backend plugin (see the
//                               Configuration section in its README file).
//
// Note that several of the values herein will be passed through the
// SquirrelMail translation engine before being displayed.  If you use
// one of the suggested strings that is already in the Server Settings
// translation file, it will be translated automatically for you (assuming
// translation availability).  If you use custom wording in any of these,
// it will not be translated unless you add your strings to the Server
// Settings translations.  These items will be translated:
//
//    TITLE
//    DESCRIPTION
//    MENU_LINK
//    WIDGET[SECTION]
//    WIDGET[CAPTION]
//    WIDGET[TRAILING_TEXT]
//    WIDGET[COMMENT]
//    WIDGET[POSVALS]
//    WIDGET[YES_TEXT]
//    WIDGET[NO_TEXT]
//TODO: possibly other attributes need to be added - see line 288 in options.php
//
// Here are the strings that are already included in the Server Settings
// translation file (note that this is subject to expand in future plugin
// versions, so please send requests for new strings):
//
//    Whitelist and Blacklist
//    Whitelist/Blacklist
//    Manage the senders whom you have either whitelisted or blacklisted.
//    Whitelisted Senders
//    Blacklisted Senders
//    Whitelist
//    Manage your sender whitelist.
//    Blacklist
//    Manage your sender blacklist.
//    Spam Controls
//    Manage the settings that control anti-spam efforts on the server.
//    Manage the settings that control how your email is protected from spam.
//    Spam Detection
//    Required Score
//    The required score is the level of confidence that a message is spam at which you want the message to actually be tagged as such.
//    1 - Extremely strict
//    3 - Strict
//    5 - Average
//    8 - Loose
//    10 - Extremely loose
//    Real-time Blacklist Scoring
//    The real-time blacklist score is the number of "spam points" that a message is given when it contains any links that are found in real-time blacklist testing.
//    Acceptable Languages
//    When a message is received in one of the selected languages, it is considered more likely not to be spam; messages in all other languages are usually spam
//    All languages are acceptable
//    Western languages, especially English
//    Japanese
//    Korean
//    Cyrillic languages
//    Thai
//    Chinese
//    Spam Report Subject
//    When a message is tagged as spam, how should its subject be changed?  You can set this as blank so the subject is not changed or set it to something such as \"[SPAM]\".
//    Spam Report Format
//    When a message is tagged as spam, should it be preserved as-is in an attachment to the spam report, attached as only a plain text email, or should it be left unaltered?
//    Leave unaltered
//    Attach to spam report
//    Attach as a text message
//    Mail Forwarding
//    Set up other email addresses to which your incoming messages will be forwarded.
//    Forward Incoming Messages To These Addresses
//    (separate multiple addresses with commas)
//    (separate multiple addresses with commas;<br />enter your own address herein to keep copies of messages in this account)   (don't use in 1.5.2+!)
//    (separate multiple addresses with commas; enter your own address herein to keep copies of messages in this account)
//    Autoresponder
//    Set up an auto-reply message for you incoming email.  This can be useful when you are away on vacation.
//    Enable Auto-Reply To Sender
//    Subject
//    Message
//
$server_settings_option_pages = array(

   // =========================================================================
   array(

      // This is an example page that contains a whitelist and blacklist
      // that are each stored in a file in the user's FTP home directory
      // (for use in some unidentified system).  Note that this could be
      // combined with the "extra buttons" functionality in the Spam
      // Buttons plugin - see that plugin's example configuration file
      // for more details.
      //
      // These wigets are configured in an advaced fashion wherein adding
      // a value to one of them removes that value (if it exists) from the
      // other.  This is accomplished by having *two* "STORAGE" arrays for
      // each widget.  Typical option pages won't usually need this kind
      // of complexity; see the "Spam Controls" (SpamAssassin) examples
      // elsewhere in this file for more typical examples.
      //
      'TITLE'       => 'Whitelist and Blacklist',
      'DESCRIPTION' => 'Manage the senders whom you have either whitelisted or blacklisted.',

      // When the user submits changes, should the page just reload?  Use
      // the RETURN_TO_SELF setting to do that; otherwise it will save
      // changes and go back to the main SquirrelMail options page.
      // Note that sorted edit lists like the ones in this example should
      // use a custom SAVE function to resort their values, otherwise the
      // page will reload with the values probably out of order.
      //
      'RETURN_TO_SELF' => 1,

      // You can put a link to this options page on the main menu at the
      // top of the screen and, if desired, then remove the link to this
      // page from the main SquirrelMail options page.
      //
      // 'MENU_LINK' => 'Whitelist/Blacklist',
      // 'NO_OPTION_PAGE' => 1,

      'WIDGETS' => array(

         // ------------------------------------------------
         array(

            // Note that there is no 'SECTION' for either widget on this
            // page because the page title is presumably enough for the
            // user to understand its context.
            //
            'NAME'           => 'whitelist',
            'CAPTION'        => 'Whitelisted Senders',
            'TYPE'           => SMOPT_TYPE_EDIT_LIST,

            // Two different layout schemes are available for "edit
            // lists" - one using a multiple select list and one using
            // an HTML table layout (default is the HTML table "LIST")
            //
            // 'LAYOUT_TYPE'    => SMOPT_EDIT_LIST_LAYOUT_LIST,
            // 'LAYOUT_TYPE'    => SMOPT_EDIT_LIST_LAYOUT_SELECT,

            // You can also control whether or not users can add or remove
            // edit list contents from this options page (perhaps users
            // can only add to the lists using a whitelist button or link
            // when reading a message (see the Spam Buttons plugin))
            //
            // 'USE_ADD_WIDGET' => 0,
            // 'USE_DELETE_WIDGET' => 0,

            // Several sizes are allowable: SMOPT_SIZE_TINY, SMOPT_SIZE_NORMAL,
            // SMOPT_SIZE_SMALL, SMOPT_SIZE_MEDIUM, SMOPT_SIZE_LARGE, and
            // SMOPT_SIZE_HUGE.  Sizes are used for many widget types, but
            // for edit lists, it is only relevant when using the "SELECT"
            // "LAYOUT_TYPE"
            //
            'SIZE'           => SMOPT_SIZE_LARGE,

            'REFRESH'        => SMOPT_REFRESH_NONE,

            // This is a custom-added entry; it is NOT specified in the
            // Server Settings API, but is instead used by the custom
            // save handler specified by this widget (for which the code
            // is at the bottom of this file).  You are free to add as
            // many other custom entries as needed.
            //
            'AFFECTED_SETTINGS' => array('blacklist'),

            // Since this is an edit list, the retrieved value
            // needs to go into "posvals" instead of the usual
            // "initial_value"
            //
            'SETUP_LOCATION' => 'posvals',

            // For the FTP backend, enabling this doesn't make much
            // difference, although see the notes below about it
            //
            // 'MULTIPLE_VALUE_UPDATES_SEPERATELY' => 1,

            // Typically, only one set of storage definitions
            // are needed to save a setting's value.  However,
            // we define two here so that when adding a value
            // to the whitelist, we remove it from the blacklist.
            // The second storage array is abbreviated because it
            // is only used for saving data, but not retrieving it.
            //
            'STORAGE'        => array(

               array(

                  // The following can be used to look up a setting value with
                  // an arbitrary function (provided by you) instead of the
                  // FTP lookup defined below.  It gets this same widget
                  // configuration array as a parameter (feel free to add
                  // any needed extra information your function will need)
                  // and should return the value of this setting
                  //
                  // 'CUSTOM' => 'get_whitelist',

                  // If the value of this setting is a constant value that never
                  // changes, you can specify it as follows.  If you do, anything
                  // defined in the rest of the backend configuration for this
                  // widget will be ignored
                  //
                  // 'VALUE' => array('spammer@sco.com', 'badguy@microsoft.com'),

                  'BACKEND'              => 'ftp',

                  // The FTP hostname is usually the same for all users as in
                  // this example, but it could itself consist of a lookup if
                  // some users are on different FTP servers (same goes for
                  // any of the settings here)
                  //
                  'HOST'                 => array('VALUE' => 'localhost'),

                  // Choose "BINARY" or "ASCII" transfer mode (defaults to
                  // "BINARY" if not given)
                  //
                  'MODE'                 => array('VALUE' => 'BINARY'),

                  'FILE'                 => array('VALUE' => 'whitelist'),

                  // In this example, each line of the file is one entry
                  // and there can be any number of lines/entries (thus the
                  // "MULTIPLE" setting).  Note the use of the "+" sign in
                  // the "PARSE_PATTERN" - it could be "*" instead, but the
                  // use of "+" ignores blank lines.  Since we are matching
                  // the whole line in "PARSE_PATTERN", "PATTERN_GROUP_NUMBER"
                  // is not needed, but in most cases, it'll probably be
                  // "1" anyway, so we'll just leave it here.
                  //
                  'PARSE_PATTERN'        => array('VALUE' => '/^(.+)$/m'),
                  'MULTIPLE'             => 'MULTIPLE',
                  'PATTERN_GROUP_NUMBER' => array('VALUE' => 1),

                  // New values merely added to the end of the file as is
                  //
                  'NEW_SETTING_TEMPLATE' => array('VALUE' => "%1\n"),

                  // Don't throw a fit when the file is not found
                  //
                  'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),

                  // Should an empty file remain when it contains no
                  // values?  In this example, we'll just remove the file.
                  //
                  'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),

                  // You'll most always want to set this to some small
                  // number.  Note that it is NOT a lookup and should be
                  // specified as a simple number.
                  //
                  'MAX_SEQUENTIAL_EMPTY_LINES' => 1,

                  // We'll strip the username of its domain part, assuming
                  // the FTP server wants just the username part.
                  //
//                  'USERNAME'             => array('STRIP_DOMAIN' => 1),
               ),

               // Here is where we build the second save action where
               // we want to delete anything added to the whitelist from
               // the blacklist.  The important entry here is
               // "DELETE_INSTEAD_OF_ADD".  You may want to note that
               // when you have "MULTIPLE_VALUE_UPDATES_SEPERATELY"
               // (see above) enabled for this widget, only the values
               // being added to the whitelist will be checked and
               // removed from the blacklist; otherwise the entire
               // whitelist will be checked against the blacklist no
               // matter what values are being updated.
               //
               array(
                  'BACKEND'               => 'ftp',
                  'HOST'                  => array('VALUE' => 'localhost'),
                  'MODE'                  => array('VALUE' => 'BINARY'),
                  'FILE'                  => array('VALUE' => 'blacklist'),
                  'DELETE_INSTEAD_OF_ADD' => array('VALUE' => 1),
                  'PARSE_PATTERN'         => array('VALUE' => '/^(.+)$/m'),
                  'PATTERN_GROUP_NUMBER'  => array('VALUE' => 1),
                  'MULTIPLE'              => 'MULTIPLE',
                  'NEW_SETTING_TEMPLATE'  => array('VALUE' => "%1\n"),
                  'MAX_SEQUENTIAL_EMPTY_LINES' => 1,
                  'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
                  'DELETE_WHEN_EMPTY'     => array('VALUE' => 1),
//                  'USERNAME'              => array('STRIP_DOMAIN' => 1),
               ),

            ),

            // If you want to define (or add) custom handling to a setting
            // when it is being saved, specify a custom function name here
            // (that function must be provided by you).  It will be passed
            // a SquirrelMail Option object that contains the widget being
            // saved as well as a second argument containing the STORAGE
            // configuration array for the widget as defined herein.
            //
            // If you merely want to massage (sort?) the value that is
            // being saved, you can do that, and then call back to
            // ss_save_option($option, $storage_info) to actually save the
            // setting.  Note the second parameter, which is the STORAGE
            // configuration array for the widget, exactly as you already
            // received as your second function argument as well.
            //
            // Also note that because edit lists require special handling,
            // it is best to, in your custom function, first call
            // ss_save_option(), then manipulate the values as needed,
            // and call back to ss_save_option() AGAIN to re-save.
            //
            // Again note that in most cases, you DO NOT need to specify
            // a "SAVE" handler.  It is only used to add supplemental
            // functionality.  The FTP backend does not sort the values
            // it retrieves, so if sorting is desired, that could be one
            // reason to use a custom save handler.
            //
            // In our example here, if we have enabled "RETURN_TO_SELF"
            // above, getting the lists to repopulate again correctly
            // may require resorting, and when the widgets delete values
            // from each other as in this example, we need some extra
            // handling for that as well.  The code for both of these
            // example handlers is at the bottom of this file.
            //
            // 'SAVE' => 'sort_and_save_edit_list',
            'SAVE' => 'sort_and_save_edit_list_for_cross_updating_settings',
         ),

         // ------------------------------------------------
         array(

            'NAME'           => 'blacklist',
            'CAPTION'        => 'Blacklisted Senders',
            'TYPE'           => SMOPT_TYPE_EDIT_LIST,
            'SIZE'           => SMOPT_SIZE_LARGE,
            'REFRESH'        => SMOPT_REFRESH_NONE,
            'AFFECTED_SETTINGS' => array('whitelist'),
            // 'MULTIPLE_VALUE_UPDATES_SEPERATELY' => 1,
            'SETUP_LOCATION' => 'posvals',
            'SAVE' => 'sort_and_save_edit_list_for_cross_updating_settings',

            'STORAGE'        => array(

               array(
                  'BACKEND'              => 'ftp',
                  'HOST'                 => array('VALUE' => 'localhost'),
                  'MODE'                 => array('VALUE' => 'BINARY'),
                  'FILE'                 => array('VALUE' => 'blacklist'),
                  'PARSE_PATTERN'        => array('VALUE' => '/^(.+)$/m'),
                  'MULTIPLE'             => 'MULTIPLE',
                  'PATTERN_GROUP_NUMBER' => array('VALUE' => 1),
                  'NEW_SETTING_TEMPLATE' => array('VALUE' => "%1\n"),
                  'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),
                  'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
                  'MAX_SEQUENTIAL_EMPTY_LINES' => 1,
//                  'USERNAME'             => array('STRIP_DOMAIN' => 1),
               ),
               array(
                  'BACKEND'               => 'ftp',
                  'HOST'                  => array('VALUE' => 'localhost'),
                  'MODE'                  => array('VALUE' => 'BINARY'),
                  'FILE'                  => array('VALUE' => 'whitelist'),
                  'DELETE_INSTEAD_OF_ADD' => array('VALUE' => 1),
                  'PARSE_PATTERN'         => array('VALUE' => '/^(.+)$/m'),
                  'PATTERN_GROUP_NUMBER'  => array('VALUE' => 1),
                  'MULTIPLE'              => 'MULTIPLE',
                  'NEW_SETTING_TEMPLATE'  => array('VALUE' => "%1\n"),
                  'MAX_SEQUENTIAL_EMPTY_LINES' => 1,
                  'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
                  'DELETE_WHEN_EMPTY'     => array('VALUE' => 1),
//                  'USERNAME'              => array('STRIP_DOMAIN' => 1),
               ),

            ),

         ),
      ),
   ),

   // =========================================================================
   array(

      // This is an example page that contains various anti-spam
      // user settings that are stored in files accessed via a FTP
      // server for use by SpamAssassin.
      //
      'TITLE'       => 'Spam Controls',
      'DESCRIPTION' => 'Manage the settings that control how your email is protected from spam.',
      // 'RETURN_TO_SELF' => 1,
      'WIDGETS' => array(

         // ------------------------------------------------
         array(
            'NAME'           => 'required_score_comment',
            'SECTION'        => 'Spam Detection',
            'CAPTION'        => 'Required Score',
            'CAPTION_WRAP'   => FALSE,
            'COMMENT'        => 'The required score is the level of confidence that a message is spam at which you want the message to actually be tagged as such.',
            'TYPE'           => SMOPT_TYPE_COMMENT,

            // Note that if you retrieve a value from a backend lookup
            // for display in a comment, you need to do the following:
            //
            // 'SETUP_LOCATION' => 'comment',
         ),

         // ------------------------------------------------
         array(
            'NAME'    => 'required_score',
            'SECTION' => 'Spam Detection',

            // The widget above set up our caption already, so don't show one here
            //
            'CAPTION' => '',

            'TYPE'    => SMOPT_TYPE_STRLIST,
            // 'POSVALS' => array(0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10),
            // 'POSVALS' => array(0 => 'Everything is spam', 1 => 'Strict',
            //                    2 => 'Tight', 3 => 'Hard', 4 => 'Decent',
            //                    5 => 'Average', 6 => 'Loose', 7 => 'Sloppy',
            //                    8 => 'Easy to fool', 9 => 'Stupid',
            //                    10 => 'Nothing is spam'),
            'POSVALS' => array(0 => '0', 1 => '1 - Extremely strict',
                               2 => '2', 3 => '3 - Strict', 4 => '4',
                               5 => '5 - Average', 6 => '6', 7 => '7',
                               8 => '8 - Loose', 9 => '9',
                               10 => '10 - Extremely loose'),
            'REFRESH' => SMOPT_REFRESH_NONE,
            'STORAGE' => array(
               'BACKEND'              => 'ftp',
               'HOST'                 => array('VALUE' => 'localhost'),
               'MODE'                 => array('VALUE' => 'BINARY'),

               // You can specify the directory to change to to find the
               // needed file, otherwise you may omit "DIRECTORY" and the
               // default/home FTP directory will be used.  When using
               // "DIRECTORY", specifying a full path is usually best.
               // 
               'DIRECTORY'            => array('VALUE' => '/.spamassassin'),

               'FILE'                 => array('VALUE' => 'user_prefs'),
               'PARSE_PATTERN'        => array('VALUE' => "/^required_score[^\S\n]+(.*)$/m"),
               'PATTERN_GROUP_NUMBER' => array('VALUE' => 1),
               'NEW_SETTING_TEMPLATE' => array('VALUE' => "required_score  %1\n"),
               'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),
               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'MAX_SEQUENTIAL_EMPTY_LINES' => 1,
               'USERNAME'             => array('STRIP_DOMAIN' => 1),
            ),
         ),

         // ------------------------------------------------
         array(
            'NAME'           => 'uribl_sbl_score_comment',
            'SECTION'        => 'Spam Detection',
            'CAPTION'        => 'Real-time Blacklist Scoring',
            'CAPTION_WRAP'   => FALSE,
            'COMMENT'        => 'The real-time blacklist score is the number of "spam points" that a message is given when it contains any links that are found in real-time blacklist testing.',
            'TYPE'           => SMOPT_TYPE_COMMENT,
         ),

         // ------------------------------------------------
         array(
            'NAME'    => 'uribl_sbl_score',
            'SECTION' => 'Spam Detection',
            'CAPTION' => '',
            'TYPE'    => SMOPT_TYPE_STRLIST,
            'POSVALS' => array('0' => 0, '0.5' => 0.5, '1' => 1, '1.5' => 1.5,
                               '2' => 2, '2.5' => 2.5, '3' => 3, '3.5' => 3.5,
                               '4' => 4, '4.5' => 4.5, '5' => 5, '5.5' => 5.5,
                               '6' => 6),
            'REFRESH' => SMOPT_REFRESH_NONE,
            'STORAGE' => array(
               'BACKEND'              => 'ftp',
               'HOST'                 => array('VALUE' => 'localhost'),
               'MODE'                 => array('VALUE' => 'BINARY'),
               'DIRECTORY'            => array('VALUE' => '/.spamassassin'),
               'FILE'                 => array('VALUE' => 'user_prefs'),
               'PARSE_PATTERN'        => array('VALUE' => "/^score[^\S\n]+URIBL_SBL[^\S\n]+(.*)$/m"),
               'PATTERN_GROUP_NUMBER' => array('VALUE' => 1),
               'NEW_SETTING_TEMPLATE' => array('VALUE' => "score URIBL_SBL  %1\n"),
               'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),
               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'MAX_SEQUENTIAL_EMPTY_LINES' => 1,
               'USERNAME'             => array('STRIP_DOMAIN' => 1),
            ),
         ),

         // ------------------------------------------------
         array(
            'NAME'           => 'ok_locales_comment',
            'SECTION'        => 'Spam Detection',
            'CAPTION'        => 'Acceptable Languages',
            'CAPTION_WRAP'   => FALSE,
            'COMMENT'        => 'When a message is received in one of the selected languages, it is considered more likely not to be spam; messages in all other languages are usually spam',
            'TYPE'           => SMOPT_TYPE_COMMENT,
         ),

         // ------------------------------------------------
         array(
            'NAME'    => 'ok_locales',
            'SECTION' => 'Spam Detection',
            'CAPTION' => '',
            'TYPE'    => SMOPT_TYPE_STRLIST_MULTI,
            'REFRESH' => SMOPT_REFRESH_NONE,
            'POSVALS' => array('all' => 'All languages are acceptable',
                               'en' => 'Western languages, especially English',
                               'ja' => 'Japanese',
                               'ko' => 'Korean',
                               'ru' => 'Cyrillic languages',
                               'th' => 'Thai',
                               'zh' => 'Chinese'),
            'STORAGE' => array(
               'BACKEND'              => 'ftp',
               'HOST'                 => array('VALUE' => 'localhost'),
               'MODE'                 => array('VALUE' => 'BINARY'),
               'DIRECTORY'            => array('VALUE' => '/.spamassassin'),
               'FILE'                 => array('VALUE' => 'user_prefs'),
               'PARSE_PATTERN'        => array('VALUE' => "/^ok_locales[^\S\n]+(.*)$/m"),
               'PATTERN_GROUP_NUMBER' => array('VALUE' => 1),
               'NEW_SETTING_TEMPLATE' => array('VALUE' => "ok_locales  %1\n"),
               'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),
               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'MAX_SEQUENTIAL_EMPTY_LINES' => 1,
               'USERNAME'             => array('STRIP_DOMAIN' => 1),

               // Note that this setting contains multiple
               // values separated by whitespace
               //
               'MULTIPLE'       => ' ',
            ),
         ),

         // ------------------------------------------------
         array(
            'NAME'           => 'rewrite_header_subject_comment',
            'SECTION'        => 'Spam Detection',
            'CAPTION'        => 'Spam Report Subject',
            'CAPTION_WRAP'   => FALSE,
            'COMMENT'        => 'When a message is tagged as spam, how should its subject be changed?  You can set this as blank so the subject is not changed or set it to something such as "[SPAM]".',
            'TYPE'           => SMOPT_TYPE_COMMENT,
         ),

         // ------------------------------------------------
         array(
            'NAME'    => 'rewrite_header_subject',
            'SECTION' => 'Spam Detection',
            'CAPTION' => '',
            'TYPE'    => SMOPT_TYPE_STRING,
            'REFRESH' => SMOPT_REFRESH_NONE,
            'STORAGE' => array(
               'BACKEND'              => 'ftp',
               'HOST'                 => array('VALUE' => 'localhost'),
               'MODE'                 => array('VALUE' => 'BINARY'),
               'DIRECTORY'            => array('VALUE' => '/.spamassassin'),
               'FILE'                 => array('VALUE' => 'user_prefs'),
               'PARSE_PATTERN'        => array('VALUE' => "/^rewrite_header[^\S\n]+subject[^\S\n]+(.*)$/m"),
               'PATTERN_GROUP_NUMBER' => array('VALUE' => 1),
               'NEW_SETTING_TEMPLATE' => array('VALUE' => "rewrite_header subject  %1\n"),
               'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),
               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'MAX_SEQUENTIAL_EMPTY_LINES' => 1,
               'USERNAME'             => array('STRIP_DOMAIN' => 1),
            ),
         ),

         // ------------------------------------------------
         array(
            'NAME'           => 'report_safe_comment',
            'SECTION'        => 'Spam Detection',
            'CAPTION'        => 'Spam Report Format',
            'CAPTION_WRAP'   => FALSE,
            'COMMENT'        => 'When a message is tagged as spam, should it be preserved as-is in an attachment to the spam report, attached as only a plain text email, or should it be left unaltered?',
            'TYPE'           => SMOPT_TYPE_COMMENT,
         ),

         // ------------------------------------------------
         array(
            'NAME'    => 'report_safe',
            'SECTION' => 'Spam Detection',
            'CAPTION' => '',
            'TYPE'    => SMOPT_TYPE_STRLIST,
            'POSVALS' => array('0' => 'Leave unaltered',
                               '1' => 'Attach to spam report',
                               '2' => 'Attach as a text message'),
            'REFRESH' => SMOPT_REFRESH_NONE,
            'STORAGE' => array(
               'BACKEND'              => 'ftp',
               'HOST'                 => array('VALUE' => 'localhost'),
               'MODE'                 => array('VALUE' => 'BINARY'),
               'DIRECTORY'            => array('VALUE' => '/.spamassassin'),
               'FILE'                 => array('VALUE' => 'user_prefs'),
               'PARSE_PATTERN'        => array('VALUE' => "/^report_safe[^\S\n]+(.*)$/m"),
               'PATTERN_GROUP_NUMBER' => array('VALUE' => 1),
               'NEW_SETTING_TEMPLATE' => array('VALUE' => "report_safe  %1\n"),
               'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),
               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'MAX_SEQUENTIAL_EMPTY_LINES' => 1,
               'USERNAME'             => array('STRIP_DOMAIN' => 1),
            ),
         ),

         // ------------------------------------------------
         array(
            'NAME'           => 'whitelist',
            'SECTION'        => 'Whitelist and Blacklist',
            'CAPTION'        => 'Whitelisted Senders',
            'TYPE'           => SMOPT_TYPE_EDIT_LIST,
            'SIZE'           => SMOPT_SIZE_LARGE,
            'REFRESH'        => SMOPT_REFRESH_NONE,
            'SETUP_LOCATION' => 'posvals',
            'STORAGE'        => array(
               'BACKEND'              => 'ftp',
               'HOST'                 => array('VALUE' => 'localhost'),
               'MODE'                 => array('VALUE' => 'BINARY'),
               'DIRECTORY'            => array('VALUE' => '/.spamassassin'),
               'FILE'                 => array('VALUE' => 'user_prefs'),
               'PARSE_PATTERN'        => array('VALUE' => "/^whitelist_from[^\S\n]+(.*)$/m"),
               'PATTERN_GROUP_NUMBER' => array('VALUE' => 1),
               'NEW_SETTING_TEMPLATE' => array('VALUE' => "whitelist_from  %1\n"),
               'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),
               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'MAX_SEQUENTIAL_EMPTY_LINES' => 1,
               'MULTIPLE'             => 'MULTIPLE',
               'USERNAME'             => array('STRIP_DOMAIN' => 1),
            ),
         ),

         // ------------------------------------------------
         array(
            'NAME'           => 'blacklist',
            'SECTION'        => 'Whitelist and Blacklist',
            'CAPTION'        => 'Blacklisted Senders',
            'TYPE'           => SMOPT_TYPE_EDIT_LIST,
            'SIZE'           => SMOPT_SIZE_LARGE,
            'REFRESH'        => SMOPT_REFRESH_NONE,
            'SETUP_LOCATION' => 'posvals',
            'STORAGE'        => array(
               'BACKEND'              => 'ftp',
               'HOST'                 => array('VALUE' => 'localhost'),
               'MODE'                 => array('VALUE' => 'BINARY'),
               'DIRECTORY'            => array('VALUE' => '/.spamassassin'),
               'FILE'                 => array('VALUE' => 'user_prefs'),
               'PARSE_PATTERN'        => array('VALUE' => "/^blacklist_from[^\S\n]+(.*)$/m"),
               'PATTERN_GROUP_NUMBER' => array('VALUE' => 1),
               'NEW_SETTING_TEMPLATE' => array('VALUE' => "blacklist_from  %1\n"),
               'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),
               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'MAX_SEQUENTIAL_EMPTY_LINES' => 1,
               'MULTIPLE'             => 'MULTIPLE',
               'USERNAME'             => array('STRIP_DOMAIN' => 1),
            ),
         ),
      ),
   ),

   // =========================================================================
   array(

      // This is an example page that contains a vacation autoresponder
      // which is stored in a series of files on a FTP server.
      //
      'TITLE'          => 'Autoresponder',
      'DESCRIPTION'    => 'Set up an auto-reply message for you incoming email.  This can be useful when you are away on vacation.',
      'RETURN_TO_SELF' => 1,
      'WIDGETS'        => array(

         // ------------------------------------------------
         array(
            'NAME'    => 'enable_autoresponder',
            'CAPTION' => 'Enable Auto-Reply To Sender',
            'TYPE'    => SMOPT_TYPE_BOOLEAN,
            'REFRESH' => SMOPT_REFRESH_NONE,
            'STORAGE' => array(
               'BACKEND'              => 'ftp',
               'HOST'                 => array('VALUE' => 'localhost'),
               'MODE'                 => array('VALUE' => 'BINARY'),
               'FILE'                 => array('VALUE' => 'autoreply_enabled'),
               'PARSE_PATTERN'        => array('VALUE' => '/^(.*)$/s'),
               'NEW_SETTING_TEMPLATE' => array('VALUE' => '%1'),
               'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),
               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'USERNAME'             => array('STRIP_DOMAIN' => 1),
            ),
         ),

         // ------------------------------------------------
         array(
            'NAME'    => 'message_subject',
            'CAPTION' => 'Subject',
            'TYPE'    => SMOPT_TYPE_STRING,
            'REFRESH' => SMOPT_REFRESH_NONE,
            'STORAGE'        => array(
               'BACKEND'              => 'ftp',
               'HOST'                 => array('VALUE' => 'localhost'),
               'MODE'                 => array('VALUE' => 'BINARY'),
               'FILE'                 => array('VALUE' => 'autoreply_subject'),
               'PARSE_PATTERN'        => array('VALUE' => '/^(.*)$/s'),
               'NEW_SETTING_TEMPLATE' => array('VALUE' => '%1'),
               'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),
               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'USERNAME'             => array('STRIP_DOMAIN' => 1),
            ),
         ),

         // ------------------------------------------------
         array(
            'NAME'    => 'message_body',
            'CAPTION' => 'Message',
            'TYPE'    => SMOPT_TYPE_TEXTAREA,
            'REFRESH' => SMOPT_REFRESH_NONE,
            'STORAGE'        => array(
               'BACKEND'              => 'ftp',
               'HOST'                 => array('VALUE' => 'localhost'),
               'MODE'                 => array('VALUE' => 'BINARY'),
               'FILE'                 => array('VALUE' => 'autoreply_message_body'),
               'PARSE_PATTERN'        => array('VALUE' => '/^(.*)$/s'),
               'NEW_SETTING_TEMPLATE' => array('VALUE' => '%1'),
               'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),
               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'USERNAME'             => array('STRIP_DOMAIN' => 1),
            ),
         ),
      ),
   ),

   // =========================================================================
   array(

      // This is an example page that contains a vacation autoresponder
      // which is stored on a FTP server in a format expected by most
      // "vacation" program implementations (.forward file with
      // .vacation.msg file).  However, please note that this example
      // does not initialize the vacation database (usually done by
      // executing "vacation -I"), is very complex and would be
      // complicated to maintain or customize.  A better solution would be
      // to use the "Local User Autoresponder and Mail Forwarder" plugin
      // (which will in the future be changed to use this plugin as its
      // backend).
      //
      'TITLE'          => 'Autoresponder',
      'DESCRIPTION'    => 'Set up an auto-reply message for you incoming email.  This can be useful when you are away on vacation.',
      'RETURN_TO_SELF' => 1,
      'WIDGETS'        => array(

         // ------------------------------------------------
         array(
            'NAME'    => 'enable_autoresponder',
            'CAPTION' => 'Enable Auto-Reply To Sender',
            'TYPE'    => SMOPT_TYPE_BOOLEAN,
            'REFRESH' => SMOPT_REFRESH_NONE,

            // The enabled flag triggers the creation of a .forward
            // file with the vacation syntax in it in the user's
            // (FTP) home directory.  Note that this example will
            // just remove the whole .forward file when the
            // autoresponder is turned off.  Keeping the file in
            // place would require a slightly different regular
            // expression (likely) and maybe the use of
            // "DELETE_KEEP_PATTERN" (not as likely), but this
            // excercise is left to you.
            //
            'STORAGE' => array(
               'BACKEND'              => 'ftp',
               'HOST'                 => array('VALUE' => 'localhost'),
               'MODE'                 => array('VALUE' => 'BINARY'),
               'FILE'                 => array('VALUE' => '.forward'),

               // Note the use of "REPLACEMENT_PATTERN", and especially
               // that the "e" regular expression modifier in
               // "PARSE_PATTERN" allows it to contain PHP code.
               //
               'PARSE_PATTERN'        => array('VALUE' => '/^(.*)$/se'),
               'REPLACEMENT_PATTERN'  => array('VALUE' => '("%1" == "1" ? "\%u, \"|/usr/bin/vacation %u\"" : "")'),
               'NEW_SETTING_TEMPLATE' => array('VALUE' => '\%u, "|/usr/bin/vacation %u"'),

               'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),
               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'USERNAME'             => array('STRIP_DOMAIN' => 1),
            ),
         ),

         // ------------------------------------------------
         array(
            'NAME'    => 'message_subject',
            'CAPTION' => 'Subject',
            'TYPE'    => SMOPT_TYPE_STRING,
            'REFRESH' => SMOPT_REFRESH_NONE,

            // The message subject and the message body are both kept in
            // the same file.  Parsing out or updating/replacing only the
            // subject or message body has to be done carefully.  It may
            // be better to simply offer a single textual edit of the
            // entire .vacation.msg file instead, where the user is held
            // responsible for keeping the subject header formatted
            // correctly at the top of the file, etc.
            //
            'STORAGE'        => array(
               'BACKEND'              => 'ftp',
               'HOST'                 => array('VALUE' => 'localhost'),
               'MODE'                 => array('VALUE' => 'BINARY'),
               'FILE'                 => array('VALUE' => '.vacation.msg'),

               // The "REPLACEMENT_PATTERN" here is not all that
               // necessary, but it does allow more control over
               // the single blank line separating the subject header
               // from the message body.
               //
               'PARSE_PATTERN'        => array('VALUE' => "/^Subject:[^\S\n]*([^\n]*)(\n*)(.*)/is"),
               'PATTERN_GROUP_NUMBER' => array('VALUE' => 1),
               'REPLACEMENT_PATTERN'  => array('VALUE' => "Subject: %1\n\n$3"),
               'NEW_SETTING_TEMPLATE' => array('VALUE' => "Subject: %1\n\n"),

               // This is crucial to making sure the subject
               // is inserted as the first line in the file
               //
               'ADD_TO_TOP'           => array('VALUE' => 1),

               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'USERNAME'             => array('STRIP_DOMAIN' => 1),
            ),
         ),

         // ------------------------------------------------
         array(
            'NAME'    => 'message_body',
            'CAPTION' => 'Message',
            'TYPE'    => SMOPT_TYPE_TEXTAREA,
            'REFRESH' => SMOPT_REFRESH_NONE,
            'STORAGE'        => array(
               'BACKEND'                => 'ftp',
               'HOST'                   => array('VALUE' => 'localhost'),
               'MODE'                   => array('VALUE' => 'BINARY'),
               'FILE'                   => array('VALUE' => '.vacation.msg'),
               'PARSE_PATTERN'          => array('VALUE' => "/^(?(?=Subject:[^\n]*)[^\n]*\n*)(.*)/is"),
               'PATTERN_GROUP_NUMBER'   => array('VALUE' => 1),

               // Working with the message body and preserving the
               // subject in the file requires some non-trivial
               // regular expressions.  The "PARSE_PATTERN" above
               // also uses the same conditional subpattern syntax
               // along with a lookahead assertion.
               //
               'UPDATE_SEARCH_PATTERN'  => array('VALUE' => '/^(?(?=Subject:[^\n]*)([^\n]*\n*)|())(.*)/is'),
               'UPDATE_REPLACE_PATTERN' => array('VALUE' => "\${1}%1"),

               'NEW_SETTING_TEMPLATE'   => array('VALUE' => '%1'),
               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'USERNAME'             => array('STRIP_DOMAIN' => 1),
            ),
         ),
      ),
   ),

   // =========================================================================
   array(

      // This is an example page that allows a user to configure mail
      // forwarding, the target addresses stored in a .forward file in
      // the user's home (FTP) directory.  The syntax of the .forward
      // file usually allows adding an address on each line, and that
      // may be the easiest/best way to manage this feature.  However,
      // you can find an example for that at the top of this file
      // (see the whitelist or blacklist settings), so in this case,
      // the addresses will be added to the file as a comma-separated
      // list, which should also be allowable for most mail systems.
      //
      // Also note that this is not compatible with the autoresponder
      // example above that also modifies the .forward file.  It is
      // entirely possible to change both this and the autoresponder
      // examples to work together, but that would involve more complexity
      // and merging the two into a single page might be the best solution.
      // That defeats the purpose of having clear examples in this file,
      // and ultimately, the "Local User Autoresponder and Mail Forwarder"
      // plugin is what should be used in most cases.
      //
      'TITLE'          => 'Mail Forwarding',
      'DESCRIPTION'    => 'Set up other email addresses to which your incoming messages will be forwarded.',
      'WIDGETS'        => array(

         // ------------------------------------------------
         array(
            'NAME'    => 'mail_forwarding_addresses',
            'CAPTION' => 'Forward Incoming Messages To These Addresses',
            // 'TRAILING_TEXT'    => '(separate multiple addresses with commas)',
            // 'TRAILING_TEXT'    => '(separate multiple addresses with commas;<br />enter your own address herein to keep copies of messages in this account)',  // don't use in 1.5.2+!
            'TRAILING_TEXT'    => '(separate multiple addresses with commas; enter your own address herein to keep copies of messages in this account)',
            'TYPE'    => SMOPT_TYPE_TEXTAREA,
            'REFRESH' => SMOPT_REFRESH_NONE,
            'STORAGE' => array(
               'BACKEND'              => 'ftp',
               'HOST'                 => array('VALUE' => 'localhost'),
               'MODE'                 => array('VALUE' => 'BINARY'),
               'FILE'                 => array('VALUE' => '.forward'),
               'PARSE_PATTERN'        => array('VALUE' => '/^(.*)$/s'),
               'NEW_SETTING_TEMPLATE' => array('VALUE' => '%1'),
               'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),
               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'USERNAME'             => array('STRIP_DOMAIN' => 1),
            ),

            // we want to make sure there are no newlines in the
            // saved text (replace with commas), so we'll do that
            // in a custom save handler (see the bottom of this
            // file)
            //
            'SAVE'    => 'save_email_addresses',
         ),
      ),
   ),

   // =========================================================================
   array(

      // This is an example page that contains widgets that don't
      // make any sense but can be used as examples for other
      // uses.
      //
      'TITLE'       => 'Miscellaneous Server Settings',
      'DESCRIPTION' => 'Manage some settings on the server.',
      'WIDGETS' => array(

         // ------------------------------------------------
         array(

            // This widget is an example of how to set up a multiple-
            // select folder list widget.
            //
            'NAME'    => 'favorite_folders',
            'SECTION' => 'Favorites',
            'CAPTION' => 'Favorite Folders',
            'REFRESH' => SMOPT_REFRESH_NONE,
            'TYPE'    => SMOPT_TYPE_FLDRLIST_MULTI,

            // Here is how you get a folder listing.  Note that you
            // are free to add some additional options as needed
            // (the example below adds "No Selection" at the top).
            //
            // 'POSVALS' => array('none'   => 'No Selection',
            //                    'ignore' => sqimap_mailbox_list($imapConnection)),
            //
            'POSVALS' => array(sqimap_mailbox_list($imapConnection)),

            'STORAGE' => array(
               'BACKEND'              => 'ftp',
               'HOST'                 => array('VALUE' => 'localhost'),
               'MODE'                 => array('VALUE' => 'BINARY'),
               'FILE'                 => array('VALUE' => 'favorite_folders'),
               'PARSE_PATTERN'        => array('VALUE' => '/^(.*)$/s'),
               'NEW_SETTING_TEMPLATE' => array('VALUE' => '%1'),
               'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),
               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'USERNAME'             => array('STRIP_DOMAIN' => 1),
               'MULTIPLE'       => ',',  // CSV format for multiple values
            ),
         ),

         // ------------------------------------------------
         array(

            // This widget is an example of how to set up a single-
            // select folder list widget.
            //
            'NAME'    => 'favorite_folder',
            'SECTION' => 'Favorites',
            'CAPTION' => 'Favorite Folder',
            'REFRESH' => SMOPT_REFRESH_NONE,
            'TYPE'    => SMOPT_TYPE_FLDRLIST,
            'POSVALS' => array(''       => 'No Selection',
                               'ignore' => sqimap_mailbox_list($imapConnection)),
            'STORAGE' => array(
               'BACKEND'              => 'ftp',
               'HOST'                 => array('VALUE' => 'localhost'),
               'MODE'                 => array('VALUE' => 'BINARY'),
               'FILE'                 => array('VALUE' => 'favorite_folder'),
               'PARSE_PATTERN'        => array('VALUE' => '/^(.*)$/s'),
               'NEW_SETTING_TEMPLATE' => array('VALUE' => '%1'),
               'DELETE_WHEN_EMPTY'    => array('VALUE' => 1),
               'TREAT_AS_EMPTY_WHEN_NOT_FOUND' => array('VALUE' => 1),
               'USERNAME'             => array('STRIP_DOMAIN' => 1),
            ),
         ),
      ),
   ),
);



/*
         // ------------------------------------------------
         array(
            'NAME'           => 'separator_1',
            'SECTION'        => 'Section Name',
            'CAPTION'        => '',
            'COMMENT'        => '',
            'TYPE'           => SMOPT_TYPE_COMMENT,
         ),

*/



/**
  * Compress multi-line input into one line (with
  * newlines replaced with commas) when saving a
  * list of email addresses.
  *
  * Actual saving of the value is passed along
  * to the SquirrelMail core.
  *
  * This also performs email format validation,
  * ignoring malformed addresses (you could signal
  * an error if desired as noted in the code).
  *
  * @param object $option       The option class
  *                             object representing
  *                             the data being saved.
  * @param array  $storage_info The configuration
  *                             array containing the
  *                             information that tells
  *                             the Server Settings
  *                             Backend plugin how to
  *                             save the data.
  *
  */
function save_email_addresses($option, $storage_info)
{

   // 1) sanitize user input (note this may be redundant because
   //    the email format validation will remove bad values too)
   // 2) turn newlines into commas
   // 3) remove all whitespace as well as leading/trailing commas
   // 4) remove duplicate commas
   // 5) add single space after commas
   //
   $option->new_value = preg_replace(array("/(\s*(\r\n|\r|\n)\s*)+/",
                                           '/(\s+|^,+|,+$)/',
                                           '/,+/',
                                           '/,/'),
                                     array(',', '', ',', ', '),
                                     htmlspecialchars($option->new_value));
                                     //$option->new_value);


   // validate that email addresses are either in local
   // format (no domain) or are full email adddresses
   //
   global $Email_RegExp_Match;
   include_once(SM_PATH . 'functions/url_parser.php');
   $account_only_regexp = substr($Email_RegExp_Match, 0, strpos($Email_RegExp_Match, '@'));
   $option->new_value = explode(', ', $option->new_value);
   $valid_address_list = '';
   foreach ($option->new_value as $address)
      if (!eregi('^' . $Email_RegExp_Match . '$', $address)
       && !eregi('^' . $account_only_regexp . '$', $address))
      {
         // could track errors here if desired
      }
      else
         $valid_address_list .= $address . ', ';
   $option->new_value = substr($valid_address_list, 0, -2);


   // save per normal
   //
   ss_save_option($option, $storage_info);

}



/**
  * Make sure an edit list is sorted when
  * saving so it is displayed in a sensible
  * order.
  *
  * Actual saving of the value is passed along
  * to the SquirrelMail core.
  *
  * @param object $option       The option class
  *                             object representing
  *                             the data being saved.
  * @param array  $storage_info The configuration
  *                             array containing the
  *                             information that tells
  *                             the Server Settings
  *                             Backend plugin how to
  *                             save the data.
  *
  */
function sort_and_save_edit_list($option, $storage_info)
{

   // Determine if we need to save all option values at once
   // (this only works in SquirrelMail 1.4.14+ or 1.5.2+)
   //
   if (empty($option->raw_option_array)
    || empty($option->raw_option_array['multiple_value_updates_seperately']))
      $save_multiple_values_all_at_once = TRUE;
   else
      $save_multiple_values_all_at_once = FALSE;



   // process and save the option
   //
   // ss_save_option($option, $storage_info);
   ss_save_option($option, $storage_info, $save_multiple_values_all_at_once);



   // sort our option's finalized value list
   //
   natcasesort($option->possible_values);



   // next save below can be tricked to just re-save
   // the possible_values list - no need to process
   // ads and removals
   //
   $orig_use_delete_widget = $option->use_delete_widget;
   $orig_use_add_widget = $option->use_add_widget;
   $option->use_delete_widget = 0;
   $option->use_add_widget = 0;



   // we must delete all the values first in the
   // case of the FTP backend because it won't
   // reorder them when re-saving if they haven't
   // otherwise changed at all
   //
   $has_file_based_backend = FALSE;
   if (!empty($storage_info[0]) && !empty($storage_info[0]['BACKEND']))
   {
      foreach ($storage_info as $storage)
         if (!empty($storage['BACKEND'])
          && ($storage['BACKEND'] == 'ftp' || $storage['BACKEND'] == 'local_file'))
            $has_file_based_backend = TRUE;
   }
   else if (!empty($storage_info['BACKEND']) 
         && ($storage_info['BACKEND'] == 'ftp' || $storage_info['BACKEND'] == 'local_file'))
      $has_file_based_backend = TRUE;
   if ($has_file_based_backend)
   {
      $orig_possible_values = $option->possible_values;
      $option->possible_values = array();
      ss_save_option($option, $storage_info, TRUE);
      $option->possible_values = $orig_possible_values;
   }



   // now, re-save the sorted option
   //
   // this is ONLY useful for backends that
   // cannot easily sort the values they are
   // retrieving
   //
   // if the backend is SQL, then removing
   // this and adding an ORDER BY clause to
   // its QUERY setting is recommended (in
   // which case this entire custom save
   // handler is NOT NEEDED at all unless
   // RETURN_TO_SELF is being used on the
   // option page or other "affected" setting
   // values need to be adjusted per below)
   //
   ss_save_option($option, $storage_info, TRUE);



   // put widgets back the way they were
   //
   $option->use_delete_widget = $orig_use_delete_widget;
   $option->use_add_widget = $orig_use_add_widget;

}



/**
  * Make sure an edit list is sorted when saving so
  * it is displayed in a sensible order.  This function
  * also compensates for option widgets that update other
  * options on the same page as itself.
  *
  * Actual saving of the value is passed along
  * to the SquirrelMail core.
  *
  * @param object $option       The option class
  *                             object representing
  *                             the data being saved.
  * @param array  $storage_info The configuration
  *                             array containing the
  *                             information that tells
  *                             the Server Settings
  *                             Backend plugin how to
  *                             save the data.
  *
  */
function sort_and_save_edit_list_for_cross_updating_settings($option, $storage_info)
{

   // Because we have settings that are also changing one another's
   // values upon update (such as delete from one when adding to
   // another), NO MATTER what the value of MULTIPLE_VALUE_UPDATES_SEPERATELY
   // is, we have to use FALSE below, otherwise the secondary
   // deletion will be subsequently re-added and will appear to
   // have no effect!
   //
   // process and save the option
   //
   // ss_save_option($option, $storage_info);
   ss_save_option($option, $storage_info, FALSE);



   // sort our option's finalized value list
   //
   natcasesort($option->possible_values);



   // next save below can be tricked to just re-save
   // the possible_values list - no need to process
   // ads and removals
   //
   $orig_use_delete_widget = $option->use_delete_widget;
   $orig_use_add_widget = $option->use_add_widget;
   $option->use_delete_widget = 0;
   $option->use_add_widget = 0;



   // we must delete all the values first in the
   // case of the FTP backend because it won't
   // reorder them when re-saving if they haven't
   // otherwise changed at all
   //
   $has_file_based_backend = FALSE;
   if (!empty($storage_info[0]) && !empty($storage_info[0]['BACKEND']))
   {
      foreach ($storage_info as $storage)
         if (!empty($storage['BACKEND'])
          && ($storage['BACKEND'] == 'ftp' || $storage['BACKEND'] == 'local_file'))
            $has_file_based_backend = TRUE;
   }
   else if (!empty($storage_info['BACKEND']) 
         && ($storage_info['BACKEND'] == 'ftp' || $storage_info['BACKEND'] == 'local_file'))
      $has_file_based_backend = TRUE;
   if ($has_file_based_backend)
   {
      $orig_possible_values = $option->possible_values;
      $option->possible_values = array();
      ss_save_option($option, $storage_info, TRUE);
      $option->possible_values = $orig_possible_values;
   }



   // now, re-save the sorted option
   //
   // this is ONLY useful for backends that
   // cannot easily sort the values they are
   // retrieving
   //
   // if the backend is SQL, then removing
   // this and adding an ORDER BY clause to
   // its QUERY setting is recommended (in
   // which case this entire custom save
   // handler is NOT NEEDED at all unless
   // RETURN_TO_SELF is being used on the
   // option page or other "affected" setting
   // values need to be adjusted per below)
   //
   ss_save_option($option, $storage_info, TRUE);



   // put widgets back the way they were
   //
   $option->use_delete_widget = $orig_use_delete_widget;
   $option->use_add_widget = $orig_use_add_widget;



   // Because we need to re-update another setting's values (that
   // is located on the same options page as this setting) because
   // we changed it in our own save routine, the name of that
   // setting should be specified in *this* option's configuration
   // array as an array entry in an array called
   // "affected_settings".  For example:
   //
   // 'name'           => 'whitelist',
   // 'caption'        => 'Whitelisted Senders',
   // 'type'           => SMOPT_TYPE_EDIT_LIST,
   // 'affected_settings' => array('blacklist'),
   // 'size'           => SMOPT_SIZE_LARGE,
   // 'refresh'        => SMOPT_REFRESH_NONE,
   //
   if (!empty($option->raw_option_array['affected_settings'])
    && is_array($option->raw_option_array['affected_settings']))
   {
      global $optpage_data;
      foreach ($option->raw_option_array['affected_settings'] as $affected_setting)
      {

         // iterate through SquirrelMail's internal option store
         // looking for the affected setting, and update if found
         //
         foreach ($optpage_data['options'] as $option_group)
            foreach ($option_group['options'] as $other_option)
               if ($other_option->name == $affected_setting)
               {

                  // get real values list from backend
                  //
                  $other_setting_values = retrieve_server_setting($other_option->raw_option_array['storage']);

                  // replace the "possible_values" with correct ones
                  //
                  $other_option->possible_values = $other_setting_values;
                  break 2;
               }
      }
   }

}



