<?php

/**
  * SquirrelMail Restrict Senders Plugin
  * Copyright (c) 2003-2011 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage restrict_senders
  *
  */


/**
  * Initialize this plugin (load config values)
  *
  * @return boolean FALSE if no configuration file could be loaded, TRUE otherwise
  *
  */
function restrict_senders_init()
{

   return load_config('restrict_senders',
                      array('../../config/config_restrict_senders.php', 'data/config.php'),
                      TRUE, TRUE);

}



/**
  * Returns a list of preference names that should not
  * be controllable by the end user
  *
  * @return string A comma-delimited list of preference
  *                names that the end-user should not
  *                have control over.
  *
  */
function rs_sensitive_preference_names_do()
{

   return 'USER_IS_POSSIBLE_SPAMMER, mass_email_send_times, todays_email_count, ';

}



/**
  * Validate that this plugin is configured correctly
  *
  * @return boolean Whether or not there was a
  *                 configuration error for this plugin.
  *
  */
function restrict_senders_configtest_do()
{

   // make sure base config is available
   //
   if (!restrict_senders_init())
   {
      do_err('Restrict Senders plugin is missing its main configuration file', FALSE);
      return TRUE;
   }


   // make sure the sender rules configuration is in place if necessary
   //
   global $restrictDomains;
   if ($restrictDomains)
   {

      $parse_result = parseSenderRules('', TRUE);
      if (is_array($parse_result))
      {
         do_err($parse_result[0], FALSE);
         return TRUE;
      }

   }

   // only need to do this pre-1.5.2, as 1.5.2 will make this
   // check for us automatically
   //
   if (!check_sm_version(1, 5, 2))
   {

      // try to find Compatibility, and then that it is v2.0.10+
      //
      if (function_exists('check_plugin_version')
       && check_plugin_version('compatibility', 2, 0, 10, TRUE))
         return FALSE;


      // something went wrong
      //
      do_err('Restrict Senders plugin requires the Compatibility plugin version 2.0.10+', FALSE);
      return TRUE;

   }

   return FALSE;

}



/**
  * Show output below SM page header, for displaying errors in 1.5.2+
  *
  * Output should already be constructed (by using templates), sanitized 
  * and ready to go, stored in global $rs_output variable.
  *
  */
function restrict_senders_error_output()
{
   global $rs_output;
   echo $rs_output;
}



/**
  * Displays any errors that blocked email from
  * sending back on top of compose screen
  *
  */
function display_error_on_compose_screen_do()
{

   global $username, $domain, $PHP_SELF, 
          $send_to, $send_to_cc, $send_to_bcc, $body,
          $use_signature;


   // only want to do this on the compose page
   //
   if (stristr($PHP_SELF, '/src/compose.php'))
   {

      // error when user has been banned from sending too many mass mails
      //
      global $banned_possible_spammer, $color, $account_disabled_message;
      if (sqgetGlobalVar('banned_possible_spammer', $banned_possible_spammer, SQ_GET) 
       && $banned_possible_spammer)
      {

         // don't need to add signature again
         //
         $use_signature = FALSE;


         // TODO: not sure if this will work with all
         // PHP versions, especially 4.3.x....
         //
         // Need to avoid endless loops (plain_error_message()
         // ends up calling to displayPageHeader(), which is
         // where the current hook call is located, so this 
         // code gets called recursively if we don't stop it
         // here...
         //
         restrict_senders_init();
         global $countMe;

         if (!$countMe)
         {
            $countMe = 1;

            sq_change_text_domain('restrict_senders');

            if (!empty($account_disabled_message))
               $msg = _($account_disabled_message);
            else
               $msg = _("We are sorry, but your account has been temporarily disabled due to suspicious activity.  If you feel we have done so by mistake, please contact us immediately to have your account reactivated.");

            sq_change_text_domain('squirrelmail');

            if (check_sm_version(1, 5, 2))
            {
               global $rs_output, $squirrelmail_plugin_hooks;
               $rs_output = error_box($msg, NULL, TRUE);
               $squirrelmail_plugin_hooks['compose_header_bottom']['restrict_senders'] = 'restrict_senders_error_output';
               $squirrelmail_plugin_hooks['page_header_bottom']['restrict_senders'] = 'restrict_senders_error_output';
            }
            else
               plain_error_message($msg, $color);

            return;
         }

      }



      // error when message had too many recipients 
      //
      global $restrict_senders_error_too_many_recipients, $color;
      if (sqgetGlobalVar('restrict_senders_error_too_many_recipients', $restrict_senders_error_too_many_recipients, SQ_GET)
       && $restrict_senders_error_too_many_recipients)
      {

         // don't need to add signature again
         //
         $use_signature = FALSE;


         // TODO: not sure if this will work with all
         // PHP versions, especially 4.3.x....
         //
         // Need to avoid endless loops (plain_error_message()
         // ends up calling to displayPageHeader(), which is
         // where the current hook call is located, so this 
         // code gets called recursively if we don't stop it
         // here...
         //
         restrict_senders_init();
         global $countMe,
                $restrictNumberOfRecipients, $vlogin_restrictNumberOfRecipients;
         if (!empty($vlogin_restrictNumberOfRecipients)) 
            $restrictNumberOfRecipients = $vlogin_restrictNumberOfRecipients;

         if (!$countMe)
         {
            $countMe = 1;
            sq_change_text_domain('restrict_senders');

            $msg = sprintf(_("Too many outgoing recipients. Please limit number of recipients to %d."), $restrictNumberOfRecipients);
            if (check_sm_version(1, 5, 2))
            {
               global $rs_output, $squirrelmail_plugin_hooks;
               $rs_output = error_box($msg, NULL, TRUE);
               $squirrelmail_plugin_hooks['compose_header_bottom']['restrict_senders'] = 'restrict_senders_error_output';
               $squirrelmail_plugin_hooks['page_header_bottom']['restrict_senders'] = 'restrict_senders_error_output';
            }
            else
               plain_error_message($msg, $color);

            sq_change_text_domain('squirrelmail');
            return;
         }

      }



      // error when too many emails sent in one day
      //
      global $restrict_senders_error_too_many_emails_today, $color;
      if (sqgetGlobalVar('restrict_senders_error_too_many_emails_today', $restrict_senders_error_too_many_emails_today, SQ_GET)
       && $restrict_senders_error_too_many_emails_today)
      {

         // don't need to add signature again
         //
         $use_signature = FALSE;


         // TODO: not sure if this will work with all
         // PHP versions, especially 4.3.x....
         //
         // Need to avoid endless loops (plain_error_message()
         // ends up calling to displayPageHeader(), which is
         // where the current hook call is located, so this 
         // code gets called recursively if we don't stop it
         // here...
         //
         restrict_senders_init();
         global $countMe, $restrictNumberOfEmailsPerDay, 
                $vlogin_restrictNumberOfEmailsPerDay;
         if (!empty($vlogin_restrictNumberOfEmailsPerDay)) 
            $restrictNumberOfEmailsPerDay = $vlogin_restrictNumberOfEmailsPerDay;

         if (!$countMe)
         {
            $countMe = 1;
            sq_change_text_domain('restrict_senders');

            $msg = sprintf(_("Daily email limit has been reached. Please limit number of emails per day to %d."), $restrictNumberOfEmailsPerDay);
            if (check_sm_version(1, 5, 2))
            {
               global $rs_output, $squirrelmail_plugin_hooks;
               $rs_output = error_box($msg, NULL, TRUE);
               $squirrelmail_plugin_hooks['compose_header_bottom']['restrict_senders'] = 'restrict_senders_error_output';
               $squirrelmail_plugin_hooks['page_header_bottom']['restrict_senders'] = 'restrict_senders_error_output';
            }
            else
               plain_error_message($msg, $color);

            sq_change_text_domain('squirrelmail');
            return;
         }

      }



      // error when all To: recipients are disallowed
      //
      global $restrict_senders_error_no_to_recipients, $color;
      if (sqgetGlobalVar('restrict_senders_error_no_to_recipients', $restrict_senders_error_no_to_recipients, SQ_GET)
       && $restrict_senders_error_no_to_recipients)
      {

         // don't need to add signature again
         //
         $use_signature = FALSE;


         // TODO: not sure if this will work with all
         // PHP versions, especially 4.3.x....
         //
         // Need to avoid endless loops (plain_error_message()
         // ends up calling to displayPageHeader(), which is
         // where the current hook call is located, so this 
         // code gets called recursively if we don't stop it
         // here...
         //
         global $countMe;
         if (!$countMe)
         {
            $countMe = 1;
            sq_change_text_domain('restrict_senders');

            $msg = _("All recipients are illegal destinations.");
            if (check_sm_version(1, 5, 2))
            {
               global $rs_output, $squirrelmail_plugin_hooks;
               $rs_output = error_box($msg, NULL, TRUE);
               $squirrelmail_plugin_hooks['compose_header_bottom']['restrict_senders'] = 'restrict_senders_error_output';
               $squirrelmail_plugin_hooks['page_header_bottom']['restrict_senders'] = 'restrict_senders_error_output';
            }
            else
               plain_error_message($msg, $color);

            sq_change_text_domain('squirrelmail');
            return;
         }

      }

   }

}



/**
  * Utility function that removes an array 
  * element (by numeric index) and renumbers
  * the array
  *
  */
function unset_and_renumber_array($theArray, $index)
{

   $size = sizeof($theArray);

   for ($j = $index; $j < $size - 1; $j++)
      $theArray[$j] = $theArray[$j+1];

   unset($theArray[$size - 1]);

   return $theArray;

}



/**
  * Checks send restrictions when sending an email.
  *
  */
function check_restrict_senders_do(&$argv)
{

   global $restrictDomains, $restrictNumberOfRecipients, 
          $vlogin_restrictNumberOfRecipients, $restrictNumberOfEmailsPerDay,
          $username, $domain, $blockedAddresses,
          $vlogin_restrictNumberOfEmailsPerDay, $data_dir,
          $restrictSubsequentMassEmails, $vlogin_restrictSubsequentMassEmails,
          $restrictSubsequentMassEmailsWhitelist, $isMassMailing,
          $vlogin_restrictSubsequentMassEmailsWhitelist, $rs_debug,
          $whitelisted_addresses, $rs_delimiter, $log_too_many_recipients,
          $log_too_many_today, $log_too_many_subsequent_mass_emails,
          $log_no_recipients, $rs_clear_prefs_cache_on_send, $draft;


   // don't restrict saving of drafts
   //
   if (!empty($draft)) return;


   if (check_sm_version(1, 5, 2))
      $message = &$argv;
   else
      $message = &$argv[1];


   // mass-mailing enabled users get a pass
   //
   if ($isMassMailing) return;
   //if ($isMassMailing) return $message;
   // (either of these lines should work)


   restrict_senders_init();


   // vlogin overrides
   //
   if (!empty($vlogin_restrictNumberOfRecipients)) 
      $restrictNumberOfRecipients = $vlogin_restrictNumberOfRecipients;
   if (!empty($vlogin_restrictNumberOfEmailsPerDay)) 
      $restrictNumberOfEmailsPerDay = $vlogin_restrictNumberOfEmailsPerDay;
   if (!empty($vlogin_restrictSubsequentMassEmails)) 
      $restrictSubsequentMassEmails = $vlogin_restrictSubsequentMassEmails;
   if (!empty($vlogin_restrictSubsequentMassEmailsWhitelist)) 
      $restrictSubsequentMassEmailsWhitelist = $vlogin_restrictSubsequentMassEmailsWhitelist;


   $now = time();


   // clear preferences cache to defeat spammers logged
   // in from more than one location/browser at once
   //
   if ($rs_clear_prefs_cache_on_send)
   {
      global $prefs_cache, $prefs_are_cached;
      $prefs_cache = FALSE;
      $prefs_are_cached = FALSE;
      sqsession_unregister('prefs_are_cached');
      sqsession_unregister('prefs_cache');
   }


   // grab TO, CC, and BCC fields from message
   //
   $toDomains = array();
   $ccDomains = array();
   $bccDomains = array();
   $toMailboxes = array();
   $ccMailboxes = array();
   $bccMailboxes = array();

   // need to keep track of index values (so don't use
   // foreach) so we make sure to block the correct 
   // ones later on
   //
   for ($i = 0; $i < sizeof($message->rfc822_header->to); $i++)
   {
      $toDomains[$i] = $message->rfc822_header->to[$i]->host;
      $toMailboxes[$i] = $message->rfc822_header->to[$i]->mailbox;
   }
   for ($i = 0; $i < sizeof($message->rfc822_header->cc); $i++)
   {
      $ccDomains[$i] = $message->rfc822_header->cc[$i]->host;
      $ccMailboxes[$i] = $message->rfc822_header->cc[$i]->mailbox;
   }
   for ($i = 0; $i < sizeof($message->rfc822_header->bcc); $i++)
   {
      $bccDomains[$i] = $message->rfc822_header->bcc[$i]->host;
      $bccMailboxes[$i] = $message->rfc822_header->bcc[$i]->mailbox;
   }



   // watch for abuse; block send if user is blasting mass mails 
   // too fast unless user is on whitelist
   //
   if ($restrictSubsequentMassEmails 
    && !in_array($username, $restrictSubsequentMassEmailsWhitelist))
   {

      // if already blocked, skip this stuff...
      // 
      if (!getPref($data_dir, $username, 'USER_IS_POSSIBLE_SPAMMER', 0))
      {

         list($mass_email_recips, $number_mass_mails, $number_minutes) 
            = explode(':', $restrictSubsequentMassEmails);


         // don't even bother if not a "mass email"
         //
         if ($mass_email_recips <= sizeof($toDomains) + sizeof($ccDomains) + sizeof($bccDomains))
         {

            // get and parse previous send stats and add current time to list
            //
            $sendTimes = getPref($data_dir, $username, 'mass_email_send_times', '');
            if ($sendTimes)
               $sendTimes = explode(':', $sendTimes);
            else
               $sendTimes = array();
            $newTimes = array();
            foreach ($sendTimes as $time)
               if ($time >= $now - ($number_minutes * 60))
                  $newTimes[] = $time;
            $newTimes[] = $now;
            setPref($data_dir, $username, 'mass_email_send_times', implode(':', $newTimes));
   

            // now check for and ban abusive users
            //
            if (sizeof($newTimes) >= $number_mass_mails)
            {

               global $send_to, $send_to_cc, $send_to_bcc, $subject, $body,
                      $report_abuse_banned_too_many_subsequent_mass_mails;


               setPref($data_dir, $username, 'USER_IS_POSSIBLE_SPAMMER', 1);
               $sentDates = '';
               foreach ($newTimes as $time)
                  $sentDates .= date('H:i:s Y-m-d', $time) . "\n";


               $replyToList = array();
               $fromList = array();
               $delimiter = '@';
               $address_delim = ', ';

               foreach ($message->rfc822_header->reply_to as $sender)
                  $replyToList[] = $sender->mailbox . $delimiter . $sender->host;
               foreach ($message->rfc822_header->from as $sender)
                  $fromList[] = $sender->mailbox . $delimiter . $sender->host;


//FIXME: could move the logging and sending of alert down below after we let the whitelist-only messages go through...??  but maybe we want to know anyway??
               // log this event - note that to use this,
               // you'll need to have the Squirrel Logger plugin
               // installed and activated and you'll have to add
               // a "RESTRICT_SENDERS" event type to its configuration
               //
               if ($log_too_many_subsequent_mass_emails && is_plugin_enabled('squirrel_logger'))
               {
                  include_once(SM_PATH . 'plugins/squirrel_logger/functions.php');
                  sl_logit('RESTRICT_SENDERS', sprintf('%d mass emails of %d or more recipients in the last %d minutes have been sent; BANNING this user.  To un-ban this user, remove the USER_IS_POSSIBLE_SPAMMER setting from their preferences.', sizeof($newTimes), $mass_email_recips, $number_minutes));
               }


               // send alert to admin
               //
               if (empty($report_abuse_banned_too_many_subsequent_mass_mails))
                  //$message_details = "TO: ###TO###\nCC: ###CC###\nBCC: ###BCC###\nFROM: ###FROM###\nREPLY-TO: ###REPLY_TO###\nSUBJECT: ###SUBJECT###\nBODY: ###BODY###";
                  $message_details = "TO: ###TO###\nCC: ###CC###\nBCC: ###BCC###\nFROM: ###FROM###\nREPLY-TO: ###REPLY_TO###\nSUBJECT: ###SUBJECT###\n";
               else
                  $message_details = $report_abuse_banned_too_many_subsequent_mass_mails;

               $message_details = str_replace(array('###USERNAME###', '###DOMAIN###', 
                                                    '###TO###', '###CC###',
                                                    '###BCC###', '###BODY###',
                                                    '###FROM###', '###REPLY_TO###',
                                                    '###SUBJECT###'),
                                              array($username, $domain, $send_to,
                                                    $send_to_cc, $send_to_bcc, $body,
                                                    implode($address_delim, $fromList),
                                                    implode($address_delim, $replyToList),
                                                    $subject),
                                              $message_details);

               sq_change_text_domain('restrict_senders');
               rs_report_abuse(sprintf(_("NOTICE: User \"%s\" (domain \"%s\") has sent %d mass emails of %d or more recipients in the last %d minutes.\n\nTimes:\n%s\n\n%s has been BANNED.\n\nTo un-ban this user, remove the USER_IS_POSSIBLE_SPAMMER setting from their preferences.\n\n%s"), $username, $domain, sizeof($newTimes), $mass_email_recips, $number_minutes, $sentDates, $username, $message_details), sprintf(_(" --- BANNED - %s"), $username));
               sq_change_text_domain('squirrelmail');

            }

         }

      }


      // did we catch someone?
      //
      if (getPref($data_dir, $username, 'USER_IS_POSSIBLE_SPAMMER', 0))
      {

         // let this mail through if it is only being sent to 
         // whitelisted address(es)
         //
         if (only_sent_to_whitelist($toDomains, $ccDomains, $bccDomains, 
                                    $toMailboxes, $ccMailboxes, $bccMailboxes,
                                    $whitelisted_addresses, $rs_delimiter))
            return;


         // redirect user w/out sending the mail
         //
         global $color, $send_to, $send_to_cc, $send_to_bcc, $subject, $body,
                $variable_sent_folder, $PHP_SELF;
         sqgetGlobalVar('variable_sent_folder', $variable_sent_folder);

         if (strpos($PHP_SELF, '?') === FALSE)
            $qs = '?';
         else
            $qs = '&';

//TODO: why not just put error in session before redirect?  (and all the other crap in the query string... just put the "query string" in session so no size limit and no one can change it in their browser).... at least put a flag in the session that there was an error message?  I guess it's not very important, because "tricking" this only stops the error message from being displayed, but does not get around the fact that the message was not sent
         header('Location: ' . $PHP_SELF . $qs . 'send_to=' . urlencode($send_to) 
            . '&send_to_cc=' . urlencode($send_to_cc) . '&send_to_bcc='
            . urlencode($send_to_bcc) . '&subject=' . urlencode($subject)
            . '&banned_possible_spammer=1&body=' . urlencode($body)
            . rs_buildQueryString('mailprio') . rs_buildQueryString('delete_draft')
            . rs_buildQueryString('identity')
            // doesn't pull default so just forget it! . rs_buildQueryString('variable_sent_folder')
            . rs_buildQueryString('session')
            // makes query string too big . rs_buildQueryString('restoremessages')
            . rs_buildQueryString('request_mdn') . rs_buildQueryString('request_dr')
            . rs_buildQueryString('passed_id') . rs_buildQueryString('custom_from')
            . rs_buildQueryString('mailbox') . rs_buildQueryString('passed_ent_id')
                // SM 1.2.x : reply_id and forward_id
            . rs_buildQueryString('startMessage') . rs_buildQueryString('reply_id')
            . rs_buildQueryString('forward_id'));

         exit;

      }

   }



   // limit number of outgoing recipients if necessary
   //
   if ($restrictNumberOfRecipients 
      && $restrictNumberOfRecipients < sizeof($toDomains) + sizeof($ccDomains) + sizeof($bccDomains))
   {

      global $color, $send_to, $send_to_cc, $send_to_bcc, $subject, $body,
             $variable_sent_folder, $PHP_SELF,
             $report_abuse_too_many_recipients_message;

      $replyToList = array();
      $fromList = array();
      $delimiter = '@';
      $address_delim = ', ';

      foreach ($message->rfc822_header->reply_to as $sender)
         $replyToList[] = $sender->mailbox . $delimiter . $sender->host;
      foreach ($message->rfc822_header->from as $sender)
         $fromList[] = $sender->mailbox . $delimiter . $sender->host;


      // log this event - note that to use this,
      // you'll need to have the Squirrel Logger plugin
      // installed and activated and you'll have to add
      // a "RESTRICT_SENDERS" event type to its configuration
      //
      if ($log_too_many_recipients && is_plugin_enabled('squirrel_logger'))
      {
         include_once(SM_PATH . 'plugins/squirrel_logger/functions.php');
         sl_logit('RESTRICT_SENDERS', sprintf('Too many recipients (%d); message blocked', sizeof($toDomains) + sizeof($ccDomains) + sizeof($bccDomains)));
      }


      // send report if needed
      //
      if (!empty($report_abuse_too_many_recipients_message))
         $msg = _($report_abuse_too_many_recipients_message);
      else
         $msg = _("NOTICE: User \"###USERNAME###\" (domain \"###DOMAIN###\") has attempted to send an email with more than their allotted recipients: ###NUMBER_RECIPIENTS###\n\nTO: ###TO###\nCC: ###CC###\nBCC: ###BCC###\nFROM: ###FROM###\nREPLY-TO: ###REPLY_TO###\nSUBJECT: ###SUBJECT###");
         //$msg = _("NOTICE: User \"###USERNAME###\" (domain \"###DOMAIN###\") has attempted to send an email with more than their allotted recipients: ###NUMBER_RECIPIENTS###\n\nTO: ###TO###\nCC: ###CC###\nBCC: ###BCC###\nFROM: ###FROM###\nREPLY-TO: ###REPLY_TO###\nSUBJECT: ###SUBJECT###\nBODY: ###BODY###");
         //$msg = _("NOTICE: User \"###USERNAME###\" (domain \"###DOMAIN###\") has attempted to send an email with more than their allotted recipients: ###NUMBER_RECIPIENTS###");

      $msg = str_replace(array('###USERNAME###', '###DOMAIN###', 
                               '###NUMBER_RECIPIENTS###', '###TO###', 
                               '###CC###', '###BCC###', '###BODY###',
                               '###FROM###', '###REPLY_TO###', '###SUBJECT###'),
                         array($username, $domain, $restrictNumberOfRecipients, 
                               $send_to, $send_to_cc, $send_to_bcc, $body,
                               implode($address_delim, $fromList),
                               implode($address_delim, $replyToList),
                               $subject),
                         $msg);

      rs_report_abuse($msg);


      // now redirect/abort send
      //
      sqgetGlobalVar('variable_sent_folder', $variable_sent_folder);

      if (strpos($PHP_SELF, '?') === FALSE)
         $qs = '?';
      else
         $qs = '&';

//TODO: why not just put error in session before redirect?  (and all the other crap in the query string... just put the "query string" in session so no size limit and no one can change it in their browser).... at least put a flag in the session that there was an error message?  I guess it's not very important, because "tricking" this only stops the error message from being displayed, but does not get around the fact that the message was not sent
      header('Location: ' . $PHP_SELF . $qs . 'send_to=' . urlencode($send_to) 
         . '&send_to_cc=' . urlencode($send_to_cc) . '&send_to_bcc=' 
         . urlencode($send_to_bcc) . '&subject=' . urlencode($subject) 
         . '&restrict_senders_error_too_many_recipients=1&body=' . urlencode($body)
         . rs_buildQueryString('mailprio') . rs_buildQueryString('delete_draft') 
         . rs_buildQueryString('identity') 
         // doesn't pull default so just forget it! . rs_buildQueryString('variable_sent_folder') 
         . rs_buildQueryString('session') 
         // makes query string too big . rs_buildQueryString('restoremessages') 
         . rs_buildQueryString('request_mdn') . rs_buildQueryString('request_dr') 
         . rs_buildQueryString('passed_id') . rs_buildQueryString('custom_from') 
         . rs_buildQueryString('mailbox') . rs_buildQueryString('passed_ent_id') 
             // SM 1.2.x : reply_id and forward_id
         . rs_buildQueryString('startMessage') . rs_buildQueryString('reply_id') 
         . rs_buildQueryString('forward_id'));

      exit;

   }



   // limit number of emails sent per day
   //
   if ($restrictNumberOfEmailsPerDay)
   {

      // get and parse email count and date
      //
      list($year, $yearDay) = explode(':', date('Y:z', $now));
      $emailCount = getPref($data_dir, $username, 'todays_email_count', 0);
      if (!empty($emailCount))
      {
         list($day, $emailCount) = explode(':', $emailCount);
         list($prefYear, $prefYearDay) = explode('_', $day);
         if ($prefYear == $year && $prefYearDay == $yearDay)
            $emailCount++;
         else
            $emailCount = 1;
      }
      else
         $emailCount = 1;
      

      // check for overage
      //
      if ($emailCount > $restrictNumberOfEmailsPerDay)
      {

         // let this mail through if it is only being sent to 
         // whitelisted address(es)
         //
         if (only_sent_to_whitelist($toDomains, $ccDomains, $bccDomains, 
                                    $toMailboxes, $ccMailboxes, $bccMailboxes,
                                    $whitelisted_addresses, $rs_delimiter))
            return;


//FIXME: move this above the block above that lets whitelisted addresses through?
         // log this event - note that to use this,
         // you'll need to have the Squirrel Logger plugin
         // installed and activated and you'll have to add
         // a "RESTRICT_SENDERS" event type to its configuration
         //
         if ($log_too_many_today && is_plugin_enabled('squirrel_logger'))
         {
            include_once(SM_PATH . 'plugins/squirrel_logger/functions.php');
            sl_logit('RESTRICT_SENDERS', sprintf('Too many messages in one day (%d); message blocked', $restrictNumberOfEmailsPerDay));
         }


         global $color, $send_to, $send_to_cc, $send_to_bcc, $subject, $body,
                $variable_sent_folder, $PHP_SELF,
                $report_abuse_too_many_today_message;


         $replyToList = array();
         $fromList = array();
         $delimiter = '@';
         $address_delim = ', ';

         foreach ($message->rfc822_header->reply_to as $sender)
            $replyToList[] = $sender->mailbox . $delimiter . $sender->host;
         foreach ($message->rfc822_header->from as $sender)
            $fromList[] = $sender->mailbox . $delimiter . $sender->host;


         // send report if needed
         //
         if (!empty($report_abuse_too_many_today_message))
            $msg = _($report_abuse_too_many_today_message);
         else
            $msg = _("WARNING: User \"###USERNAME###\" (domain \"###DOMAIN###\") has attempted to send more than their allotted number of emails in one day: ###NUMBER_MAILS_TODAY###\n\nTo allow this user to send mail again today, remove the \"todays_email_count\" setting from their preferences.\n\nTO: ###TO###\nCC: ###CC###\nBCC: ###BCC###\nFROM: ###FROM###\nREPLY-TO: ###REPLY_TO###\nSUBJECT: ###SUBJECT###");
            //$msg = _("WARNING: User \"###USERNAME###\" (domain \"###DOMAIN###\") has attempted to send more than their allotted number of emails in one day: ###NUMBER_MAILS_TODAY###\n\nTo allow this user to send mail again today, remove the \"todays_email_count\" setting from their preferences.\n\nTO: ###TO###\nCC: ###CC###\nBCC: ###BCC###\nFROM: ###FROM###\nREPLY-TO: ###REPLY_TO###\nSUBJECT: ###SUBJECT###\nBODY: ###BODY###");
            //$msg = _("WARNING: User \"###USERNAME###\" (domain \"###DOMAIN###\") has attempted to send more than their allotted number of emails in one day: ###NUMBER_RECIPIENTS###");

         $msg = str_replace(array('###USERNAME###', '###DOMAIN###', 
                                  '###NUMBER_MAILS_TODAY###', '###TO###', 
                                  '###CC###', '###BCC###', '###BODY###',
                                  '###FROM###', '###REPLY_TO###', '###SUBJECT###'),
                            array($username, $domain, $restrictNumberOfEmailsPerDay, 
                                  $send_to, $send_to_cc, $send_to_bcc, $body,
                                  implode($address_delim, $fromList),
                                  implode($address_delim, $replyToList),
                                  $subject),
                            $msg);

         rs_report_abuse($msg);


         // now redirect/abort send
         //
         sqgetGlobalVar('variable_sent_folder', $variable_sent_folder);

         if (strpos($PHP_SELF, '?') === FALSE)
            $qs = '?';
         else
            $qs = '&';

//TODO: why not just put error in session before redirect?  (and all the other crap in the query string... just put the "query string" in session so no size limit and no one can change it in their browser).... at least put a flag in the session that there was an error message?  I guess it's not very important, because "tricking" this only stops the error message from being displayed, but does not get around the fact that the message was not sent
         header('Location: ' . $PHP_SELF . $qs . 'send_to=' . urlencode($send_to) 
            . '&send_to_cc=' . urlencode($send_to_cc) . '&send_to_bcc=' 
            . urlencode($send_to_bcc) . '&subject=' . urlencode($subject) 
            . '&restrict_senders_error_too_many_emails_today=1&body=' . urlencode($body)
            . rs_buildQueryString('mailprio') . rs_buildQueryString('delete_draft') 
            . rs_buildQueryString('identity') 
            // doesn't pull default so just forget it! . rs_buildQueryString('variable_sent_folder') 
            . rs_buildQueryString('session') 
            // makes query string too big . rs_buildQueryString('restoremessages') 
            . rs_buildQueryString('request_mdn') . rs_buildQueryString('request_dr') 
            . rs_buildQueryString('passed_id') . rs_buildQueryString('custom_from') 
            . rs_buildQueryString('mailbox') . rs_buildQueryString('passed_ent_id') 
                // SM 1.2.x : reply_id and forward_id
            . rs_buildQueryString('startMessage') . rs_buildQueryString('reply_id') 
            . rs_buildQueryString('forward_id'));

         exit;

      }


      // otherwise, continue, incrementing counter
      //
      setPref($data_dir, $username, 'todays_email_count', $year . '_' . $yearDay . ':' . $emailCount);

   }



   // limit send domains?  if not, just return (but not before 
   // setting the sent_confirmation plugin straight if necessary)
   //
   if (!$restrictDomains)
   {

      // check if the sent_confirmation plugin
      // is installed and if so, if it was delayed
      // in order for this plugin to run first
      //
      global $plugins;

      if (in_array('sent_confirmation', $plugins))
      {
         global $restrict_senders_finished, $sent_confirmation_was_delayed;

         $restrict_senders_finished = 1;

         if ($sent_confirmation_was_delayed)
            sent_conf_message_sent_do();

      }

      return;

   }



   // check if we've been here before - is the user's restrict
   // info already in the session?  if so, use it... otherwise
   // parse it out of the data file
   //
   $restrict_sender_rules = '';
   if ($rs_debug & 2
    || (!sqgetGlobalVar('restrict_sender_rules', $restrict_sender_rules, SQ_SESSION)
    || empty($restrict_sender_rules)))
   {

      global $vlogin_restrict_senders_rules_file;
      if (!empty($vlogin_restrict_senders_rules_file)) 
         $rules_file = $vlogin_restrict_senders_rules_file;
      else
         $rules_file = '';

      $restrict_sender_rules = parseSenderRules($rules_file);

      // put it in the session
      //
      sqsession_register($restrict_sender_rules, 'restrict_sender_rules');

   }
   

   // dump ruleset for debugging
   //
   if ($rs_debug & 4)
   {
      global $username, $domain;
      sm_print_r("Restrict Senders plugin send restriction rules found for $username and $domain:",
                 '================================================================',
                 $restrict_sender_rules,
                 '================================================================');
      exit;
   }


   // now find out which addresses are not allowable
   //
   list($toResults, $ccResults, $bccResults) 
         = filterAddresses($toDomains, $ccDomains, $bccDomains, 
                           $restrict_sender_rules,
                           $toMailboxes, $ccMailboxes, $bccMailboxes,
                           $whitelisted_addresses, $rs_delimiter);



   // eliminate disallowed addresses from message
   //
   // loop thru return results of filter function and call my custom func
   //
   $blockedAddresses = array();
   $badAddrs = 0;
   for ($i = 0; $i < sizeof($toResults); $i++)
      if (!$toResults[$i])
      {
         $blockedAddresses[] = $message->rfc822_header->to[$i - $badAddrs]->mailbox . '@'
                             . $message->rfc822_header->to[$i - $badAddrs]->host;
         $message->rfc822_header->to = unset_and_renumber_array($message->rfc822_header->to, 
                                                                $i - $badAddrs);
         $badAddrs++;
      }

   $badAddrs = 0;
   for ($i = 0; $i < sizeof($ccResults); $i++)
      if (!$ccResults[$i])
      {
         $blockedAddresses[] = $message->rfc822_header->cc[$i - $badAddrs]->mailbox . '@'
                             . $message->rfc822_header->cc[$i - $badAddrs]->host;
         $message->rfc822_header->cc = unset_and_renumber_array($message->rfc822_header->cc, 
                                                                $i - $badAddrs);
         $badAddrs++;
      }

   $badAddrs = 0;
   for ($i = 0; $i < sizeof($bccResults); $i++)
      if (!$bccResults[$i])
      {
         $blockedAddresses[] = $message->rfc822_header->bcc[$i - $badAddrs]->mailbox . '@'
                             . $message->rfc822_header->bcc[$i - $badAddrs]->host;
         $message->rfc822_header->bcc = unset_and_renumber_array($message->rfc822_header->bcc, 
                                                                $i - $badAddrs);
         $badAddrs++;
      }



   // check message for at least one address in the TO, CC, or BCC fields
   //
   if (sizeof($message->rfc822_header->to) == 0
    && sizeof($message->rfc822_header->cc) == 0
    && sizeof($message->rfc822_header->bcc) == 0)
   {
      // log this event - note that to use this,
      // you'll need to have the Squirrel Logger plugin
      // installed and activated and you'll have to add
      // a "RESTRICT_SENDERS" event type to its configuration
      //
      if ($log_no_recipients && is_plugin_enabled('squirrel_logger'))
      {
         include_once(SM_PATH . 'plugins/squirrel_logger/functions.php');
         sl_logit('RESTRICT_SENDERS', 'All recipients are blacklisted per sender restriction rules');
      }


      global $color, $send_to, $send_to_cc, $send_to_bcc, $subject, $body,
             $variable_sent_folder, $PHP_SELF;
      sqgetGlobalVar('variable_sent_folder', $variable_sent_folder);

      if (strpos($PHP_SELF, '?') === FALSE)
         $qs = '?';
      else
         $qs = '&';

      header('Location: ' . $PHP_SELF . $qs . 'send_to=' . urlencode($send_to) 
         . '&send_to_cc=' . urlencode($send_to_cc) . '&send_to_bcc=' 
         . urlencode($send_to_bcc) . '&subject=' . urlencode($subject) 
         . '&restrict_senders_error_no_to_recipients=1&body=' . urlencode($body)
         . rs_buildQueryString('mailprio') . rs_buildQueryString('delete_draft') 
         . rs_buildQueryString('identity') 
         // doesn't pull default so just forget it! . rs_buildQueryString('variable_sent_folder') 
         . rs_buildQueryString('session') 
         // makes query string too big . rs_buildQueryString('restoremessages') 
         . rs_buildQueryString('request_mdn') . rs_buildQueryString('request_dr') 
         . rs_buildQueryString('passed_id') . rs_buildQueryString('custom_from') 
         . rs_buildQueryString('mailbox') . rs_buildQueryString('passed_ent_id') 
             // SM 1.2.x : reply_id and forward_id
         . rs_buildQueryString('startMessage') . rs_buildQueryString('reply_id') 
         . rs_buildQueryString('forward_id'));

      exit;
   }



   // put blocked addresses into session so they can be displayed
   // by sent_confirmation (or other)
   //
   sqsession_register($blockedAddresses, 'blockedAddresses');



   // check if the sent_confirmation plugin
   // is installed and if so, if it was delayed
   // in order for this plugin to run first
   //
   global $plugins;

   if (in_array('sent_confirmation', $plugins))
   {
      global $restrict_senders_finished, $sent_confirmation_was_delayed;

      $restrict_senders_finished = 1;

      // sent_confirmation gets its information from the 
      // $send_to, $send_to_cc, and $send_to_bcc variables,
      // so as much of a pain as it is, we need to reset
      // them now
// TODO: doing this means we lose formatting of email addresses 
//       like "Joe Smith" <joe@smith.com>  and so forth.  It 
//       is harder, but we could parse up these variables in
//       the same way as we do the message itself above, in
//       order to preserve this formatting
      //
      global $send_to, $send_to_cc, $send_to_bcc;

      $send_to = '';
      for ($i = 0; $i < sizeof($message->rfc822_header->to); $i++)
      {
         if ($i > 0) $send_to .= ', ';
         $send_to .= $message->rfc822_header->to[$i]->mailbox . '@' . $message->rfc822_header->to[$i]->host;
      }


      $send_to_cc = '';
      for ($i = 0; $i < sizeof($message->rfc822_header->cc); $i++)
      {
         if ($i > 0) $send_to_cc .= ', ';
         $send_to_cc .= $message->rfc822_header->cc[$i]->mailbox . '@' . $message->rfc822_header->cc[$i]->host;
      }


      $send_to_bcc = '';
      for ($i = 0; $i < sizeof($message->rfc822_header->bcc); $i++)
      {
         if ($i > 0) $send_to_bcc .= ', ';
         $send_to_bcc .= $message->rfc822_header->bcc[$i]->mailbox . '@' . $message->rfc822_header->bcc[$i]->host;
      }


      if ($sent_confirmation_was_delayed)
         sent_conf_message_sent_do();

   }



   // return the message
   //
   return $message;


}



/**
  * Utility function that builds up a location string.
  *
  * Note that it always prepends & and thus won't work
  * for the first item in the query string.
  *
  * The variable will be renamed in the query string 
  * if $queryName is given
  *
  * If the variable is empty, it will NOT be added to 
  * the query string (blank string is returned).
  *
  * @param string $varName The name of the variable to push 
  *                        into query string style
  * @param string $queryName The name of the variable to use
  *                          instead of the variable name from 
  *                          which the value comes (optional;
  *                          default not used - same as var name).
  *
  * @return string Variable formatted in query string style
  *
  */
function rs_buildQueryString($varName, $queryName = '')
{
   global $$varName;
   ////if (isset($$varName))
   if (!empty($$varName))
      return '&' . (!empty($queryName) ? $queryName : $varName) . '=' . $$varName;
   else
      return '';
}



/**
  * Utility function that parses the sender rules configuration
  * file and returns sender rules for the current user
  *
  * @param string  $rules_file The full path to the rules file
  *                            to be parsed.  (OPTIONAL; default
  *                            is to use the file called 
  *                            send_restrictions.php in the 
  *                            SquirrelMail config directory, or
  *                            if that file is not found, the file
  *                            called send_restrictions.php in the
  *                            restrict_senders/data directory)
  * @param boolean $configtest When TRUE, this function will test
  *                            whether or not the sender rules
  *                            configuration file is available and
  *                            is error-free.  (OPTIONAL; default = FALSE)
  *
  * @return mixed Usually returns list (whitespace separated in one string) 
  *               of any sender rules found for the current user, unless 
  *               $configtest is TRUE, in which case if the sender rules 
  *               configuration file is either not available or contains
  *               errors, an array is returned, containing a string with
  *               information about the configuration error (putting the
  *               error information inside an array merely differentiates
  *               it from otherwise valid output of this function)
  *
  */
function parseSenderRules($rules_file='', $configtest=FALSE)
{

   global $username, $domain;

   $lineNo = 0;


   if (empty($rules_file))
      $rules_file = SM_PATH . 'config/send_restrictions.php';
   if (!file_exists($rules_file))
      $rules_file = SM_PATH . 'plugins/restrict_senders/data/send_restrictions.php';


   if ($RESTRICTTABLE = @fopen($rules_file, 'r'))
   {

      while (!feof($RESTRICTTABLE))
      {

         $line = fgets($RESTRICTTABLE, 4096);
         $line = trim($line);
         $lineNo++;


         // skip blank lines and comment lines and php
         // open/close tag lines
         //
         if (strpos($line, '#') === 0 || strlen($line) < 3
          || strpos($line, '<?php') === 0 || strpos($line, '*/ ?>') === 0)
            continue;


         // is this a domain line?
         //
         if (preg_match('/^\s*domain:\s*(\S+)\s+(.*)$/', $line, $matches))
         {

            // check for match with hostname, return restrictions if so
            //
            if (preg_match('/^' . str_replace(array('.', '?', '*'), array('\.', '\w{1}', '.*?'), 
                          strtoupper($matches[1])) . '$/', strtoupper($domain)))
            {
               fclose($RESTRICTTABLE);
               return $matches[2];
            }

         }


         // is this a user line?
         //
         else if (preg_match('/^\s*user:\s*(\S+)\s+(.*)$/', $line, $matches))
         {

            // check for match with username, return restrictions if so
            //
            if (preg_match('/^' . str_replace(array('.', '?', '*'), array('\.', '\w{1}', '.*?'), 
                          strtoupper($matches[1])) . '$/', strtoupper($username)))
            {
               fclose($RESTRICTTABLE);
               return $matches[2];
            }

         }


         // is this a catchall line?
         //
         else if (preg_match('/^\s*(allow|deny)\s*$/', $line, $matches))
         {

            // just return the restriction
            //
            fclose($RESTRICTTABLE);
            return $matches[1];

         }


         // line format not understood
         //
         else
         {

            sq_change_text_domain('restrict_senders');
            $msg = sprintf(_("The \"%s\" plugin is not configured correctly.  The file \"%s\" has a syntax error on line %d."),
                           'Restrict Senders',
                           $rules_file,
                           $lineNo);
            sq_change_text_domain('squirrelmail');

            if ($configtest) return array($msg);

            global $color;
            if (check_sm_version(1, 5, 2))
               plain_error_message($msg . ' ' . _("Message not sent."));
            else
               plain_error_message(str_replace('  ', '<br />', $msg) . '<br /><center>' . _("Message not sent.") . '</center>', $color);
            exit;

         }

      }


      // nothing found, so allow everything
      //
      fclose($RESTRICTTABLE);
      return 'allow';

   }
   else
   {

      sq_change_text_domain('restrict_senders');
      $msg = sprintf(_("The \"%s\" plugin is not configured correctly.  The file \"%s\" could not be opened."), 'Restrict Senders', $rules_file);
      sq_change_text_domain('squirrelmail');

      if ($configtest) return array($msg);

      global $color;
      if (check_sm_version(1, 5, 2))
         plain_error_message($msg . ' ' . _("Message not sent."));
      else
         plain_error_message(str_replace('  ', '<br />', $msg) . '<br /><center>' . _("Message not sent.") . '</center>', $color);
      exit;

   }

}



/**
  * Checks to see if the current message destinations ONLY contain
  * whitelisted addresses.
  *
  * @param array  $toDomains All domains for To: addresses.
  * @param array  $ccDomains All domains for Cc: addresses.
  * @param array  $bccDomains All domains for Bcc: addresses.
  * @param array  $toMailboxes All mailbox part for To: addresses.
  * @param array  $ccMailboxes All mailbox part for Cc: addresses.
  * @param array  $bccMailboxes All mailbox part for Bcc: addresses.
  * @param array  $whitelisted_addresses A list of whitelisted email addresses.
  * @param string $rs_delimiter The mailbox/host delimiter string (usually "@")
  *                             to use for parsing whitelisted email addresses.
  *
  * @return boolean TRUE if all destination addresses are whitelisted,
  *                 FALSE otherwise.
  *
  */
function only_sent_to_whitelist($toDomains, $ccDomains, $bccDomains, 
                                $toMailboxes, $ccMailboxes, $bccMailboxes,
                                $whitelisted_addresses, $rs_delimiter)
{

   // loop thru TO: addresses
   // 
   for ($i = 0; $i < sizeof($toDomains); $i++)
   {
      // if even one address is not whitelisted, return FALSE
      //
      if (!in_array($toMailboxes[$i] . $rs_delimiter . $toDomains[$i], $whitelisted_addresses))
         return FALSE;
   }


   // loop thru CC: addresses
   // 
   for ($i = 0; $i < sizeof($ccDomains); $i++)
   {
      // if even one address is not whitelisted, return FALSE
      //
      if (!in_array($ccMailboxes[$i] . $rs_delimiter . $ccDomains[$i], $whitelisted_addresses))
         return FALSE;
   }


   // loop thru BCC: addresses
   // 
   for ($i = 0; $i < sizeof($bccDomains); $i++)
   {
      // if even one address is not whitelisted, return FALSE
      //
      if (!in_array($bccMailboxes[$i] . $rs_delimiter . $bccDomains[$i], $whitelisted_addresses))
         return FALSE;
   }


   return TRUE;

}



/**
  * Utility function that looks at all email addresses and determines
  * which ones are OK and which ones are bad, according to the rules
  * provided.
  *
  * @param array  $toDomains All domains for To: addresses.
  * @param array  $ccDomains All domains for Cc: addresses.
  * @param array  $bccDomains All domains for Bcc: addresses.
  * @param string $restrict_sender_rules The sender rules as a whitespace-
  *                                      separated list in a single string.
  * @param array  $toMailboxes All mailbox part for To: addresses.
  * @param array  $ccMailboxes All mailbox part for Cc: addresses.
  * @param array  $bccMailboxes All mailbox part for Bcc: addresses.
  * @param array  $whitelisted_addresses A list of whitelisted email addresses
  *                                      that will be exempted from any blocked
  *                                      addresses.
  * @param string $rs_delimiter The mailbox/host delimiter string (usually "@")
  *                             to use for parsing whitelisted email addresses.
  *
  * @return array A three-element list in this order: $toResults, 
  *               $ccResults, $bccResults
  *
  */
function filterAddresses($toDomains, $ccDomains, $bccDomains, 
                         $restrict_sender_rules,
                         $toMailboxes, $ccMailboxes, $bccMailboxes,
                         $whitelisted_addresses, $rs_delimiter)
{

   $theRules = preg_split('/\s+/', str_replace(array('.', '?', '*'),
                                               array('\.', '\w{1}', '.*?'),
                                               $restrict_sender_rules));
   $toResults = array();
   $ccResults = array();
   $bccResults = array();


   // loop thru TO: domains
   // 
   for ($i = 0; $i < sizeof($toDomains); $i++)
   {


      // ignore rules for whitelisted addresses 
      //
      if (in_array($toMailboxes[$i] . $rs_delimiter . $toDomains[$i], $whitelisted_addresses))
      {
         $toResults[$i] = 1;
         continue;
      }


      // loop thru each rule
      //
      for ($j = 0; $j < sizeof($theRules); $j++)
      {

         if ($theRules[$j] == 'allow') 
         {
            $toResults[$i] = 1;
            continue 2;
         }
         else if ($theRules[$j] == 'deny') 
         {
            $toResults[$i] = 0;
            continue 2;
         }
         else
         {
            $compare_rule = explode(':', $theRules[$j], 2);
            if (sizeof($compare_rule) == 2)
            {
               // the following if/else structure could be combined 
               // into one nasty regexp, but for the sake of clarity
               // we spell it out...
               //
               if (strpos($compare_rule[1], '@') !== FALSE)
               {
                  if (preg_match('/^' . $compare_rule[1] . '$/', 
                                 $toMailboxes[$i] . '@' . $toDomains[$i]))
                  {
                     $toResults[$i] = ($compare_rule[0] == 'allow' ? 1 : 0);
                     continue 2;
                  }
               }
               else
               {
                  if (preg_match('/^' . $compare_rule[1] . '$/', 
                                 $toDomains[$i]))
                  {
                     $toResults[$i] = ($compare_rule[0] == 'allow' ? 1 : 0);
                     continue 2;
                  }
               }
            }
         }

      }

   }



   // loop thru CC: domains
   // 
   for ($i = 0; $i < sizeof($ccDomains); $i++)
   {


      // ignore rules for whitelisted addresses 
      //
      if (in_array($ccMailboxes[$i] . $rs_delimiter . $ccDomains[$i], $whitelisted_addresses))
      {
         $ccResults[$i] = 1;
         continue;
      }


      // loop thru each rule
      //
      for ($j = 0; $j < sizeof($theRules); $j++)
      {

         if ($theRules[$j] == 'allow') 
         {
            $ccResults[$i] = 1;
            continue 2;
         }
         else if ($theRules[$j] == 'deny') 
         {
            $ccResults[$i] = 0;
            continue 2;
         }
         else
         {
            $compare_rule = explode(':', $theRules[$j], 2);
            if (sizeof($compare_rule) == 2)
            {
               // the following if/else structure could be combined
               // into one nasty regexp, but for the sake of clarity
               // we spell it out...
               //
               if (strpos($compare_rule[1], '@') !== FALSE)
               {
                  if (preg_match('/^' . $compare_rule[1] . '$/',
                                 $ccMailboxes[$i] . '@' . $ccDomains[$i]))
                  {
                     $ccResults[$i] = ($compare_rule[0] == 'allow' ? 1 : 0);
                     continue 2;
                  }
               }
               else
               {
                  if (preg_match('/^' . $compare_rule[1] . '$/',
                                 $ccDomains[$i]))
                  {
                     $ccResults[$i] = ($compare_rule[0] == 'allow' ? 1 : 0);
                     continue 2;
                  }
               }
            }
         }

      }

   }



   // loop thru BCC: domains
   // 
   for ($i = 0; $i < sizeof($bccDomains); $i++)
   {


      // ignore rules for whitelisted addresses 
      //
      if (in_array($bccMailboxes[$i] . $rs_delimiter . $bccDomains[$i], $whitelisted_addresses))
      {
         $bccResults[$i] = 1;
         continue;
      }


      // loop thru each rule
      //
      for ($j = 0; $j < sizeof($theRules); $j++)
      {

         if ($theRules[$j] == 'allow') 
         {
            $bccResults[$i] = 1;
            continue 2;
         }
         else if ($theRules[$j] == 'deny') 
         {
            $bccResults[$i] = 0;
            continue 2;
         }
         else
         {
            $compare_rule = explode(':', $theRules[$j], 2);
            if (sizeof($compare_rule) == 2)
            {
               // the following if/else structure could be combined
               // into one nasty regexp, but for the sake of clarity
               // we spell it out...
               //
               if (strpos($compare_rule[1], '@') !== FALSE)
               {
                  if (preg_match('/^' . $compare_rule[1] . '$/',
                                 $bccMailboxes[$i] . '@' . $bccDomains[$i]))
                  {
                     $bccResults[$i] = ($compare_rule[0] == 'allow' ? 1 : 0);
                     continue 2;
                  }
               }
               else
               {
                  if (preg_match('/^' . $compare_rule[1] . '$/',
                                 $bccDomains[$i]))
                  {
                     $bccResults[$i] = ($compare_rule[0] == 'allow' ? 1 : 0);
                     continue 2;
                  }
               }
            }
         }

      }

   }


   // return results
   //
   return array($toResults, $ccResults, $bccResults);

}



/**
  * Reports abuse to administrator
  *
  * @param string $error   Error text indicating what the abuse problem is
  * @param string $title   Short text for inclusion in email subject line
  * @param object $message The message object that is being attempted to 
  *                        be sent (optional, ignored if not given)
  *
  */
function rs_report_abuse($error, $title='', $message='')
{

   $delim = '@';

   global $rs_report_addresses, $domain, $username,
          $rs_use_Vadmin_delete_link, $rs_useSendmail,
          $rs_smtpServerAddress, $rs_smtpPort,
          $rs_sendmail_path, $rs_sendmail_args,
          $rs_pop_before_smtp, $rs_encode_header_key,
          $rs_smtp_auth_mech, $rs_smtp_sitewide_user,
          $rs_smtp_sitewide_pass, $useSendmail, $smtpServerAddress,
          $smtpPort, $sendmail_path, $sendmail_args, $pop_before_smtp,
          $encode_header_key, $smtp_auth_mech, $smtp_sitewide_user,
          $smtp_sitewide_pass,
          $include_server_env_with_abuse_report;


   // get global variable for versions of PHP < 4.1
   //
   if (!check_php_version(4,1)) {
      global $HTTP_SERVER_VARS;
      $_SERVER = $HTTP_SERVER_VARS;
   }


   restrict_senders_init();

   if (empty($rs_report_addresses)) 
      return;

   sq_change_text_domain('restrict_senders');

   $user = substr($username, 0, strpos($username, $delim));

   $body = $error . "\n\n" . _("User account:") . ' ' . $username
         . "\n" . _("Domain:") . ' ' . $domain
         . "\n" . _("Timestamp: ") . date('Y-m-d H:i:s')
         . ($rs_use_Vadmin_delete_link ? "\n" . _("Delete account:") . ' '
         . sqm_baseuri() . 'plugins/vadmin/vadmin_main.php?LVL=admin&MOD=accounts&ACT=deluser&userid='
         . $user . ' ' . _("(must be logged into administrative interface)")
         : '');

   if (!empty($message))
   {
      $body .= "\n\n=========================================================\n"
            . _("OFFENDING MESSAGE")
            . "\n=========================================================\n\n";
      ob_start();
      print_r($message);
      $body .= ob_get_contents();
      ob_end_clean();
   }

   if ($include_server_env_with_abuse_report)
   {
      $body .= "\n\n=========================================================\n"
            . _("OFFENDING SERVER ENVIRONMENT")
            . "\n=========================================================\n\n";
      ob_start();
      global $_SERVER;
      print_r($_SERVER);
      $body .= ob_get_contents();
      ob_end_clean();
   }


   $subject = _("[WEBMAIL ABUSE]")
            . (empty($title) ? '' : ': ' . $title)
            . ' (' . $username . ')';


   sq_change_text_domain('squirrelmail');


   // override any SMTP/Sendmail settings...
   //
   $orig_useSendmail = $useSendmail;
   if (!is_null($rs_useSendmail))
      $useSendmail = $rs_useSendmail;
   $orig_smtpServerAddress = $smtpServerAddress;
   if (!is_null($rs_smtpServerAddress))
      $smtpServerAddress = $rs_smtpServerAddress;
   $orig_smtpPort = $smtpPort;
   if (!is_null($rs_smtpPort))
      $smtpPort = $rs_smtpPort;
   $orig_sendmail_path = $sendmail_path;
   if (!is_null($rs_sendmail_path))
      $sendmail_path = $rs_sendmail_path;
   $orig_sendmail_args = $sendmail_args;
   if (!is_null($rs_sendmail_args))
      $sendmail_args = $rs_sendmail_args;
   $orig_pop_before_smtp = $pop_before_smtp;
   if (!is_null($rs_pop_before_smtp))
      $pop_before_smtp = $rs_pop_before_smtp;
   $orig_encode_header_key = $encode_header_key;
   if (!is_null($rs_encode_header_key))
      $encode_header_key = $rs_encode_header_key;
   $orig_smtp_auth_mech = $smtp_auth_mech;
   if (!is_null($rs_smtp_auth_mech))
      $smtp_auth_mech = $rs_smtp_auth_mech;
   $orig_smtp_sitewide_user = $smtp_sitewide_user;
   if (!is_null($rs_smtp_sitewide_user))
      $smtp_sitewide_user = $rs_smtp_sitewide_user;
   $orig_smtp_sitewide_pass = $smtp_sitewide_pass;
   if (!is_null($rs_smtp_sitewide_pass))
      $smtp_sitewide_pass = $rs_smtp_sitewide_pass;


   // function found in SM core 1.5.2+ and Compatibility plugin 2.0.11+
   // (suppress include error, since file is only needed for 1.5.2+,
   // otherwise Compatibility has us covered)
   //
   @include_once(SM_PATH . 'functions/compose.php');
   $success = sq_send_mail($rs_report_addresses, $subject, $body, 'noreply@' . $domain);


   // return SMTP/Sendmail settings to what they were
   //
   $useSendmail = $orig_useSendmail;
   $smtpServerAddress = $orig_smtpServerAddress;
   $smtpPort = $orig_smtpPort;
   $sendmail_path = $orig_sendmail_path;
   $sendmail_args = $orig_sendmail_args;
   $pop_before_smtp = $orig_pop_before_smtp;
   $encode_header_key = $orig_encode_header_key;
   $smtp_auth_mech = $orig_smtp_auth_mech;
   $smtp_sitewide_user = $orig_smtp_sitewide_user;
   $smtp_sitewide_pass = $orig_smtp_sitewide_pass;

}



