<?php

include_once(SM_PATH . 'plugins/proon/define.php');
include_once(SM_PATH . 'plugins/proon/defineui.php');

/**
 * Look for a suitable delimiter character.  For each
 * candidate delimiter character in $trythese, see if it
 * happens to appear in any of the keys or values in 
 * $some_array.
 */
function proon_pick_delimiter($some_array, $trythese) {
  if ($some_array) {
    $line = '';
    foreach ($some_array as $key => $val) {
      $line .= "$key$val";
    }
    $len = strlen($trythese);
    for ($ii=0; $ii<$len; ++$ii) {
      $maybe = $trythese{$ii};
      if (!strstr($line, $maybe)) {
	return $maybe;
      }
    }
  }
  return null;
}

/**
 * Encode an associative array of key/value pairs into a delimited
 * string of the form "=k1=v1=k2=v2".  The string starts with the
 * delimiter character, which is selected so as to not conflict with
 * any of the keys or values.  There is no delimiter at the end of
 * the returned string.
 */
function proon_to_pref(&$some_array) {
  if (!isset($some_array)  ||  count($some_array) == 0) return null;
  $d1 = '~!@#$%^\\&*()_+=-`\'[]{};:"/?.>,<|';
  $d2 = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
  $d3 = 'abcdefghijklmnopqrstuvwxyz';
  $d4 = '0123456789';
  $delim = proon_pick_delimiter($some_array, $d1 . $d2 . $d3 . $d4);
  if ($delim == null) return false;
  $pref = '';
  if ($some_array) foreach($some_array as $key=>$value) {
    $pref .= "$delim$key$delim$value";
  }
  return $pref;
}

/**
 * The logical reverse of the proon_to_pref function.  It unpacks a
 * delimited string and returns an associative array of the key/value
 * pairs found.  It's very tolerant, so you're likely to get some
 * kind of result even if the input is grossly malformed.
 */
function proon_from_pref($pref) {
  $delim = $pref{0};
  $pref  = substr($pref, 1);
  $pref  = explode($delim, $pref);
  $len = count($pref);
  for ($ii=0; $ii<$len; ) {
    $k = $pref[$ii];
    ++$ii;
    $v = $pref[$ii];
    ++$ii;
    $kv[$k] = $v;
  }
  return $kv;
}

/**
 * Factor out login logic to avoid multiple connects when just one
 * will do.  In fact, if the SM connection is already hanging around
 * we'll use that one.
 */
function proon_login(& $imapStr) {
  if ($imapStr == null) {
    global $imapConnection;
    if (isset($imapConnection)  &&  $imapConnection !== null) {
      global $proon_log_verbose;
      $proon_log_verbose && proon_log("logon  imapConnection $imapConnection");
      $imapStr = $imapConnection;
    }
    else {
      include_once(SM_PATH . 'functions/imap.php');
      global $imapServerAddress, $imapPort;
      $imapStr = sqimap_login($_SESSION['username'], $_COOKIE['key'], $imapServerAddress, $imapPort, 0);
    }
  }
}

/**
 * Only actually do an IMAP logout if we did the login.  Otherwise,
 * it's somebody else's job.
 */
function proon_logout(& $imapStr) {
  if (!isset($imapStr)  ||  $imapStr == null) return;

  global $imapConnection;
  if (isset($imapConnection)  &&  $imapConnection == $imapStr) {
    global $proon_log_verbose;
    $proon_log_verbose && proon_log("logoff imapConnection $imapConnection");
  }
  else {
    sqimap_logout($imapStr);
  }
  $imapStr = null;
}

function proon_check_spans(&$spans, $rawdate, $rawsize, $rawcount) {
  if (!$rawdate  &&  !$rawsize  &&  !$rawcount) {
    return _("None of the span values has been set for this folder.");
  }
  $datespan = proon_date_in_seconds($rawdate);
  $sizespan = proon_size_in_bytes($rawsize);
  $countspan = proon_count_in_each($rawcount);

  if ($rawdate != ''  &&  $datespan === false) {
    return _("The date span is malformed.");
  }
  if ($rawsize != ''  &&  $sizespan === false) {
    return _("The size span is malformed.");
  }
  if ($rawcount != ''  &&  $countspan === false) {
    return _("The count span is malformed.");
  }
  $spans[PROON_SPAN_DATE] = $datespan;
  $spans[PROON_SPAN_SIZE] = $sizespan;
  $spans[PROON_SPAN_COUNT] = $countspan;
  return null;
}

function proon_date_in_seconds($rawspan) {
  $rawspan = trim($rawspan);
  if (!isset($rawspan)  ||  $rawspan == ''  ||  $rawspan == '/') return false;
  $first_slash = strpos($rawspan, '/');
  $last_slash = strrpos($rawspan, '/');
  if ($first_slash != $last_slash) return false;

  if (false !== $first_slash) {
    list($days, $hours) = explode('/', $rawspan);
    $days = trim($days);
    $hours = trim($hours);
    if (! $days) $days = 0;
    if (! $hours) $hours = 0;
  }
  else {
    $days = trim($rawspan);
    $hours = 0;
  }
  if (!is_numeric($days)  ||  !is_numeric($hours)) return false;

  $spansecs = 3600 * ((24 * $days) + $hours);
  // not so reliable around or before ye olde epoch date
  if ((time() - $spansecs) < 100000) return false;
  if ($spansecs < 0) return false;
  return $spansecs;
}

function proon_size_in_bytes($rawsize) {
  return proon_scaled_singles($rawsize, 'm');
}

function proon_count_in_each($rawsize) {
  return proon_scaled_singles($rawsize, 'b');
}

function proon_scaled_singles($rawvalue, $default_suffix) {
  $rawvalue = trim($rawvalue);
  if (!isset($rawvalue) || $rawvalue == '') return false;
  $suffix = substr($rawvalue, strlen($rawvalue) - 1);
  if (is_numeric($suffix)) {
    $number = $rawvalue;
    $suffix = $default_suffix;
  }
  else {
    $number = substr($rawvalue, 0, strlen($rawvalue) - 1);
  }
  if (!is_numeric($number)) return false;
  switch ($suffix) {
  case 'm': $multiplier = 1000 * 1000; break;
  case 'M': $multiplier = 1024 * 1024; break;
  case 'k': $multiplier = 1000; break;
  case 'K': $multiplier = 1024; break;
  case 'b': $multiplier = 1; break;
  case 'B': $multiplier = 1; break;
  default:
    return false;
  }
  $tally = ($number * $multiplier);
  if ($tally <= 0) return false;
  return $tally;
}

function proon_pull_foldername($key) {
  list($word, $foldername) = explode('_', $key);
  return base64_decode($foldername);
}

function proon_encode_foldername($folder) {
  return base64_encode($folder);
}

function proon_arrays_differ(&$a, &$b) {
  if ($a === $b) return false;
  if (is_array($a)  &&  !is_array($b)) return true;
  if (is_array($b)  &&  !is_array($a)) return true;
  // we'd use array_diff_assoc here, but it only comes in at PHP 4.3
  // array_diff isn't good enough because we care about the keys
  foreach ($a as $k => $v) {
    if (!isset($b[$k])  ||  $b[$k] !== $v) return true;
  }
  foreach ($b as $k => $v) {
    if (!isset($a[$k])  ||  $a[$k] !== $v) return true;
  }
  return false;
}

/**
 * Write all proon-related preference items to the user's store.
 */
function proon_set_prefs(&$p) {
  global $data_dir, $username;

  $pref = proon_to_pref($p[PROON_F_DATES]);
  if ($pref == '') $pref = PROON_PREF_BLANK;
  setPref($data_dir, $username, PROON_O_F_DATE_SPAN, $pref);

  $pref = proon_to_pref($p[PROON_F_SIZES]);
  if ($pref == '') $pref = PROON_PREF_BLANK;
  setPref($data_dir, $username, PROON_O_F_SIZE_SPAN, $pref);

  $pref = proon_to_pref($p[PROON_F_COUNTS]);
  if ($pref == '') $pref = PROON_PREF_BLANK;
  setPref($data_dir, $username, PROON_O_F_COUNT_SPAN, $pref);

  $pref = proon_to_pref($p[PROON_F_UNSEENS]);
  if ($pref == '') $pref = PROON_PREF_BLANK;
  setPref($data_dir, $username, PROON_O_F_TOSS_UNSEEN, $pref);

  $proonopts = $p[PROON_P_OPTIONS];

  $opt = PROON_O_U_PRUNE_INTERVAL;
  $pref = (isset($proonopts[$opt])) ? $proonopts[$opt] : PROON_PREF_BLANK;
  setPref($data_dir, $username, $opt, $pref);

  $opt = PROON_O_U_VIA_TRASH;
  $pref = (isset($proonopts[$opt])) ? $proonopts[$opt] : PROON_PREF_BLANK;
  setPref($data_dir, $username, $opt, $pref);

  $opt = PROON_O_U_TRASH_TIME;
  $pref = (isset($proonopts[$opt])) ? $proonopts[$opt] : PROON_PREF_BLANK;
  setPref($data_dir, $username, $opt, $pref);

  $opt = PROON_O_U_UNSUBSCRIBED;
  $pref = (isset($proonopts[$opt])) ? $proonopts[$opt] : PROON_PREF_BLANK;
  setPref($data_dir, $username, $opt, $pref);

  $opt = PROON_O_U_DIS_DATE;
  $pref = (isset($proonopts[$opt])) ? $proonopts[$opt] : PROON_PREF_BLANK;
  setPref($data_dir, $username, $opt, $pref);

  $opt = PROON_O_U_DIS_SIZE;
  $pref = (isset($proonopts[$opt])) ? $proonopts[$opt] : PROON_PREF_BLANK;
  setPref($data_dir, $username, $opt, $pref);

  $opt = PROON_O_U_DIS_COUNT;
  $pref = (isset($proonopts[$opt])) ? $proonopts[$opt] : PROON_PREF_BLANK;
  setPref($data_dir, $username, $opt, $pref);

  $opt = PROON_O_U_SC_ORDER;
  $pref = (isset($proonopts[$opt])) ? $proonopts[$opt] : PROON_PREF_BLANK;
  setPref($data_dir, $username, $opt, $pref);

  $opt = PROON_O_U_MESSAGE_AFTER;
  $pref = (isset($proonopts[$opt])) ? $proonopts[$opt] : PROON_PREF_BLANK;
  setPref($data_dir, $username, $opt, $pref);

  $opt = PROON_O_U_SCREEN_AFTER;
  $pref = (isset($proonopts[$opt])) ? $proonopts[$opt] : PROON_PREF_BLANK;
  setPref($data_dir, $username, $opt, $pref);

  $opt = PROON_O_U_DIS_COLOR;
  $pref = (isset($proonopts[$opt])) ? $proonopts[$opt] : PROON_PREF_BLANK;
  setPref($data_dir, $username, $opt, $pref);

  $opt = PROON_O_U_PRUNE_LINK;
  $pref = (isset($proonopts[$opt])) ? $proonopts[$opt] : PROON_PREF_BLANK;
  setPref($data_dir, $username, $opt, $pref);

}

/**
 * Fetch all proon-related items from the user's preferences store.
 */
function proon_get_prefs() {
  global $data_dir, $username;
  $saw_pref_count = 0; // not used, but tells if at least one pref was read

  $blob = getPref($data_dir, $username, PROON_O_F_DATE_SPAN);
  if ($blob) ++$saw_pref_count;
  $p[PROON_F_DATES] = ($blob  &&  $blob != PROON_PREF_BLANK) ? proon_from_pref($blob) : array();

  $blob = getPref($data_dir, $username, PROON_O_F_SIZE_SPAN);
  if ($blob) ++$saw_pref_count;
  $p[PROON_F_SIZES] = ($blob  &&  $blob != PROON_PREF_BLANK) ? proon_from_pref($blob) : array();

  $blob = getPref($data_dir, $username, PROON_O_F_COUNT_SPAN);
  if ($blob) ++$saw_pref_count;
  $p[PROON_F_COUNTS] = ($blob  &&  $blob != PROON_PREF_BLANK) ? proon_from_pref($blob) : array();

  $blob = getPref($data_dir, $username, PROON_O_F_TOSS_UNSEEN);
  if ($blob) ++$saw_pref_count;
  $p[PROON_F_UNSEENS] = ($blob  &&  $blob != PROON_PREF_BLANK) ? proon_from_pref($blob) : array();

  $dateorsizeorcountlist = array();
  foreach ($p[PROON_F_DATES] as $folder => $value) {
    $dateorsizeorcountlist[$folder] = $value;
  }
  foreach ($p[PROON_F_SIZES] as $folder => $value) {
    $dateorsizeorcountlist[$folder] = $value;
  }
  ksort($dateorsizeorcountlist);  // cosmetic
  $p[PROON_F_DATE_OR_SIZE_OR_COUNT] = $dateorsizeorcountlist;
  

  $proonopts = array();

  $opt = PROON_O_U_PRUNE_INTERVAL;
  $val = getPref($data_dir, $username, $opt);
  if ($val) ++$saw_pref_count;
  if ($val  &&  $val != PROON_PREF_BLANK) $proonopts[$opt] = $val;

  $opt = PROON_O_U_VIA_TRASH;
  $val = getPref($data_dir, $username, $opt);
  if ($val) ++$saw_pref_count;
  if ($val  &&  $val != PROON_PREF_BLANK) $proonopts[$opt] = $val;

  $opt = PROON_O_U_TRASH_TIME;
  $val = getPref($data_dir, $username, $opt);
  if ($val) ++$saw_pref_count;
  if ($val  &&  $val != PROON_PREF_BLANK) $proonopts[$opt] = $val;

  $opt = PROON_O_U_UNSUBSCRIBED;
  $val = getPref($data_dir, $username, $opt);
  if ($val) ++$saw_pref_count;
  if ($val  &&  $val != PROON_PREF_BLANK) $proonopts[$opt] = $val;

  $opt = PROON_O_U_DIS_DATE;
  $val = getPref($data_dir, $username, $opt);
  if ($val) ++$saw_pref_count;
  if ($val  &&  $val != PROON_PREF_BLANK) $proonopts[$opt] = $val;

  $opt = PROON_O_U_DIS_SIZE;
  $val = getPref($data_dir, $username, $opt);
  if ($val) ++$saw_pref_count;
  if ($val  &&  $val != PROON_PREF_BLANK) $proonopts[$opt] = $val;

  $opt = PROON_O_U_DIS_COUNT;
  $val = getPref($data_dir, $username, $opt);
  if ($val) ++$saw_pref_count;
  if ($val  &&  $val != PROON_PREF_BLANK) $proonopts[$opt] = $val;

  $opt = PROON_O_U_SC_ORDER;
  $val = getPref($data_dir, $username, $opt);
  if ($val) ++$saw_pref_count;
  if ($val  &&  $val != PROON_PREF_BLANK) $proonopts[$opt] = $val;

  $opt = PROON_O_U_MESSAGE_AFTER;
  $val = getPref($data_dir, $username, $opt);
  if ($val) ++$saw_pref_count;
  if ($val  &&  $val != PROON_PREF_BLANK) $proonopts[$opt] = $val;

  $opt = PROON_O_U_SCREEN_AFTER;
  $val = getPref($data_dir, $username, $opt);
  if ($val) ++$saw_pref_count;
  if ($val  &&  $val != PROON_PREF_BLANK) $proonopts[$opt] = $val;

  $opt = PROON_O_U_DIS_COLOR;
  $val = getPref($data_dir, $username, $opt);
  if ($val) ++$saw_pref_count;
  if ($val  &&  $val != PROON_PREF_BLANK) $proonopts[$opt] = $val;

  $opt = PROON_O_U_PRUNE_LINK;
  $val = getPref($data_dir, $username, $opt);
  if ($val) ++$saw_pref_count;
  if ($val  &&  $val != PROON_PREF_BLANK) $proonopts[$opt] = $val;

  $p[PROON_P_OPTIONS]  = $proonopts;
  $p[PROON_P_PREFS_SEEN] = $saw_pref_count;
  
  return $p;
}

/**
 * Read preference values from storage and compare to copies
 * received from form submission.  If any don't exactly match,
 * even if logically equivalent, the memory copy is considered
 * dirty.
 *
 * @return boolean dirty flag
 */
function proon_is_dirty(&$p) {
  $pref_p = proon_get_prefs();
  if (proon_arrays_differ($p[PROON_F_DATES], $pref_p[PROON_F_DATES])) return true;
  if (proon_arrays_differ($p[PROON_F_SIZES], $pref_p[PROON_F_SIZES])) return true;
  if (proon_arrays_differ($p[PROON_F_COUNTS], $pref_p[PROON_F_COUNTS])) return true;
  if (proon_arrays_differ($p[PROON_F_UNSEENS], $pref_p[PROON_F_UNSEENS])) return true;
  return proon_arrays_differ($p[PROON_P_OPTIONS], $pref_p[PROON_P_OPTIONS]);
}

function proon_log_delta($tag, $t1, $t2 = null, $folder = null) {
  if ($t2 == null) $t2 = microtime();
  list($u1, $s1) = explode(' ', $t1);
  list($u2, $s2) = explode(' ', $t2);
  $delta = (($s2-$s1) + ($u2-$u1));
  $delta = substr($delta, 0, 8); // ought to be enough precision for any geek
  proon_log($delta.' '.$tag, $folder);
}

function proon_log($s, $folder = null) {
  global $proon_log_file;
  if ($proon_log_file != null) {
    $l = fopen($proon_log_file, 'a+');
    $f = ($folder ? "[$folder] " : '');
    fputs($l, date('Y-m-d H:i:s O').' ['.$_SESSION['username'].'] '.$f); fputs($l, $s); fputs($l, "\n");
    fflush($l);
  }
}

function proon_calc_report_widths(&$p, &$outcome, $verbose) {
  $w = array();
  $w['duelabel'] = max(strlen(PROON_UI_DATE_SPAN), 
		       strlen(PROON_UI_SIZE_SPAN), 
		       strlen(PROON_UI_COUNT_SPAN),
		       strlen(PROON_UI_UNSEEN_TOO));
  $w['fname'] = strlen(PROON_UI_FOLDER);
  $w['spanval'] = 0;
  $w['total'] = 0;
  foreach ($outcome as $f => $an) {
    $w['fname'] = max($w['fname'], strlen($f));
    $d = isset($p[PROON_F_DATES][$f]) ? $p[PROON_F_DATES][$f] : '';
    $s = isset($p[PROON_F_SIZES][$f]) ? $p[PROON_F_SIZES][$f] : '';
    $c = isset($p[PROON_F_COUNTS][$f]) ? $p[PROON_F_COUNTS][$f] : '';
    $u = isset($p[PROON_F_UNSEENS][$f]) ? _("yes") : '';
    $w['spanval'] = max($w['spanval'], strlen($d), strlen($s), strlen($c), strlen($u));
    $dp = $an[PROON_OUTCOME_DUE_TO_DATE];
    $sp = $an[PROON_OUTCOME_DUE_TO_SIZE];
    $cp = $an[PROON_OUTCOME_DUE_TO_COUNT];
    $tot_pruned = $dp + $sp + $cp;
    $w['total'] += $tot_pruned;
  }  
  return $w;
}

function proon_format_outcome(&$p, &$outcome, $verbose) {
  $w = proon_calc_report_widths($p, $outcome, $verbose);
  if (! $verbose  &&  $w['total'] < 1) return null;
  $format_f = '%-'.$w['fname'].'s   %s = %s - %s   %s'."\n\n";
  $format_p = '    %'.$w['duelabel'].'s [%'.$w['spanval'].'s]: %3s'."\n";
  $report = null;
  $did_legend = false;
  foreach ($outcome as $f => $f_out) {
    $d = isset($p[PROON_F_DATES][$f]) ? $p[PROON_F_DATES][$f] : '';
    $s = isset($p[PROON_F_SIZES][$f]) ? $p[PROON_F_SIZES][$f] : '';
    $c = isset($p[PROON_F_COUNTS][$f]) ? $p[PROON_F_COUNTS][$f] : '';
    $u = isset($p[PROON_F_UNSEENS][$f]) ? $p[PROON_F_UNSEENS][$f] : '';
    $dp = $f_out[PROON_OUTCOME_DUE_TO_DATE];
    $sp = $f_out[PROON_OUTCOME_DUE_TO_SIZE];
    $cp = $f_out[PROON_OUTCOME_DUE_TO_COUNT];
    $tot_pruned = $dp + $sp + $cp;
    $err = (isset($f_out[PROON_OUTCOME_ERROR]) ? $f_out[PROON_OUTCOME_ERROR] : '');
    if (! $verbose  &&  ! $err  &&  ($tot_pruned < 1)) continue;
    $span_value_width = max(strlen($d), strlen($s), strlen($c));
    $tot_before = $f_out[PROON_OUTCOME_TOTAL_BEFORE];
    if (! $did_legend) {
      $report .= sprintf($format_f, PROON_UI_FOLDER.':', _("Remainder"), _("Before"), _("Pruned"), '');
      $did_legend = true;
    }
    $report .= sprintf($format_f, $f.':', ($tot_before - $tot_pruned), $tot_before, $tot_pruned, $err);
    if ($d !== ''  &&  $d !== null) {
      $report .= sprintf($format_p, PROON_UI_DATE_SPAN, $d, $dp);
    }
    if ($s !== ''  &&  $s !== null) {
      $report .= sprintf($format_p, PROON_UI_SIZE_SPAN, $s, $sp);
    }
    if ($c !== ''  &&  $c !== null) {
      $report .= sprintf($format_p, PROON_UI_COUNT_SPAN, $c, $cp);
    }
    if ($u !== ''  &&  $u !== null) {
      $report .= sprintf($format_p, PROON_UI_UNSEEN_TOO, _("yes"), '');
    }
    $report .= "\n\n";
  }
  return $report;
}

function proon_format_headers(&$p, &$outcome, $verbose) {
  $w = proon_calc_report_widths($p, $outcome, $verbose);
  include_once(SM_PATH . 'class/mime/Rfc822Header.class.php');
  $hdr = new Rfc822Header();
  // I'd prefer to have a '<>' from address, but SM (as of 1.4.4) doesn't
  // fare well with those when they are displayed.
  global $proon_site_from_address;
  $f = 'From: '.$proon_site_from_address."\r\n";
  $s = 'Subject: '._("Messages automatically pruned:").' '.$w['total']."\r\n";
  $hdr->parseHeader($f . $s);
  return $hdr;
}

/**
 * This just abstracts getting the list of mailboxes since we need it
 * in a couple of unrelated places and don't want to bother to get it
 * more than once.
 */
function proon_mailbox_list(&$p, &$imapStr) {
  proon_login($imapStr);
  if (! isset($p[PROON_P_OPTIONS][PROON_P_MAILBOXES])) {
    $p[PROON_P_MAILBOXES] = sqimap_mailbox_list($imapStr);
    foreach($p[PROON_P_MAILBOXES] as $mbox) {
      $name = $mbox['unformatted-disp'];
      $p[PROON_P_BOXNAMES][$name] = true;
    }
  }
  return $p[PROON_P_MAILBOXES];
}

function proon_is_subscribed(&$p, &$target, $imapStr) {
  proon_mailbox_list($p, $imapStr);
  return (isset($p[PROON_P_BOXNAMES][$target]));
}

?>
