<?php
/**
 * Functions used by ldapquery plugin.
 * @copyright 2005-2006 The SquirrelMail Project Team
 * @license http://opensource.org/licenses/gpl-license.php GNU Public License
 * @version $Id: functions.php,v 1.15 2006/04/10 06:13:04 tokul Exp $
 * @package sm-plugins
 * @subpackage ldapquery
 */

/**
 * SM_PATH define
 * @ignore
 */
if (!defined('SM_PATH'))  {
    define('SM_PATH','../../');
}

/* load class */
include_once(SM_PATH . 'plugins/ldapquery/class.schema.php');

/* load forms.php */
include_once(SM_PATH . 'functions/forms.php');
/* error_box() function */
include_once(SM_PATH . 'functions/display_messages.php');

global $ldq_authreqd, $ldq_attributes, $ldq_searchobjs, $ldq_searchattrs, $ldq_allow_asterisk_search, $ldq_debug;
/* load default config */
if (file_exists(SM_PATH . 'plugins/ldapquery/config_default.php')) {
    include_once(SM_PATH . 'plugins/ldapquery/config_default.php');
} else {
    // somebody removed default config.
    $ldq_authreqd = false;
    $ldq_attributes = array ('cn','mail');
    $ldq_searchobjs = array ('person','*');
    $ldq_searchattrs = array ('cn','mail');
    $ldq_allow_asterisk_search=true;
    $ldq_debug=false;
}

/** Including site config */
if (file_exists(SM_PATH . 'config/ldapquery_config.php')) {
    include_once(SM_PATH . 'config/ldapquery_config.php');
} elseif (file_exists(SM_PATH . 'plugins/ldapquery/config.php')) {
    include_once(SM_PATH . 'plugins/ldapquery/config.php');
}

/** lowercase ldap attributes */
array_walk($ldq_attributes,'ldq_arraytolower');
array_walk($ldq_searchobjs,'ldq_arraytolower');
array_walk($ldq_searchattrs,'ldq_arraytolower');

/** array callback functions */

/**
 * lowercase elements of array
 * callback function
 * @param string $value array value
 * @param string $key array key
 */
function ldq_arraytolower(&$value,&$key) {
    $value=strtolower($value);
    $key=strtolower($key);
}

/**
 * trim values of array
 * callback function
 * @param string $value array value
 * @param string $key array key
 * @since 2.1
 */
function ldq_arraytrim(&$value,&$key) {
    $value=trim($value);
}

/* internal functions for setup.php */

/**
 * Show version info
 * (internal function)
 * @return string version info
 */
function ldapquery_version_function() {
    return '2.1';
}

/**
 * Adds entry to squirrelmail menu line
 * (internal function)
 */
function ldapquery_menuline_function() {
    // switch to ldapquery domain
    bindtextdomain('ldapquery',SM_PATH . 'locale');
    textdomain('ldapquery');

    displayInternalLink ("plugins/ldapquery/ldapquery.php", _("Directory"), "right");
    echo "&nbsp;&nbsp;\n";

    // revert to sm domain
    bindtextdomain('squirrelmail',SM_PATH . 'locale');
    textdomain('squirrelmail');
}

/**
 * Add options block
 * (internal function)
 */
function ldapquery_options_function() {
    global $optpage_blocks;

    bindtextdomain('ldapquery',SM_PATH . 'locale');
    textdomain('ldapquery');

    $optpage_blocks[] = array(
        'name' => _("Directory Preferences"),
        'url'  => SM_PATH . 'plugins/ldapquery/options.php',
        'desc' => _("These settings control what type of output you will see when you use the Directory link to search Directory Services."),
        'js'   => false
        );

    bindtextdomain('squirrelmail',SM_PATH . 'locale');
    textdomain('squirrelmail');
}

/**
 * Save options
 * (internal function)
 */
function ldapquery_save_function() {
    global $username, $data_dir;
    global $ldq_attributes;

    include_once (SM_PATH . 'plugins/ldapquery/functions.php');

    if (sqgetGlobalVar('submit_ldapquery',$ldq_submit,SQ_POST)) {
        if (! sqgetGlobalVar('ldapquery_output_type',$ldapquery_output_type,SQ_POST))
            $ldapquery_output_type='OneTable';

        foreach ($ldq_attributes as $attr) {
            $Var = 'ldapquery_showattr_'.$attr;
            if (! sqgetGlobalVar($Var,$$Var,SQ_POST))
                $$Var = 'off';

            setPref($data_dir, $username, $Var, $$Var);
        }

        setPref($data_dir, $username, 'ldapquery_output_type',$ldapquery_output_type);
    }
}

/**
 * Set optpage_name displayed when settings are saved
 * (internal function)
 */
function ldapquery_set_loadinfo_function() {
    global $optpage, $optpage_name;
    if ($optpage=='ldapquery') {
        bindtextdomain('ldapquery',SM_PATH . 'locale');
        textdomain('ldapquery');
        $optpage_name = _("Directory Preferences");
        bindtextdomain('squirrelmail',SM_PATH . 'locale');
        textdomain('squirrelmail');
    }
}

/**
 * Load plugin preferences
 * (internal function)
 */
function ldapquery_prefs_function() {
    global $username, $data_dir;
    global $ldq_attributes;

    global $ldapquery_output_type;
    $ldapquery_output_type = getPref($data_dir, $username, 'ldapquery_output_type','OneTable');

    foreach ($ldq_attributes as $attr) {
        $Var = "ldapquery_showattr_" . $attr;
        global $$Var;
        $$Var = getPref($data_dir, $username,$Var,'on');
    }
}

/**
 * compare attributes for sorting results
 * @param mixed $left
 * @param mixed $right
 * @return mixed
 */
function ldapquery_compattrs ($left, $right) {
    return strcasecmp($left,$right);
}

/** 
 * Convert a string with a name in it in firstname first
 * format to a lastname first formatted string
 * @param mixed $fnf
 * @return mixed
 */
function ldapquery_lnf ($fnf) {
    $name = explode(" ", $fnf);
    if (count($name) > 1)
        $lnf = $name[count($name)-1];
    else
        $lnf = "";
    for ($i = 0; $i < count($name)-1; $i++)
        $lnf .= $name[$i];
    return $lnf;
}

/**
 * compare cn attributes for sorting results by lastname
 * This is a bit of a hack for those of us who don't have surname
 * attributes in our LDAP records (so we can't just select it in
 * "sort by".
 * @param mixed $left
 * @param mixed $right
 * @return mixed
 */
function ldapquery_compcns ($left, $right) {
    // find last names and put it at start of strings
    $left_lnf = ldapquery_lnf ($left);
    $right_lnf = ldapquery_lnf ($right);
    
    return strcasecmp($left_lnf,$right_lnf);
}

/**
 * ldapquery_dispresultsMulti - display HTML results from an LDAP query
 *   takes the list of attributes that were searched for and the
 *   returned results as parameters
 * @param mixed $attributes
 * @param mixed $entry
 * @param mixed $sortby
 * @param string $charset charset used in entry data (since v.2.0)
 * @return mixed
 */
function ldapquery_dispresultsMulti ($attributes, $entry, $sortby, $charset='utf-8') {
    global $ldq_loaded_schemas;
    // sort the entries. First, build an array with the sortby as the key
    // and the entry index as the value
    global $compose_new_win, $base_uri;
    if ($entry["count"] > 0) {
        for ($i=0 ; $i < $entry["count"]; $i++) {
            $Val = $entry[$i][$sortby][0] . $i;
            $sorted[$Val] = $i;
        }
        if ($sortby == "cn")
            uksort ($sorted, "ldapquery_compcns");
        else
            uksort ($sorted, "ldapquery_compattrs");

        foreach ($sorted as $key=>$i) {
            echo '<table border="border">';
   
            foreach ($attributes as $attr) {
                $Var = "ldapquery_showattr_" . $attr;
                global $$Var;
                if ($$Var == "on") {
                    if (isset($entry[$i][$attr])) {
                        echo "<tr>\n";
                        echo '<td>' .$ldq_loaded_schemas->get_attrib_name($attr)."</td><td>";
                        for ($x=0 ; $x < $entry[$i][$attr]["count"] ; $x++) {
                            echo ldapquery_format_results($attr,$entry[$i][$attr][$x],$charset);
                        }
                        echo "</td></tr>\n";
                    } else {
                        echo "<tr>\n";
                        echo '<td>' . $ldq_loaded_schemas->get_attrib_name($attr) . "</td>";
                        echo "<td>&nbsp;</td></tr>";
                    }
                }
            }
            echo "</table>\n";
            echo ldapquery_abook_form($entry[$i],false);

            echo '<hr />';
        }
    } else {
        echo _("No records found") . "<br />\n";
    }
}

/**
 * display HTML results from an LDAP query
 *   with each record being one row in a single table
 *   takes the list of attributes that were searched for and the
 *   returned results as parameters
 */
function ldapquery_dispresultsSingle ($attributes, $entry, $sortby, $charset='utf-8') {
    global $ldq_loaded_schemas;
    // sort the entries. First, build an array with the sortby as the key
    // and the entry index as the value
    if ($entry["count"] > 0) {
        for ($i=0 ; $i < $entry["count"]; $i++) {
            $Val = $entry[$i][$sortby][0] . $i;
            $sorted[$Val] = $i;
        }
        if ($sortby == "cn")
            uksort ($sorted, "ldapquery_compcns");
        else
            uksort ($sorted, "ldapquery_compattrs");

        echo '<table border="border">';

        // print table headers
        echo '<tr>';
        foreach ($attributes as $attr) {
            $Var = "ldapquery_showattr_" . $attr;
            global $$Var;
            if ($$Var == "on") {
                echo '<td><b>' . $ldq_loaded_schemas->get_attrib_name($attr) . "</b></td>";
            }
        }
        echo "<td>&nbsp;</td>\n";
        echo "</tr>\n";

        foreach ($sorted as $key=>$i) {
            foreach ($attributes as $attr) {
                $Var = "ldapquery_showattr_" . $attr;
                global $$Var;
                if ($$Var == "on") {
                    echo "<td>";

                    if (!isset( $entry[$i][$attr]) || $entry[$i][$attr]["count"] == 0) {
                        echo "<br />\n";  // to avoid totally empty cells
                    }

                    if (isset($entry[$i][$attr])) {
                        for ($x=0 ; $x < $entry[$i][$attr]["count"] ; $x++) {
                            // Since there might be multiple values per attribute,
                            // prefix each one with <p> so they're on separate lines
                            // within the cell
                            echo "<p>";

                            echo ldapquery_format_results($attr,$entry[$i][$attr][$x],$charset);
                        }
                    }
                    echo '</td>';
                }
            }
            echo '<td>';
            echo ldapquery_abook_form($entry[$i]);
            echo '</td>';
            echo "</tr>\n";
        }
        echo "</table>\n";
    } else {
        echo _("No records found") . "<br />\n";
    }
}

/**
 * Formats and sanitizes ldap results.
 *
 * Provides different handling for labeleduri and mail values.
 * @param string $attr attribute name
 * @param string $attr_value attribute value
 * @param string $charset charset used by ldap server
 * @return string html formated and sanitized string
 */
function ldapquery_format_results($attr,$attr_value,$charset='utf-8') {
    $ret_result='';

    switch ($attr) {
    case "labeledurl":
    case "labeleduri":
        // split up url and label parts
        $val = charset_decode($charset,$attr_value) . " <END>";
        $isfirsttoken = true;
        for ($token = strtok($val, " "); $token != "<END>";
             $token = strtok(" "))
            {
                if ($isfirsttoken) {
                    $ret_result.= '<a href="' . $token . '" target="_blank">';
                }
                $ret_result.=$token;
                if ($isfirsttoken) {
                    $ret_result.="</a> \n";
                    $isfirsttoken = false;
                }
            }
        break;
    case "mail":
        // FIXME: add email address format check and other email attributes
        $val = htmlspecialchars($attr_value);
        $ret_result.=makeComposeLink('src/compose.php?send_to='.$val,$val);
        break;
    default:
        $val = charset_decode($charset,$attr_value) . "$<END>";
        for ($token = strtok($val, "\$");
             $token != "<END>"; $token = strtok("\$"))
            {
                $ret_result.=$token . "<br />\n";
            }
    }
    return $ret_result;
}

/**
 * Create html option tags
 * @param string $Var
 * @param string $value
 * @param string $Desc
 */
function ldapquery_ShowOption($Var, $value, $Desc) {
    $Var = 'ldapquery_' . $Var;
       
    global $$Var;
       
    echo '<option value="' . $value . '"';
    if ($$Var == $value)
        {
            echo ' selected';
        }
    echo '>' . $Desc . "</option>\n";
}

/**
 * Create checkboxes
 * @param array $attributes
 */
function ldapquery_ShowCheckboxes($attributes) {
    global $ldq_loaded_schemas;
    foreach ($attributes as $attr) {
        $Var = "ldapquery_showattr_" . $attr;
        global $$Var;
        echo '<tr><td>' . $ldq_loaded_schemas->get_attrib_name($attr) . "</td><td><input type=\"checkbox\" name=" . $Var;
        if ($$Var == "on")
            echo ' checked';
        echo "></td></tr>\n";
    }
}

/**
 * Reads data about supported ldap schemas and loads it into object.
 * @return object
 */
function ldq_load_schemas() {
    $ldq_attrib=array();
    $ldq_object=array();

    if (file_exists(SM_PATH . 'plugins/ldapquery/schema/core.php'))
        include_once(SM_PATH . 'plugins/ldapquery/schema/core.php');
    if (file_exists(SM_PATH . 'plugins/ldapquery/schema/cosine.php'))
        include_once(SM_PATH . 'plugins/ldapquery/schema/cosine.php');
    if (file_exists(SM_PATH . 'plugins/ldapquery/schema/inetorgperson.php'))
        include_once(SM_PATH . 'plugins/ldapquery/schema/inetorgperson.php');
    // add custom schema, if it exists.
    if (file_exists(SM_PATH . 'plugins/ldapquery/schema/custom.php'))
        include_once(SM_PATH . 'plugins/ldapquery/schema/custom.php');

    $ret = new ldq_schemas();

    foreach ($ldq_attrib as $attrib => $data) {
        $ret->add_attrib($attrib,$data);
    }

    foreach ($ldq_object as $object => $data) {
        $ret->add_object($object,$data);
    }

    return $ret;
}

/**
 * Sanitizes ldap query strings.
 * See rfc2254
 * @link http://www.faqs.org/rfcs/rfc2254.html
 * @param string $string
 * @return string sanitized string
 */
function ldapquery_ldapspecialchars($string) {
    $sanitized=array('\\' => '\5c',
                     '*' => '\2a',
                     '(' => '\28',
                     ')' => '\29',
                     "\x00" => '\00');

    return str_replace(array_keys($sanitized),array_values($sanitized),$string);
}

/**
 * Creates address book form.
 * All data is in hidden fields. Only button is visible.
 * @param array $entry single record from ldap search results.
 * @param boolean $smallsize use short string in button name
 * @return string html formated address book form.
 */
function ldapquery_abook_form($entry,$smallsize=true) {
    $ret = '';
    // check if $entry contains email address
    if (isset($entry['mail']) || isset($entry['email']) || isset($entry['rfc822mailbox'])) {
        $ret.= "<!-- start abook form -->\n";
        // start form
        $ret.= addForm(sqm_baseuri() . 'plugins/ldapquery/addressbook.php');
        // add nickname field (remove all spaces from entries)
        if (isset($entry['commonname']['count']) && $entry['commonname']['count']>0 && isset($entry['commonname'][0])) {
            $ret.= addHidden('addaddr[nickname]',str_replace(' ','_',$entry['commonname'][0]));
        } elseif (isset($entry['cn']['count']) && $entry['cn']['count']>0 && isset($entry['cn'][0])) {
            $ret.= addHidden('addaddr[nickname]',str_replace(' ','_',$entry['cn'][0]));
        } else {
            $ret.= addHidden('addaddr[nickname]','');
        }
        // add email field
        if (isset($entry['mail']['count']) && $entry['mail']['count']>0 && isset($entry['mail'][0])) {
            $ret.= addHidden('addaddr[email]',$entry['mail'][0]);
        } elseif (isset($entry['email']['count']) && $entry['email']['count']>0 && isset($entry['email'][0])) {
            $ret.= addHidden('addaddr[email]',$entry['email'][0]);
        } elseif (isset($entry['rfc822mailbox']['count']) && $entry['rfc822mailbox']['count']>0 && isset($entry['rfc822mailbox'][0])) {
            $ret.= addHidden('addaddr[email]',$entry['rfc822mailbox'][0]);
        } else {
            $ret.= addHidden('addaddr[email]','');
        }
        // add first name
        if (isset($entry['givenname']['count']) && $entry['givenname']['count']>0 && isset($entry['givenname'][0])) {
            $ret.= addHidden('addaddr[firstname]',$entry['givenname'][0]);
        } elseif (isset($entry['gn']['count']) && $entry['gn']['count']>0 && isset($entry['gn'][0])) {
            $ret.= addHidden('addaddr[firstname]',$entry['gn'][0]);
        } else {
            $ret.= addHidden('addaddr[firstname]','');
        }
        // add surname
        if (isset($entry['surname']['count']) && $entry['surname']['count']>0 && isset($entry['surname'][0])) {
            $ret.= addHidden('addaddr[lastname]',$entry['surname'][0]);
        } elseif (isset($entry['sn']['count']) && $entry['sn']['count']>0 && isset($entry['sn'][0])) {
            $ret.= addHidden('addaddr[lastname]',$entry['sn'][0]);
        } else {
            $ret.= addHidden('addaddr[lastname]','');
        }
        // add info
        if (isset($entry['labeleduri']['count']) && $entry['labeleduri']['count']>0 && isset($entry['labeleduri'][0])) {
            // use labeleduri
            $ret.= addHidden('addaddr[label]',$entry['labeleduri'][0]);
        } elseif (isset($entry['phone']['count']) && $entry['phone']['count']>0 && isset($entry['phone'][0])) {
            // use phone
            $ret.= addHidden('addaddr[label]',$entry['phone'][0]);
        } elseif (isset($entry['telephonenumber']['count']) && 
                  $entry['telephonenumber']['count']>0 && 
                  isset($entry['telephonenumber'][0])) {
            // use telephonenumber
            $ret.= addHidden('addaddr[label]',$entry['telephonenumber'][0]);
        } elseif (isset($entry['mobile']['count']) && 
                  $entry['mobile']['count']>0 && 
                  isset($entry['mobile'][0])) {
            // use mobile
            $ret.= addHidden('addaddr[label]',$entry['mobile'][0]);
        } else {
            $ret.= addHidden('addaddr[label]','');
        }
        if ($smallsize) {
            $ret.= addSubmit(_("Save"));
        } else {
            $ret.= addSubmit(_("Add to address book"));
        }
        // save search
        $ret.= ldq_hidden_search_fields();

        $ret.= "</form>\n";
        $ret.= "<!-- end abook form -->\n";
    } else {
        $ret.="<!-- email address info is missing. no 'add to address book' form. -->\n";
    }
    return $ret;
}

/**
 * Checks if SquirrelMail configuration and PHP setup are correct.
 * @return string error message or empty string.
 * @since 2.1
 */
function ldq_check_setup() {
    global $ldap_server;

    if (!extension_loaded('ldap')) {
        return _("Plugin requires PHP with LDAP support.");
    } elseif (empty($ldap_server)) {
        return _("LDAP address books are not configured in SquirrelMail configuration.");
    }
    return '';
}

/**
 * Creates hidden form fields from POST data
 * @return string
 * @since 2.1
 */
function ldq_hidden_search_fields() {
    $ret = '';
    if (sqgetGlobalVar('ldq_searchfor',$ldq_searchfor,SQ_POST)) {
        $ret.= addHidden('ldq_searchfor',$ldq_searchfor);
    }
    if (sqgetGlobalVar('ldq_searchby',$ldq_searchby,SQ_POST)) {
        $ret.= addHidden('ldq_searchby',$ldq_searchby);
    }
    if (sqgetGlobalVar('ldq_comparetype',$ldq_comparetype,SQ_POST)) {
        $ret.= addHidden('ldq_comparetype',$ldq_comparetype);
    }
    if (sqgetGlobalVar('ldq_querystr',$ldq_querystr,SQ_POST)) {
        $ret.= addHidden('ldq_querystr',$ldq_querystr);
    }
    if (sqgetGlobalVar('ldq_sortby',$ldq_sortby,SQ_POST)) {
        $ret.= addHidden('ldq_sortby',$ldq_sortby);
    }
    if (sqgetGlobalVar('ldq_rdn',$ldq_rdn,SQ_POST)) {
        $ret.= addHidden('ldq_rdn',$ldq_rdn);
    }
    if (sqgetGlobalVar('ldq_pass',$ldq_pass,SQ_POST)) {
        $ret.= addHidden('ldq_pass',$ldq_pass);
    }
    if (sqgetGlobalVar('ldq_server',$ldq_server,SQ_POST)) {
        $ldq_server = (int) $ldq_server;
        $ret.= addHidden('ldq_server',$ldq_server);
    }
    return $ret;
}

/**
 * Wrapper arround standard error_box function.
 *
 * Function is used to create error boxes that are centered and
 * does not take whole page width. It also takes switches gettext 
 * domains before calling error_box() function.
 * @since 2.1
 */
function ldq_error_box($sError) {
    global $color;

    // switch to squirrelmail domain
    bindtextdomain('squirrelmail',SM_PATH . 'locale');
    textdomain('squirrelmail');

    echo '<table align="center"><tr><td>';
    error_box($sError,$color);
    echo '</td></tr></table>';

    // switch to ldapquery domain
    bindtextdomain('ldapquery',SM_PATH . 'locale');
    textdomain('ldapquery');
}
?>