<?php

/**
  * SquirrelMail Gzip Compressed Output Plugin
  * Copyright (C) 2002-2004 Tyler Akins, Paul Lesneiwski <pdontthink@angrynerds.com>
  * This program is licensed under GPL. See COPYING for details
  *
  */


/** 
  * Shows user configuration options
  *
  */
function gzip_options_do()
{

   global $gzip_minimum_size_default, $gzip_allow_overrides, $gzip_enable_default;

   include_once(SM_PATH . 'plugins/gzip/config.php');


   // bail if admin is disallowing user settings
   //
   if (!$gzip_allow_overrides) return;


   global $data_dir, $username;
   $gzip_enable = getPref($data_dir, $username, 'gzip_enable', $gzip_enable_default);
   $gzip_size = getPref($data_dir, $username, 'gzip_size', $gzip_minimum_size_default);

   bindtextdomain('gzip', SM_PATH . 'plugins/gzip/locale');
   textdomain('gzip');

   global $optpage_data;
   $optpage_data['grps']['plugin_gzip'] = _("Output Compression");
   $optionValues = array();
   $optionValues[] = array(
      'name'          => 'gzip_enable',
      'caption'       => _("Compressed Output"),
      'type'          => SMOPT_TYPE_BOOLEAN,
      'initial_value' => $gzip_enable,
      'refresh'       => SMOPT_REFRESH_NONE,
   );
   $optionValues[] = array(
      'name'          => 'gzip_size',
      'caption'       => (check_sm_version(1, 4, 4) ? _("Compress if Page Size is Larger Than")
                                                    : _("Compress if Page Size is Larger Than (KB)")),
      'trailing_text' => '(KB)',
      'type'          => SMOPT_TYPE_INTEGER,
      'initial_value' => $gzip_size,
      'refresh'       => SMOPT_REFRESH_NONE,
      'size'          => SMOPT_SIZE_TINY,
   );
   $optpage_data['vals']['plugin_gzip'] = $optionValues;

   bindtextdomain('squirrelmail', SM_PATH . 'locale');
   textdomain('squirrelmail');

}



/** 
  * Determine if browser supports gzip compression and if 
  * compression is enabled in SM settings, if so, start
  * output buffering and register callback 
  *
  */
function gzip_start_do()
{

   global $data_dir, $username, $gzip_enable_default, $gzip_allow_overrides;

   include_once(SM_PATH . 'plugins/gzip/config.php');

   $gzip_enable = getPref($data_dir, $username, 'gzip_enable', $gzip_enable_default);
   if (!$gzip_allow_overrides) $gzip_enable = $gzip_enable_default;


   if (!$gzip_enable) return;


   // get browser accept methods
   //
   sqgetGlobalVar('HTTP_ACCEPT_ENCODING', $acceptMethods, SQ_SERVER);
   if (empty($acceptMethods)) 
      return;


   // will browser understand gzip compression?
   //
   global $gzip_supported;
   $gzip_supported = '';


   // nope...
   //
   if (strpos($acceptMethods, 'gzip') === FALSE) return;


   // yep...
   //
   if (strpos($acceptMethods, 'x-gzip') === FALSE)
      $gzip_supported = 'gzip';
   else
      $gzip_supported = 'x-gzip';


   // turn on output buffering and make sure implicit flushing is off
   //
   ob_start('gzip_end');
   ob_implicit_flush(0);

}



/**
  * Compresses output if page size is over threshold
  *
  * @param string $buffer The page output to be compressed
  *
  * @return string The compressed output
  *
  */
function gzip_end($buffer)
{

   global $gzip_minimum_size_default, $gzip_supported, $gzip_enable_default, 
          $gzip_allow_overrides, $gzip_binary, $username, $data_dir, $attachment_dir,
          $gzip_compression_level;


   // some plugins set SM_PATH incorrectly... 
   // try to guess where we are in those cases...
   //
   if (file_exists(SM_PATH . 'plugins/gzip/config.php'))
      include_once(SM_PATH . 'plugins/gzip/config.php');
   else if (file_exists('../' . 'plugins/gzip/config.php'))
      include_once('../' . 'plugins/gzip/config.php');
   else if (file_exists('../../' . 'plugins/gzip/config.php'))
      include_once('../../' . 'plugins/gzip/config.php');
   else 
      return $buffer;


   $gzip_size = getPref($data_dir, $username, 'gzip_size', $gzip_minimum_size_default);
   $gzip_enable = getPref($data_dir, $username, 'gzip_enable', $gzip_enable_default);
   if (!$gzip_allow_overrides) 
   {
      $gzip_enable = $gzip_enable_default;
      $gzip_size = $gzip_minimum_size_default;
   }


   // turned off or not supported?
   //
   if (!$gzip_enable || $gzip_supported == '')
      return $buffer;


   // check size threshold
   //
   if (strlen($buffer) < $gzip_size * 1024)
      return $buffer;



   // do the compression
   //
   if ($gzip_supported == 'x-gzip' || $gzip_supported == 'gzip')
   {

      header("Content-Encoding: $gzip_supported");

      // use built-in zlib functionality
      //
      if (extension_loaded('zlib'))
      {

         if ($gzip_compression_level === 'system_load') 
           $gzip_compression_level = get_complevel();

         $prefix = "\x1f\x8b\x08\x00\x00\x00\x00\x00";
         $size = strlen($buffer);
         $crc = crc32($buffer);
         $buffer = gzcompress($buffer, $gzip_compression_level);
         $buffer = $prefix
                 . substr($buffer, 0, strlen($buffer) - 4) 
                 . pack('V', $crc) 
                 . pack('V', $size);
        
         return $buffer;

      }

      // have to use gzip binary instead
      //
      else
      {

         // pipe buffer through gzip call
         //
         $tempFile = tempnam($attachment_dir, 'sm-gz');
         $fp = popen($gzip_binary . ' > ' . $tempFile, 'w');
         fwrite($fp, $buffer);
         pclose($fp);


         // read file back and return it
         //
         $buffer = '';
         $fp = fopen($tempFile, "rb");
         while (!feof($fp)) $buffer .= fread($fp, 4096);
         fclose($fp);
         unlink($tempFile);
         return $buffer;

      }

   }


   // other compression methods here...
   //
   else
      return $buffer;

}



    /*
     * Code from here to bottom stolen from:
     *
     * gzip_encode - a class to gzip encode php output
     *
     * http://Leknor.com/code/
     *
     * By Sandy McArthur, Jr. <Leknor@Leknor.com>
     *
     * Copyright 2001 (c) All Rights Reserved, All Responsibility Yours.
     *
     * This code is released under the GNU LGPL Go read it over here:
     * http://www.gnu.org/copyleft/lesser.html
     *
     * Version 0.67
     *
     */
    /*
     * get_complevel() - The level of compression we should use.
     *
     * Returns an int between 0 and 9 inclusive.
     *
     * Tip: $gzleve = gzip_encode::get_complevel(); to get the compression level
     *      that will be used with out actually compressing the output.
     *
     * Help: if you use an OS other then linux please send me code to make
     * this work with your OS - Thanks
     */
    function get_complevel() {
    $uname = posix_uname();
    switch ($uname['sysname']) {
        case 'Linux':
        $cl = (1 - linux_loadavg()) * 10;
        $level = (int)max(min(9, $cl), 0);
        break;
        case 'FreeBSD':
        $cl = (1 - freebsd_loadavg()) * 10;
        $level = (int)max(min(9, $cl), 0);
        break;  
        default:
        $level = 3;
        break;
    }
    return $level;
    }
     
    /*  
     * linux_loadavg() - Gets the max() system load average from /proc/loadavg
     *
     * The max() Load Average will be returned
     */
    function linux_loadavg() {
    $buffer = "0 0 0";
    $f = fopen("/proc/loadavg","r");
    if (!feof($f)) {
        $buffer = fgets($f, 1024);
    }
    fclose($f);
    $load = explode(" ",$buffer);
    return max((float)$load[0], (float)$load[1], (float)$load[2]);
    }   
        
    /*
     * freebsd_loadavg() - Gets the max() system load average from uname(1)
     *  
     * The max() Load Average will be returned
     *  
     * I've been told the code below will work on solaris too, anyone wanna
     * test it?
     */ 
    function freebsd_loadavg() {
    $buffer= `uptime`;
    ereg("averag(es|e): ([0-9][.][0-9][0-9]), ([0-9][.][0-9][0-9]), ([0-9][.][0-9][0-9]*)", $buffer, $load); 
        
    return max((float)$load[2], (float)$load[3], (float)$load[4]);
    }   



?>
