<?php

/**
  * SquirrelMail Empty Folders Plugin
  * Copyright (c) 2003-2008 Paul Lesniewski <paul@squirrelmail.org>
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage empty_folders
  *
  */



/**
  * Validate that this plugin is configured correctly
  *
  * @return boolean Whether or not there was a
  *                 configuration error for this plugin.
  *
  */
function empty_folders_check_configuration_do()
{

   // make sure compatibility plugin is there at all (have to do this
   // because the empty_folders_init() function uses load_config()
   //
   if (!function_exists('check_file_contents'))
   {
      do_err('Empty Folders plugin requires the Compatibility plugin version 2.0.12+', FALSE);
      return TRUE;
   }



   // make sure base config is available
   //
   if (!empty_folders_init())
   {
      do_err('Empty Folders plugin is missing its main configuration file', FALSE);
      return TRUE;
   }



   // check that the needed patch is in place for SM versions that need it
   //
   if ((!check_sm_version(1, 3)
     && !check_file_contents(SM_PATH . 'src/left_main.php', 'do_hook_function\(\'left_main_after_each_folder\''))
    || (check_sm_version(1, 4, 0) && !check_sm_version(1, 4, 1)
     && !check_file_contents(SM_PATH . 'src/left_main.php', 'concat_hook_function\(\'left_main_after_each_folder\'')))
   {
      do_err('Empty Folders plugin requires a patch with your version of SquirrelMail, but it has not been applied', FALSE);
      return TRUE;
   }
   if (check_sm_version(1, 5, 0) && !check_sm_version(1, 5, 1)
    && !check_file_contents(SM_PATH . 'src/left_main.php', 'let plugins fiddle with end of line'))
   {
      do_err('Empty Folders plugin requires a source file replacement for src/left_main.php with your version of SquirrelMail, but it has not been done', FALSE);
      return TRUE;
   }

}



/**
  * Initialize this plugin (load config values)
  *
  * @return boolean FALSE if no configuration file could be loaded, TRUE otherwise
  *
  */
function empty_folders_init()
{

   return load_config('empty_folders',
                      array('../../config/config_empty_folders.php',
                            'config.php',
                            'config_default.php'),
                      TRUE, TRUE);

}



/**
  * Display empty links on target folders
  *
  * @param array $box Parameters passed to this hook by SquirrelMail - 
  *                   in 1.5.2+, this is an array of mailbox info,
  *                   in 1.4.x, the first element is number of messages,
  *                   the second is mailbox name, and the third is a
  *                   usable IMAP server connection.
  *                     
  */
function empty_folders_show_link_do(&$box) 
{

   global $show_empty_link_allow_override, $data_dir, $username,
          $show_purge_link_allow_override, $confirm_empty_link,
          $confirm_purge_link, $folders_to_not_show_empty_link,
          $folders_to_show_empty_link, $folders_to_show_purge_link,
          $folders_to_not_show_purge_link, $purge_link_text,
          $empty_link_text, $empty_link_confirm_text,
          $purge_link_title_text, $delete_link_title_text,
          $purge_link_confirm_text,
          $empty_folders_link_onclick, $javascript_on;


   if (check_sm_version(1, 5, 2))
      $folder_name = $box['MailboxFullName'];
   else
      $folder_name = $box[1];


   empty_folders_init();


   // grab user preference for confirm purge (just do it once)
   //
   static $confirm_purge_link_setting = NULL;
   if (is_null($confirm_purge_link_setting))
   {
      if ($show_purge_link_allow_override)
         $confirm_purge_link_setting = getPref($data_dir, $username,
                                               'confirm_purge_link',
                                               $confirm_purge_link);
      else
         $confirm_purge_link_setting = $confirm_purge_link;
   }



   // grab user preference for confirm empty (just do it once)
   //
   static $confirm_empty_link_setting = NULL;
   if (is_null($confirm_empty_link_setting))
   {
      if ($show_empty_link_allow_override)
         $confirm_empty_link_setting = getPref($data_dir, $username,
                                               'confirm_empty_link',
                                               $confirm_empty_link);
      else
         $confirm_empty_link_setting = $confirm_empty_link;
   }



   // grab user preference for purge link folders (just do it once)
   //
   static $empty_folders_show_purge_link = 0;
   if ($empty_folders_show_purge_link === 0)
   {
      if ($show_purge_link_allow_override)
         $allFolders = getPref($data_dir, $username,
                               'empty_folders_show_purge_link',
                               NULL);
      else
         $allFolders = NULL;


      // reformat folder list if not NULL (which means defaults apply below)
      //
      if ($allFolders == 'NONE') $allFolders = ''; // see note elsewhere about why we use "NONE"
      if (!is_null($allFolders)) $empty_folders_show_purge_link = explode('###', $allFolders);
      else $empty_folders_show_purge_link = NULL;
   }



   // grab user preference for empty link folders (just do it once)
   //
   static $empty_folders_show_empty_link = 0;
   if ($empty_folders_show_empty_link === 0)
   {
      if ($show_empty_link_allow_override)
         $allFolders = getPref($data_dir, $username,
                               'empty_folders_show_empty_link',
                               NULL);
      else
         $allFolders = NULL;


      // reformat folder list if not NULL (which means defaults apply below)
      //
      if ($allFolders == 'NONE') $allFolders = ''; // see note elsewhere about why we use "NONE"
      if (!is_null($allFolders)) $empty_folders_show_empty_link = explode('###', $allFolders);
      else $empty_folders_show_empty_link = NULL;
   }



   // now figure out if this folder gets a purge link
   //
   // if the user specified a list, just use it to
   // test for what folders have the link turned on
   //
   $show_purge_link = FALSE;
   if (is_array($empty_folders_show_purge_link))
   {
      if (in_array($folder_name, $empty_folders_show_purge_link))
         $show_purge_link = TRUE;
   }


   // or apply the defaults from the configuration file
   //
   // (link enabled if in default list, or if NOT in the
   // default "don't show" list (and that list is non-empty))
   //
   else
   {
      if (in_array($folder_name, $folders_to_show_purge_link)
       || (!empty($folders_to_not_show_purge_link) && !in_array($folder_name, $folders_to_not_show_purge_link)))
         $show_purge_link = TRUE;
   }



   // now figure out if this folder gets an empty link
   //
   // if the user specified a list, just use it to
   // test for what folders have the link turned on
   //
   $show_empty_link = FALSE;
   if (is_array($empty_folders_show_empty_link))
   {
      if (in_array($folder_name, $empty_folders_show_empty_link))
         $show_empty_link = TRUE;
   }


   // or apply the defaults from the configuration file
   //
   // (link enabled if in default list, or if NOT in the
   // default "don't show" list (and that list is non-empty))
   //
   else
   {
      if (in_array($folder_name, $folders_to_show_empty_link)
       || (!empty($folders_to_not_show_empty_link) && !in_array($folder_name, $folders_to_not_show_empty_link)))
         $show_empty_link = TRUE;
   }



   // in 1.5.2+, we get a nice array being prepared for
   // template use - we just add to it to the "ExtraOutput"
   // element
   //
   if (check_sm_version(1, 5, 2))
   {

      // don't bother with the trash folder
      //
      global $oTemplate, $trash_folder;
      if ($box['MailboxFullName'] == $trash_folder)
         return;


      // get message count if missing
      //
      if (($show_purge_link || $show_empty_link)
       && empty($box['MessageCount']))
      {
         global $imapConnection;
         $aStatus = sqimap_status_messages($imapConnection,
                                           $box['MailboxFullName'],
                                           array('MESSAGES'));
         $box['MessageCount'] = $aStatus['MESSAGES'];
      }


      // generate purge link (1.5.x)
      //
      if ($show_purge_link
       && $box['MessageCount'] > 0)
      {

         sq_change_text_domain('empty_folders');

         // build the onclick handler if necessary
         //
         if ($javascript_on && $confirm_purge_link)
            $onclick = str_replace('###TEXT###', sprintf(_($purge_link_confirm_text), $box['MessageCount']), $empty_folders_link_onclick);
         else
            $onclick = '';

         // build the link URI
         //
         $urlMailbox = urlencode($box['MailboxFullName']);
         $empty_uri = sqm_baseuri()
                    . 'plugins/empty_folders/empty_folder.php?ef_act=ef_purge&mailbox='
                    . $urlMailbox;

         $oTemplate->assign('empty_uri', $empty_uri);
         $oTemplate->assign('link_text', sprintf(_($purge_link_text), $box['MessageCount']));
         $oTemplate->assign('title_text', sprintf(_($purge_link_title_text), $box['MessageCount']));
         $oTemplate->assign('onclick', $onclick);
         $oTemplate->assign('square_brackets', TRUE);
         $output = $oTemplate->fetch('plugins/empty_folders/empty_link.tpl');

         sq_change_text_domain('squirrelmail');

         if (empty($box['ExtraOutput']))
            $box['ExtraOutput'] = '';
         $box['ExtraOutput'] .= $output;

      }


      // generate empty link (1.5.x)
      //
      if ($show_empty_link
       && $box['MessageCount'] > 0)
      {  

         sq_change_text_domain('empty_folders');
   
         // build the onclick handler if necessary
         // 
         if ($javascript_on && $confirm_empty_link)
            $onclick = str_replace('###TEXT###', sprintf(_($empty_link_confirm_text), $box['MessageCount']), $empty_folders_link_onclick);
         else
            $onclick = '';

         // build the link URI
         // 
         $urlMailbox = urlencode($box['MailboxFullName']);
         $empty_uri = sqm_baseuri()
                    . 'plugins/empty_folders/empty_folder.php?ef_act=ef_empty&mailbox='
                    . $urlMailbox;
   
         $oTemplate->assign('empty_uri', $empty_uri);
         $oTemplate->assign('link_text', sprintf(_($empty_link_text), $box['MessageCount']));
         $oTemplate->assign('title_text', sprintf(_($delete_link_title_text), $box['MessageCount']));
         $oTemplate->assign('onclick', $onclick);
         $oTemplate->assign('square_brackets', TRUE);
         $output = $oTemplate->fetch('plugins/empty_folders/empty_link.tpl');

         sq_change_text_domain('squirrelmail');

         if (empty($box['ExtraOutput']))
            $box['ExtraOutput'] = '';
         $box['ExtraOutput'] .= $output;

      }  

   }



   // in 1.4.x, we have three parameters
   //
   else
   {

      $output = '';

      // don't need to skip trash - in 1.4.x, this hook isn't run for it

      $numMessages = $box[0];
      $real_box = $box[1];
      $imapConnection = $box[2];

      // get message count if missing
      //
      if (($show_purge_link || $show_empty_link)
       && empty($numMessages))
         $numMessages = sqimap_get_num_messages($imapConnection, $real_box);


      // generate purge link (1.4.x)
      //
      if ($show_purge_link
       && $numMessages > 0)
      {
         global $t, $square_brackets, $onclick, $empty_uri,
                $link_text, $title_text;
         sq_change_text_domain('empty_folders');

         // build the link URI
         //
         $urlMailbox = urlencode($real_box);
         $empty_uri = sqm_baseuri()
                    . 'plugins/empty_folders/empty_folder.php?ef_act=ef_purge&mailbox='
                    . $urlMailbox;

         // build the onclick handler if necessary
         //
         if ($javascript_on && $confirm_purge_link)
            $onclick = str_replace('###TEXT###', sprintf(_($purge_link_confirm_text), $numMessages), $empty_folders_link_onclick);
         else
            $onclick = '';

         $link_text = sprintf(_($purge_link_text), $numMessages);
         $title_text = sprintf(_($purge_link_title_text), $numMessages);

         $square_brackets = FALSE;
         $t = array(); // no need to put config vars herein, they are already globalized

         ob_start();
         include(SM_PATH . 'plugins/empty_folders/templates/default/empty_link.tpl');
         $output .= ob_get_contents();
         ob_end_clean();

         sq_change_text_domain('squirrelmail');
      }


      // generate empty link (1.4.x)
      //
      if ($show_empty_link
       && $numMessages > 0)
      {
         global $t, $square_brackets, $onclick, $empty_uri,
                $link_text, $title_text;
         sq_change_text_domain('empty_folders');

         // build the link URI
         //
         $urlMailbox = urlencode($real_box);
         $empty_uri = sqm_baseuri()
                    . 'plugins/empty_folders/empty_folder.php?ef_act=ef_empty&mailbox='
                    . $urlMailbox;

         // build the onclick handler if necessary
         //
         if ($javascript_on && $confirm_empty_link)
            $onclick = str_replace('###TEXT###', sprintf(_($empty_link_confirm_text), $numMessages), $empty_folders_link_onclick);
         else
            $onclick = '';

         $link_text = sprintf(_($empty_link_text), $numMessages);
         $title_text = sprintf(_($delete_link_title_text), $numMessages);

         $square_brackets = FALSE;
         $t = array(); // no need to put config vars herein, they are already globalized

         ob_start();
         include(SM_PATH . 'plugins/empty_folders/templates/default/empty_link.tpl');
         $output .= ob_get_contents();
         ob_end_clean();

         sq_change_text_domain('squirrelmail');
      }

      return $output;

   }

}



/**
  * Display purge/delete all button in target mailbox listings
  *
  * @param array $buttons The list of buttons being added to when
  *                       using 1.5.2+
  *
  */
function empty_folders_show_button_do(&$buttons)
{

   global $numMessages, $aMailbox, $trash_folder, $purge_button_confirm_text,
          $username, $data_dir, $show_purge_trash_button,
          $purge_trash_text, $show_purge_trash_button_allow_override,
          $show_purge_button_allow_override, $folders_to_show_purge_button,
          $folders_to_not_show_purge_button, $purge_button_text,
          $purge_button_title_text, $delete_button_title_text,
          $show_delete_all_button_allow_override, $javascript_on,
          $folders_to_show_delete_all_button, $delete_button_text,
          $folders_to_not_show_delete_all_button, $delete_button_confirm_text,
          $confirm_delete_all_button, $confirm_purge_button,
          $empty_folders_button_onclick;

   empty_folders_init();

   sqgetGlobalVar('mailbox', $mailbox, SQ_FORM);
   if (empty($mailbox)) $mailbox = 'INBOX';


   if (check_sm_version(1, 5, 1))
      $message_count = $aMailbox['EXISTS'];
   else
      $message_count = $numMessages;



   // get user configured settings if allowed
   //
   if ($show_purge_trash_button_allow_override)
      $show_purge_trash_button = getPref($data_dir, $username,
                                         'show_purge_trash_button',
                                         $show_purge_trash_button);



   // the trash folder can only have a purge button
   //
   if ($mailbox == $trash_folder)
   {

      if ($show_purge_trash_button)
      {

         sq_change_text_domain('empty_folders');

         // grab user preference for confirm purge
         //
         if ($show_purge_button_allow_override)
            $confirm_purge_button = getPref($data_dir, $username,
                                            'confirm_purge_button',
                                            $confirm_purge_button);

         if ($confirm_purge_button && $javascript_on)
            $onclick = str_replace('###TEXT###', sprintf(_($purge_button_confirm_text), $message_count), $empty_folders_button_onclick);
         else
            $onclick = '';

         if (check_sm_version(1, 5, 2))
            $buttons['efPurgeTrash'] = array('value' => sprintf(_($purge_trash_text),
                                                                $message_count),
                                             'type' => 'submit',
                                             'extra_attrs' => array('onclick' => $onclick));
         else if (check_sm_version(1, 5, 1))
            $buttons[1]['efPurgeTrash'] = array(0 => sprintf(_($purge_trash_text),
                                                           $message_count),
                                              1 => 'submit');
         else
            echo '<input type="submit" name="efPurgeTrash" onclick="' . $onclick . '" value="' . sprintf(_($purge_trash_text), $message_count) . "\" />\n";

         sq_change_text_domain('squirrelmail');

      }


      // we're done - no other button controls allowed here
      //
      return;

   }



   // grab user preference for purge button folders
   //
   if ($show_purge_button_allow_override)
      $allFolders = getPref($data_dir, $username,
                            'empty_folders_show_purge_button',
                            NULL);
   else
      $allFolders = NULL;



   // reformat folder list if not NULL (which means defaults apply below)
   //
   if ($allFolders == 'NONE') $allFolders = ''; // see note elsewhere about why we use "NONE"
   if (!is_null($allFolders)) $empty_folders_show_purge_button = explode('###', $allFolders);
   else $empty_folders_show_purge_button = NULL;



   // now figure out if this folder gets a purge button
   //       
   // if the user specified a list, just use it to
   // test for what folders have the button turned on
   //
   $show_purge_button = FALSE;
   if (is_array($empty_folders_show_purge_button))
   {
      if (in_array($mailbox, $empty_folders_show_purge_button))
         $show_purge_button = TRUE;
   }


   // or apply the defaults from the configuration file
   //
   // (button enabled if in default list, or if NOT in the
   // default "don't show" list (and that list is non-empty))
   //
   else
   {
      if (in_array($mailbox, $folders_to_show_purge_button)
       || (!empty($folders_to_not_show_purge_button) && !in_array($mailbox, $folders_to_not_show_purge_button)))
         $show_purge_button = TRUE;
   }



   // grab user preference for delete all button folders
   //
   if ($show_delete_all_button_allow_override)
      $allFolders = getPref($data_dir, $username,
                            'empty_folders_show_delete_all',
                            NULL);
   else
      $allFolders = NULL;



   // reformat folder list if not NULL (which means defaults apply below)
   //
   if ($allFolders == 'NONE') $allFolders = ''; // see note elsewhere about why we use "NONE"
   if (!is_null($allFolders)) $empty_folders_show_delete_all = explode('###', $allFolders);
   else $empty_folders_show_delete_all = NULL;



   // now figure out if this folder gets a delete all button
   //
   // if the user specified a list, just use it to
   // test for what folders have the button turned on
   //
   $show_delete_all_button = FALSE;
   if (is_array($empty_folders_show_delete_all))
   {
      if (in_array($mailbox, $empty_folders_show_delete_all))
         $show_delete_all_button = TRUE;
   }


   // or apply the defaults from the configuration file
   //
   // (button enabled if in default list, or if NOT in the
   // default "don't show" list (and that list is non-empty))
   //
   else
   {
      if (in_array($mailbox, $folders_to_show_delete_all_button)
       || (!empty($folders_to_not_show_delete_all_button) && !in_array($mailbox, $folders_to_not_show_delete_all_button)))
         $show_delete_all_button = TRUE;
   }



   sq_change_text_domain('empty_folders');


   // delete all button?
   //
   if ($show_delete_all_button)
   {

      // grab user preference for confirm delete all
      //
      if ($show_delete_all_button_allow_override)
         $confirm_delete_all_button = getPref($data_dir, $username,
                                              'confirm_delete_all_button',
                                              $confirm_delete_all_button);

      if ($confirm_delete_all_button && $javascript_on)
         $onclick = str_replace('###TEXT###', sprintf(_($delete_button_confirm_text), $message_count), $empty_folders_button_onclick);
      else
         $onclick = '';

      if (check_sm_version(1, 5, 2))
         $buttons['efDeleteAll'] = array('value' => sprintf(_($delete_button_text),
                                                            $message_count),
                                        'type' => 'submit',
                                        'extra_attrs' => array('onclick' => $onclick,
                                                               'title' => sprintf(_($delete_button_title_text), $message_count)));
      else if (check_sm_version(1, 5, 1))
         $buttons[1]['efDeleteAll'] = array(0 => sprintf(_($delete_button_text),
                                                         $message_count),
                                           1 => 'submit');
      else
         echo '<input type="submit" name="efDeleteAll" title="' . sprintf(_($delete_button_title_text), $message_count) . '" onclick="' . $onclick . '" value="' . sprintf(_($delete_button_text), $message_count) . "\" />\n";

   }



   // purge button?
   //
   if ($show_purge_button)
   {

      // grab user preference for confirm purge
      //
      if ($show_purge_button_allow_override)
         $confirm_purge_button = getPref($data_dir, $username,
                                         'confirm_purge_button',
                                         $confirm_purge_button);

      if ($confirm_purge_button && $javascript_on)
         $onclick = str_replace('###TEXT###', sprintf(_($purge_button_confirm_text), $message_count), $empty_folders_button_onclick);
      else
         $onclick = '';

      if (check_sm_version(1, 5, 2))
         $buttons['efPurgeAll'] = array('value' => sprintf(_($purge_button_text),
                                                           $message_count),
                                        'type' => 'submit',
                                        'extra_attrs' => array('onclick' => $onclick,
                                                               'title' => sprintf(_($purge_button_title_text), $message_count)));
      else if (check_sm_version(1, 5, 1))
         $buttons[1]['efPurgeAll'] = array(0 => sprintf(_($purge_button_text),
                                                        $message_count),
                                           1 => 'submit');
      else
         echo '<input type="submit" name="efPurgeAll" title="' . sprintf(_($purge_button_title_text), $message_count) . '" onclick="' . $onclick . '" value="' . sprintf(_($purge_button_text), $message_count) . "\" />\n";

   }


   sq_change_text_domain('squirrelmail');

}



/**
  * Perform purge or delete-all actions when such a button
  * has been clicked.
  *
  * @param array (only used in 1.5.2) An array consisting of: button name (provided
  *                                   by an external caller - ignore for now), the
  *                                   mailbox cache, the account number, the mailbox
  *                                   name (ignore for now), and a UID list (should
  *                                   be null when we care about handling any actions).
  *
  * @return boolean (only in 1.5.2) TRUE when a button action
  *                                 was handled, FALSE otherwise
  *
  */
function empty_folders_handle_button_click_do(&$args)
{

   global $trash_folder;
   if (!sqGetGlobalVar('mailbox', $mailbox, SQ_FORM) || empty($mailbox))
      $mailbox = 'INBOX';


   if (check_sm_version(1, 5, 2))
   {
      $mbox_cache = &$args[1];
      $account_number = $args[2];
      //$mailbox = $args[3];
   }
   else
   {
      $mbox_cache = NULL;
      $account_number = 0;
      //if (!sqGetGlobalVar('mailbox', $mailbox, SQ_FORM) || empty($mailbox))
      //   $mailbox = 'INBOX';
   }



   // purge trash?
   //
   if ($mailbox == $trash_folder)
   {

      if (sqGetGlobalVar('efPurgeTrash', $efPurgeTrash, SQ_FORM) && !empty($efPurgeTrash))
      {

         // do the purge - yikes
         //
         empty_folder($mailbox, FALSE, $mbox_cache, $account_number);


         // in 1.5.2, we need to return true to indicate we handled this action
         //
         if (check_sm_version(1, 5, 2))
            return TRUE;


         // in 1.4.x, redirect back to message list
         //
         if (!sqGetGlobalVar('location', $location, SQ_FORM)) $location = php_self();
         $location = set_url_var($location,'startMessage', 1, false);
         header('Location: ' . $location);
         exit;

      }

      // otherwise, return having done nothing
      //
      return FALSE;

   }



   // purge some other folder?
   //
   if (sqGetGlobalVar('efPurgeAll', $efPurgeAll, SQ_FORM) && !empty($efPurgeAll))
   {

      // do the purge - yikes
      //
      empty_folder($mailbox, FALSE, $mbox_cache, $account_number);


      // in 1.5.2, we need to return true to indicate we handled this action
      //
      if (check_sm_version(1, 5, 2))
         return TRUE;


      // in 1.4.x, redirect back to message list
      //
      if (!sqGetGlobalVar('location', $location, SQ_FORM)) $location = php_self();
      $location = set_url_var($location,'startMessage', 1, false);
      header('Location: ' . $location);
      exit;

   }



   // delete some other folder?
   //
   if (sqGetGlobalVar('efDeleteAll', $efDeleteAll, SQ_FORM) && !empty($efDeleteAll))
   {

      // do the delete
      //
      empty_folder($mailbox, TRUE, $mbox_cache, $account_number);


      // in 1.5.2, we need to return true to indicate we handled this action
      //
      if (check_sm_version(1, 5, 2))
         return TRUE;


      // in 1.4.x, redirect back to message list
      //
      if (!sqGetGlobalVar('location', $location, SQ_FORM)) $location = php_self();
      $location = set_url_var($location,'startMessage', 1, false);
      header('Location: ' . $location);
      exit;

   }

   return FALSE;

}



/**
  * Delete all messages in a given folder
  *
  * @param string  $mailbox     The name of the folder
  * @param boolean $moveToTrash Whether or not to move to
  *                             trash (TRUE) or just remove
  *                             messages from the server (FALSE).
  * @param mixed   $mbox_cache  An array of mailbox cache information
  *                             (used in SM 1.5.x), which is optional,
  *                             but will be updated if given.  If given
  *                             as boolean TRUE, this function will attempt
  *                             to first retrieve it from the PHP session.
  *                             NOTE that this parameter is passed by reference
  *                             and is potentially modified herein.  Because it
  *                             is passed by reference, PHP 4 doesn't like a default
  *                             value being specified, so callers should pass NULL
  *                             when it is not available or applicable.
  * @param int     $account_num The account number in use (only applicable under
  *                             SquirrelMail 1.5.2+) (OPTIONAL; default = 0)
  * @param boolean $quiet       When TRUE, errors are not shown
  *                             on screen (OPTIONAL; default FALSE)
  *
  * @return boolean TRUE if the operation succeeded, FALSE otherwise
  *                 (in versions prior to 1.5.2, instead of FALSE,
  *                 an error will usually be shown on screen and
  *                 execution will halt)
  *
  */
function empty_folder($mailbox, $moveToTrash, &$mbox_cache, $account_num=0, $quiet=FALSE)
{

   global $imapConnection, $trash_folder, $uid_support;
   if (!check_sm_version(1, 5, 2)) $uid_support = TRUE;


   // has all the imap function file includes that we need
   //
   include_once (SM_PATH . 'functions/imap.php');


   // needed in case errors occur
   //
   include_once(SM_PATH . 'functions/display_messages.php');


   // create IMAP connection if needed
   //
   if (!is_resource($imapConnection))
   {
      global $username, $imapServerAddress, $imapPort;
      $key = FALSE;
      if (!check_sm_version(1, 5, 2))
         sqgetGlobalVar('key', $key, SQ_COOKIE);
//TODO: in future, for 1.5.2, we should be able to use the last argument to tell this function (sqimap_login) to return any errors.... in which case we have to inspect the return value and handle errors ourselves if there are any
      $imap_stream = sqimap_login($username, $key, $imapServerAddress, $imapPort, 0);
      $close_imap_stream = TRUE;
   }
   else
   {
      $imap_stream = $imapConnection;
      $close_imap_stream = FALSE;
   }



   // if needed, get the mailbox cache ourselves
   //
   if (check_sm_version(1, 5, 2) && $mbox_cache === TRUE)
   {
      include_once(SM_PATH . 'functions/mailbox_display.php');
      $mbox_cache = sqm_api_mailbox_select($imap_stream, $account_num, $mailbox,
                                           array(), array());
   }



   // grab message count
   //
   if (!empty($mbox_cache['EXISTS']))
      $message_count = $mbox_cache['EXISTS'];
   else
   {
      $mbx_response = sqimap_mailbox_select($imap_stream, $mailbox);
      $message_count = $mbx_response['EXISTS'];
   }



   // check of there are any messages to delete at all
   //
   if ($message_count > 0)
   {

      // copy all messages to trash first if needed
      //
      if ($moveToTrash
       && $mailbox != $trash_folder
       && sqimap_mailbox_exists($imap_stream, $trash_folder))
      {

         // turn off internal error handling (third argument = false) and
         // ignore copy to trash errors (allows deleting messages when over quota)
         //
         $read = sqimap_run_command($imap_stream, "COPY 1:* \"$trash_folder\"", false, $response, $message, $uid_support);

      }


      // now flag all messages as deleted
      //
      $read = sqimap_run_command($imap_stream, 'STORE 1:* +FLAGS (\\Deleted)',
                                 !$quiet, $response, $message, $uid_support);
      if ($quiet && ($read === FALSE || $response != 'OK'))
         return FALSE;


      // and expunge them and unselect the folder at once
      //
      $read = sqimap_run_command($imap_stream, 'CLOSE', !$quiet, $response, $message);
      if ($quiet && ($read === FALSE || $response != 'OK'))
         return FALSE;


      // synchronize mailbox cache (only in 1.5.2+)
      //
      if ($mbox_cache && check_sm_version(1, 5, 2))
      {

         // this is probably slower than manipulating the message cache ourselves,
         // but not sure what parts of the cache need to be updated (there is no
         // documentation on the cache anywhere that I know of)... perhaps set
         // $mbox_cache['EXISTS'] = 0
         // What about: $mbox_cache['TOTAL'][$mbox_cache['SETINDEX']]
         // What about: $mbox_cache['PAGEOFFSET'] = 1
         // What about: $mbox_cache['PAGEOFFSET'] = 0
         // Uh, and do we just reset/unset: $mbox_cache['UIDSET'][$mbox_cache['SETINDEX']]
         // What about: $mbox_cache['MSG_HEADERS']
         // These are just guesses looking at a VERY SIMPLE part of
         // handleMessageListForm(), but looking at sqm_api_mailbox_select(),
         // it seems like there should be more things updated than that,
         // even just the timestamp(?).... and what about re-saving to session?
         //
         // instead, for now, just rebuild the whole thing:
         //
         $mbox_cache = sqm_api_mailbox_select($imap_stream, $account_num, $mailbox, array(), array());

      } 

   }



   if ($close_imap_stream) sqimap_logout($imap_stream);

   return TRUE;

}



/**
  * Provide an RPC interface to this plugin.
  *
  * Actions that are handled are:
  *
  *    empty_folders_purge_trash - removes all messages from the
  *                                trash folder
  *    empty_folders_purge_all   - removes all messages permanently
  *                                from the given folder (specified
  *                                by the "mailbox" argument in the
  *                                RPC request) (does NOT copy them
  *                                to the trash folder)
  *    empty_folders_delete_all  - removes all messages from the given
  *                                folder (specified by the "mailbox"
  *                                argument in the RPC request) and
  *                                places copies of them in the trash
  *                                folder)
  *
  * Error codes that are returned:
  *
  *    500 - empty_folders_purge_trash failed
  *    501 - empty_folders_purge_all failed
  *    502 - empty_folders_delete_all failed
  *
  * @param array The first element is the requested RPC action
  *
  * @return boolean FALSE if the action was not one that
  *                 this plugin handles - otherwise this
  *                 function will never exit (ala
  *                 sm_rpc_return_success())
  *
  */
function empty_folders_rpc_do($args)
{

   switch ($args[0])
   {

      // purge trash folder
      //
      case 'empty_folders_purge_trash':

         // get mailbox cache if no one has done it yet
         //
         global $mailbox_cache;
         if (empty($mailbox_cache))
            sqgetGlobalVar('mailbox_cache', $mailbox_cache, SQ_SESSION);

         $mbox_cache = TRUE;
         global $trash_folder;
         if (sqgetGlobalVar('account', $account_number, SQ_FORM) === false)
            $account_number = 0;
         sq_change_text_domain('empty_folders');
         $success_message = _("The trash folder has been emptied");
         $error_message = _("The trash folder could not be emptied");
         sq_change_text_domain('squirrelmail');
         if (empty_folder($trash_folder, FALSE, $mbox_cache, $account_number, TRUE))
         {
            // save mailbox cache back to session
            //
            $mailbox_cache[$account_number . '_' . $mbox_cache['NAME']] = $mbox_cache;
            sqsession_register($mailbox_cache, 'mailbox_cache');

            sm_rpc_return_success(0, $success_message);
         }
         else
            sm_rpc_return_error(500, $error_message);
         break;


      // purge other folder
      //
      case 'empty_folders_purge_all':

         // get mailbox cache if no one has done it yet
         //
         global $mailbox_cache;
         if (empty($mailbox_cache))
            sqgetGlobalVar('mailbox_cache', $mailbox_cache, SQ_SESSION);

         $mbox_cache = TRUE;
         if (!sqGetGlobalVar('mailbox', $mailbox, SQ_FORM) || empty($mailbox))
            $mailbox = 'INBOX';
         if (sqgetGlobalVar('account', $account_number, SQ_FORM) === false)
            $account_number = 0;
         sq_change_text_domain('empty_folders');
         $success_message = _("Folder has been emptied");
         $error_message = _("Folder could not be emptied");
         sq_change_text_domain('squirrelmail');
         if (empty_folder($mailbox, FALSE, $mbox_cache, $account_number, TRUE))
         {
            // save mailbox cache back to session
            //
            $mailbox_cache[$account_number . '_' . $mbox_cache['NAME']] = $mbox_cache;
            sqsession_register($mailbox_cache, 'mailbox_cache');

            sm_rpc_return_success(0, $success_message);
         }
         else
            sm_rpc_return_error(501, $error_message);
         break;


      // delete from other folder
      //
      case 'empty_folders_delete_all':

         // get mailbox cache if no one has done it yet
         //
         global $mailbox_cache;
         if (empty($mailbox_cache))
            sqgetGlobalVar('mailbox_cache', $mailbox_cache, SQ_SESSION);

         $mbox_cache = TRUE;
         if (!sqGetGlobalVar('mailbox', $mailbox, SQ_FORM) || empty($mailbox))
            $mailbox = 'INBOX';
         if (sqgetGlobalVar('account', $account_number, SQ_FORM) === false)
            $account_number = 0;
         sq_change_text_domain('empty_folders');
         $success_message = _("Folder has been emptied");
         $error_message = _("Folder could not be emptied");
         sq_change_text_domain('squirrelmail');
         if (empty_folder($mailbox, TRUE, $mbox_cache, $account_number, TRUE))
         {
            // save mailbox cache back to session
            //
            $mailbox_cache[$account_number . '_' . $mbox_cache['NAME']] = $mbox_cache;
            sqsession_register($mailbox_cache, 'mailbox_cache');

            sm_rpc_return_success(0, $success_message);
         }
         else
            sm_rpc_return_error(502, $error_message);
         break;

   }


   // the RPC action was not one of our own
   //
   return FALSE;

}



