<?php

/**
  * SquirrelMail Email Footer Plugin
  * Copyright (c) 2001 Ray Black III <allah@accessnode.net>,
  * Copyright (c) 2003-2009 Paul Lesniewski <paul@squirrelmail.org>,
  * Licensed under the GNU GPL. For full terms see the file COPYING.
  *
  * @package plugins
  * @subpackage email_footer
  *
  */


/**
  * Validate that this plugin is configured correctly
  *
  * @return boolean Whether or not there was a
  *                 configuration error for this plugin.
  *
  */
function email_footer_check_configuration_do()
{

   global $enable_footer_configuration_in_vadmin;


   // make sure base config is available
   //
   if (!email_footer_init())
   {
      do_err('Email Footer plugin is missing its main configuration file', FALSE);
      return TRUE;
   }


   // if the footer is to be configured using Vadmin, make
   // sure Vadmin is present
   //
   if ($enable_footer_configuration_in_vadmin)
   {

      // best way to check
      //
      if (function_exists('check_plugin_version'))
      {
         if (!check_plugin_version('vadmin', 3, 0, 0))
         {
            do_err('Email Footer plugin is configured to be administered using Vadmin, but Vadmin version 3.0+ was not found', FALSE);
            return TRUE;
         }
      }


      // otherwise, just guess
      //
      else if (!file_exists(SM_PATH . 'plugins/vadmin/includes/sql/sql_functions.inc'))
      {
         do_err('Email Footer plugin is configured to be administered using Vadmin, but Vadmin version 3.0+ was not found', FALSE);
         return TRUE;
      }

   }

}



/**
  * Initialize this plugin (load config values)
  *
  * @return boolean FALSE if no configuration file could be loaded, TRUE otherwise
  *
  */
function email_footer_init()
{

   if (!@include_once (SM_PATH . 'config/config_email_footer.php'))
      if (!@include_once (SM_PATH . 'plugins/email_footer/config.php'))
         if (!@include_once (SM_PATH . 'plugins/email_footer/config_default.php'))
            return FALSE;

   return TRUE;

}



/**
  * Add the footer to an outgoing mail
  *
  */
function email_footer_addition_do(&$argv)
{

   global $plain_text_email_footer, $config_override, $draft,
          $plugins, $email_footer_debug, $email_footer_on_drafts,
          $enable_footer_configuration_in_vadmin;
   email_footer_init();


   // bail if this is a draft and we aren't adding footers to drafts
   //
   if ($draft && !$email_footer_on_drafts)
      return;


   // if Vadmin is being used to configure the footer,
   // get that and forget the rest
   //
   if ($enable_footer_configuration_in_vadmin)
      $email_footer = get_email_footer_from_vadmin();
   else
   {

      $email_footer = $plain_text_email_footer;


      // allow vlogin or multilogin plugins to override the footer
      //
      sqgetGlobalVar('config_override', $config_override, SQ_SESSION);
      if (isset($config_override['email_footer']))
         $email_footer = $config_override['email_footer'];

   }


   // if user is composing in HTML using the "html_mail" plugin, 
   // check for $html_email_footer and if not available, turn
   // newlines of regular footer into <br>'s
   //
   if (in_array('html_mail', $plugins))
   {

      global $strip_html_send_plain, $html_email_footer;
      sqGetGlobalVar('strip_html_send_plain', $strip_html_send_plain, SQ_FORM);

      include_once(SM_PATH . 'plugins/html_mail/functions.php');
      if (!$strip_html_send_plain && html_area_is_on_and_is_supported_by_users_browser())
      {

         // if Vadmin is being used to configure the footer,
         // get that and forget the rest
         //
         if ($enable_footer_configuration_in_vadmin)
            $email_footer = get_email_footer_from_vadmin(FALSE);
         else
         {

            if (!empty($html_email_footer))
               $email_footer = $html_email_footer;
            else
               $email_footer = preg_replace("/(\015\012)|(\015)|(\012)/", '<br />', $email_footer);


            // allow vlogin or multilogin plugins to override the footer
            //
            if (isset($config_override['email_footer_HTML']))
               $email_footer = $config_override['email_footer_HTML'];

         }

      }

   }



   // grab the message - probably not compatible with 1.5.0 or 1.5.1
   //
   if (check_sm_version(1, 5, 2))
      $message = &$argv;
   else
      $message = &$argv[1];



   // do the actual footer insertion
   //
   if (is_array($message->entities) && sizeof($message->entities) > 0)
      $message->entities[0]->body_part = $message->entities[0]->body_part . $email_footer;
   else
      $message->body_part = $message->body_part . $email_footer;



   // debugging...
   //
   if ($email_footer_debug) 
   {
      sm_print_r($message);
      exit;
   }



   return $message;

}



/**
  * Retrieve email footer text for the given (or current
  * user's) domain from Vadmin.
  *
  * @param boolean $plain_text    When TRUE, obtains the plain text
  *                               footer, otherwise the HTML version
  *                               thereof is returned (OPTIONAL; default
  *                               is TRUE).
  * @param string  $vadmin_domain The domain for which to retrieve
  *                               the footer (OPTIONAL; default
  *                               empty, in which case the current
  *                               login user's domain is used)
  *
  * @return string The requested email footer
  *
  */
function get_email_footer_from_vadmin($plain_text=TRUE, $vadmin_domain='')
{

   // obtain the current user's domain if domain not given
   //
   if (empty($vadmin_domain))
   {
      global $domain, $username;
      if (($pos = strpos($username, '@')) !== FALSE)
         $vadmin_domain = substr($username, $pos + 1);
      else
         $vadmin_domain = $domain;
   }


   // return nothing if not enabled
   //
   $email_footer_enabled = vadmin_get_pref($vadmin_domain, 'email_footer_enabled');
   if (!$email_footer_enabled) return '';


   // plain text footer
   //
   if ($plain_text)
   {
      $footer = ef_htmlspecialchars_decode(vadmin_get_pref($vadmin_domain, 'email_footer_plain_text'));
      if (empty($footer)) return '';
      else return $footer;
   }


   // HTML footer
   //
   else
   {
      $footer = ef_htmlspecialchars_decode(vadmin_get_pref($vadmin_domain, 'email_footer_html'));
      // if not found, we could fall back to a converted version of
      // the plain text footer, but we want to respect if the
      // administrator set it to empty
      if (empty($footer)) return '';
      else return $footer;
   }

}



/**
  * Add a menu item to the Vadmin domain prefs/user permissions
  * menu to let administrators control the email footer in the
  * Vadmin interface
  *
  */
function email_footer_vadmin_menu_do($args)
{

   $menu_item = '';
   $vadmin_domain = $args[0];
   $desig = $args[1];
   global $vadmin_email_footer_enable_admin_levels,
          $vadmin_email_footer_admin_levels,
          $enable_footer_configuration_in_vadmin,
          $plugins;
   email_footer_init();
   $orig_text_domain = sq_change_text_domain('email_footer');


   if (!$enable_footer_configuration_in_vadmin)
      return '';


   // determine if email footer is currently enabled for this domain
   //
   $email_footer_enabled = vadmin_get_pref($vadmin_domain, 'email_footer_enabled');


   // create menu item for enabled switch if needed
   //
   if (in_array($desig, $vadmin_email_footer_enable_admin_levels))
   {
      $menu_item .= '  <tr>'
                 . '   <td>'
                 . '    <input type="checkbox" name="email_footer_enabled" id="email_footer_enabled" value="1"'
                 . (!empty($email_footer_enabled) ? ' checked="yes"' : '')
                 . ' /><label for="email_footer_enabled"> ' . _("Add footer to all outgoing emails")
                 . '   </label></td>'
                 . '  </tr>';
   }


   // create menu item(s) for email footer(s) if needed
   //
   if ($email_footer_enabled
    && in_array($desig, $vadmin_email_footer_admin_levels))
   {
      $email_footer_plain_text = get_email_footer_from_vadmin(TRUE, $vadmin_domain);
      $menu_item .= '  <tr>'
                 . '   <td>'
                 . (in_array('html_mail', $plugins) ? _("Footer to be added to all outgoing plain text emails") : _("Footer to be added to all outgoing emails"))
                 . '<br /><textarea cols="50" rows="5" name="email_footer_plain_text" id="email_footer_plain_text">'
                 . (!empty($email_footer_plain_text) ? $email_footer_plain_text : '')
                 . '</textarea>'
                 . '   </td>'
                 . '  </tr>';


      if (in_array('html_mail', $plugins))
      {
         $email_footer_html = get_email_footer_from_vadmin(FALSE, $vadmin_domain);
         $menu_item .= '  <tr>'
                    . '   <td>'
                    . _("Footer to be added to all outgoing HTML emails")
                    . '<br /><textarea cols="50" rows="5" name="email_footer_html" id="email_footer_html">'
                    . (!empty($email_footer_html) ? $email_footer_html : '')
                    . '</textarea>'
                    . '   </td>'
                    . '  </tr>';
      }
   }


   sq_change_text_domain($orig_text_domain);
   return $menu_item;

}



/**
  * Add a menu item to the Vadmin domain prefs/user permissions
  * menu to let administrators control the email footer in the
  * Vadmin interface
  *
  */
function email_footer_vadmin_menu_submit_do($args)
{

   $me = 'email_footer_vadmin_menu_submit_do';
   $vadmin_domain = $args[0];
   $desig = $args[1];
   global $vadmin_email_footer_enable_admin_levels,
          $vadmin_email_footer_admin_levels,
          $enable_footer_configuration_in_vadmin,
          $plugins;
   email_footer_init();
   $orig_text_domain = sq_change_text_domain('vadmin_cronomagic');
   $output = '';


   if (!$enable_footer_configuration_in_vadmin)
      return '';


   $orig_email_footer_enabled = vadmin_get_pref($vadmin_domain, 'email_footer_enabled');


   // set value for enabled switch if needed
   //
   if (in_array($desig, $vadmin_email_footer_enable_admin_levels))
   {

      // look for our checkbox and set pref accordingly
      //
      $email_footer_enabled = vadmin_getvar('POST', 'email_footer_enabled');
      if (isset($email_footer_enabled) && $email_footer_enabled == '1')
         $email_footer_enabled = 1;
      else
         $email_footer_enabled = FALSE;


      if ($email_footer_enabled != $orig_email_footer_enabled)
      {
         spew("$me: changing the preference for email footer enabled");
         vadmin_put_pref($vadmin_domain, 'email_footer_enabled', $email_footer_enabled);
         if ($email_footer_enabled)
            $result = _("Enabled");
         else
            $result = _("Disabled");
      }
      else
         $result = _("No change");


      $output .= '<tr>'
              . ' <td>' . _("Changing email footer settings") . ':</td>'
              . ' <td>' . $result . '</td>'
              . '</tr>';

   }


   // save email footer(s) if necessary
   //
   if ($email_footer_enabled && $orig_email_footer_enabled
    && in_array($desig, $vadmin_email_footer_admin_levels))
   {

      // look for our text input and set pref accordingly
      //
      $email_footer_plain_text = vadmin_getvar('POST', 'email_footer_plain_text');
      if (empty($email_footer_plain_text))
         $email_footer_plain_text = FALSE;

      if ($email_footer_plain_text != vadmin_get_pref($vadmin_domain, 'email_footer_plain_text'))
      {
         spew("$me: changing the preference for email footer (plain text)");
         vadmin_put_pref($vadmin_domain, 'email_footer_plain_text', $email_footer_plain_text);
         if ($email_footer_plain_text)
            $result = _("Saved");
         else
            $result = _("Deleted");
      }
      else
         $result = _("No change");


      $output .= '<tr>'
              . ' <td>'
              . (in_array('html_mail', $plugins) ? _("Changing email footer (plain text)") : _("Changing email footer"))
              . ':</td>'
              . ' <td>' . $result . '</td>'
              . '</tr>';


      // do the same for the HTML footer
      //
      if (in_array('html_mail', $plugins))
      {

         $email_footer_html = vadmin_getvar('POST', 'email_footer_html');
         if (empty($email_footer_html))
            $email_footer_html = FALSE;
   
         if ($email_footer_html != vadmin_get_pref($vadmin_domain, 'email_footer_html'))
         {
            spew("$me: changing the preference for email footer (html)");
            vadmin_put_pref($vadmin_domain, 'email_footer_html', $email_footer_html);
            if ($email_footer_html)
               $result = _("Saved");
            else
               $result = _("Deleted");
         }
         else
            $result = _("No change");


         $output .= '<tr>'
                 . ' <td>' . _("Changing email footer (HTML)") . ':</td>'
                 . ' <td>' . $result . '</td>'
                 . '</tr>';

      }

   }


   sq_change_text_domain($orig_text_domain);
   return $output;

}



/**
  * Invert htmlspecialchars() for PHP 4 and PHP 5
  *
  * @param string $string The string to be converted
  *                       (htmlspecialchars reversed)
  * @param int    $style  The quote style originally
  *                       used in htmlspecialchars encoding
  *
  * @return string The converted string
  *
  */
function ef_htmlspecialchars_decode($string, $style=ENT_COMPAT)
{
   if (function_exists('htmlspecialchars_decode'))
      return htmlspecialchars_decode($string);

   $translation = array_flip(get_html_translation_table(HTML_SPECIALCHARS, $style));
   if ($style === ENT_QUOTES) $translation['&#039;'] = '\'';
   return strtr($string, $translation);
}



