<?php

/**
 * autorespond/lib.php
 * 
 * The primary function library for the SquirrelMail "autorespond" plugin,
 * which allows a user to modify an email forward file and vacation files
 * over FTP, using their IMAP credentials for authentication.
 * 
 * @copyright Copyright (c) 2002-2007 O'Shaughnessy Evans <shaug-spamrule @ wumpus.org>
 * @version $Id: lib.php,v 1.20 2007/09/11 18:52:13 shaug Exp $
 * @license http://opensource.org/licenses/artistic-license-2.0.php
 * @package plugins
 * @subpackage autorespond
 */

/*
 * Table of Contents:
 * 
 * ar_print_header($mesg, $columns)
 * ar_print_footer($mesg, $columns)
 * ar_ftp_get($path)
 * ar_ftp_put($path, $data)
 * ar_syslog($message)
 * ar_change_autoresponse()
 * ar_install_autoresponse()
 * ar_read_vacation()
 * ar_edit_vacation()
 */


/**
 * ar_print_header($mesg, $columns)
 * 
 * Print out the beginning of a table, with $mesg in the first row.
 * 
 * @param string $mesg  text to print out
 * @param int $columns  how many columns will be in the table
 * 
 * Returns:
 *   nothing
 */
function ar_print_header($mesg, $columns)
{
    global $color;

    $title = _("Options") . ' - '. ar_gettext("Automatic Forward and Reply");

    echo <<<EOtable_top
<br>
<table bgcolor="{$color[0]}" border=0 width=95%
       cellspacing=0 cellpadding=2 align=center>
  <tr bgcolor="{$color[0]}">
    <th>{$title}</b></th>
  </tr>
  <tr><td>
    <table bgcolor="{$color[4]}" border=0 width=100% cellspacing=0 cellpadding=5 align=center valign=top>
      <tr align=center bgcolor="{$color[4]}">
        <td colspan=$columns>
EOtable_top;

    if (isset($mesg)) {
        echo "<p>{$mesg}</p>\n";
    }

    echo "</td>\n</tr>\n\n";
}


/**
 * ar_print_footer($mesg, $columns)
 * 
 * Print out the end of a table, with $mesg in the last row.
 * 
 * @param string $mesg  text to print out
 * @param int $columns  how many columns the table has
 * 
 * Returns:
 *   nothing
 */
function ar_print_footer($mesg, $columns)
{
    global $color;

    if (isset($mesg)) {
        echo <<<EOmesg

  <tr align=center bgcolor="{$color[4]}">
    <td colspan=$columns>
      <p>{$mesg}</p>
    </td>
  </tr>
EOmesg;
    }

    echo <<<EOfooter

    </table>
  </td></tr>
</table>

EOfooter;
}



/**
 * ar_ftp_get($path)
 * 
 * Downloads the given file name from the user's account on
 * $AUTORESPOND_OPTS['ftphost'] and returns the contents as an array.
 * 
 * @param string $path  path to the file to be downloaded via FTP
 * 
 * @returns array (boolean status, string info)
 *   2-element array:
 *   On success, the 1st element will be TRUE and the 2nd will contain a
 *   string with the contents of the requested file.
 *   On failure, the 1st element will be FALSE and the 2nd will contain
 *   an error message describing the problem.
 */
function ar_ftp_get($path)
{
    global $AUTORESPOND_OPTS, $key, $onetimepad, $username;
    sqgetGlobalVar('AUTORESPOND_OPTS', $AUTORESPOND_OPTS, SQ_SESSION);
    sqgetGlobalVar('key', $key, SQ_COOKIE);
    sqgetGlobalVar('onetimepad', $onetimepad, SQ_SESSION);
    sqgetGlobalVar('username', $username, SQ_SESSION);

    $ftphost = $AUTORESPOND_OPTS['ftphost'];
    if (!$ftphost) {
        return array(FALSE, ar_gettext("Sorry, but this plugin is not ".
                            "completely set up.  Please contact your System ".
                            "Administrator about configuring the Autorespond ".
                            "plugin."));
    }

    if (empty($path)) {
        return array(FALSE, ar_gettext("no file was given to upload"));
    }

    // decrypt the user's password so we can pass it to the ftp site
    $password = OneTimePadDecrypt($key, $onetimepad);
    //echo "connecting to $username@$ftphost ($key/$onetimepad => '$password')<br>\n";

    // upload the file to the user's home
    $ftp = ftp_connect($ftphost);
    if (!$ftp) {
        return array(FALSE, ar_gettext("cannot connect to"). " $ftphost".
                            ($php_errormsg ? ": $php_errormsg" : ''));
    }
    $stat = ftp_login($ftp, $username, $password);
    if (!$stat) {
        return array(FALSE, ar_gettext("cannot log in to"). " $ftphost".
                            ($php_errormsg ? ": $php_errormsg" : ''));
    }

    if (! @ftp_chdir($ftp, dirname($path))) {
        return array(FALSE);
    }

    $file = basename($path);
    if (! @ftp_size($ftp, $file)) {
        return array(FALSE);
    }

    // create a local temp file to store the rules
    $temp = tmpfile();
    $stat = @ftp_fget($ftp, $temp, $file, FTP_ASCII);
    @ftp_quit($ftp);

    // put each line of the temp file into an array
    rewind($temp);
    $data = array();
    while (fstat($temp) && !feof($temp)) {
        array_push($data, fgets($temp, 1024));
    }
    fclose($temp);
    return array(TRUE, $data);

    // download the mailfilter file and return it in an array
    //$data = @file("ftp://$username:$password@$ftphost/$to_get");
    //echo "data is: <pre>"; print_r($data); echo "</pre>\n";
    //return isset($data) ? array_map('trim', $data) : array();
}


/**
 * ar_ftp_put($path, $data)
 * 
 * Uploads the text in $data to $path in the user's account on 
 * $AUTORESPOND_OPTS['ftphost'].
 * 
 * @param string $path  the file to be modified
 * @param string $data  the data to be written to $path; if it's an empty string, $path will be removed
 * 
 * Returns:
 *   nothing
 */
function ar_ftp_put($path, $data)
{
    global $AUTORESPOND_OPTS, $key, $onetimepad, $username;
    sqgetGlobalVar('AUTORESPOND_OPTS', $AUTORESPOND_OPTS, SQ_SESSION);
    sqgetGlobalVar('key', $key, SQ_COOKIE);
    sqgetGlobalVar('onetimepad', $onetimepad, SQ_SESSION);
    sqgetGlobalVar('username', $username, SQ_SESSION);

    $ftphost = $AUTORESPOND_OPTS['ftphost'];
    if (!$ftphost) {
        return array(FALSE, ar_gettext("Sorry, but this plugin is not ".
                            "completely set up.  Please contact your System ".
                            "Administrator about configuring the Autorespond ".
                            "plugin."));
    }

    if (!$path) {
        return array(FALSE, ar_gettext("no file was given to upload"));
    }

    // decrypt the user's password so we can pass it to the ftp site
    // (borrowed from the vacation plugin; thanks!)
    $password = OneTimePadDecrypt($key, $onetimepad);

    // upload the file to the user's home
    $ftp = ftp_connect($ftphost);
    if (!$ftp) {
        return array(FALSE, ar_gettext("cannot connect to"). " $ftphost".
                            ($php_errormsg ? ": $php_errormsg" : ''));
    }
    $stat = ftp_login($ftp, $username, $password);
    if (!$stat) {
        return array(FALSE, ar_gettext("cannot log in to"). " $ftphost".
                            ($php_errormsg ? ": $php_errormsg" : ''));
    }
    // if there is data to upload, write it to a temporary file and upload
    // its contents
    if ($data) {
        // create a file w/the new rules
        $temp = tmpfile();
        fwrite($temp, $data);
        rewind($temp);

        // search the path to the file and create each parent dir if it
        // doesn't exist
        $dir = dirname($path);
        $dirs = preg_split('/[\\/\\\]+/', $dir, -1, PREG_SPLIT_NO_EMPTY);
        foreach ($dirs as $d) {
            if ($d == '.') {
                continue;
            }
            if ($d == '/') {
                continue;
            }
            if (! @ftp_chdir($ftp, $d)) {
                if (!(@ftp_mkdir($ftp, $d) && @ftp_site($ftp, "chmod 0700 $d")
                     && @ftp_chdir($ftp, $d))) {
                    return array(FALSE, ar_gettext("error creating"). " $dir".
                                 ($php_errormsg ? ": $php_errormsg" : ''));
                }
            }
        }

        $cwd = ftp_pwd($ftp);
        /*
        if ($cwd != "/" and $cwd != $dir and $cwd != "/$dir") {
            return array(FALSE, "error creating $dir (could only get to ".
                            "$cwd): $php_errormsg");
        }
        */

        $file = basename($path);
        $stat = ftp_fput($ftp, $file, $temp, FTP_ASCII);
        if (!$stat) {
            @ftp_close($ftp);
            return array(FALSE, ar_gettext("error changing"). " $cwd/$file".
                                ($php_errormsg ? ": $php_errormsg" : ''));
        }
        @ftp_site($ftp, "chmod 0600 $file");
        fclose($temp);
    }
    // otherwise delete the remote file (we don't want to leave an empty file)
    else {
        $stat = @ftp_delete($ftp, $path);
        if (!$stat) {
            @ftp_close($ftp);
            return array(FALSE, ar_gettext("error removing"). " $path".
                                ($php_errormsg ? ": $php_errormsg" : ''));
        }
    }

    @ftp_quit($ftp);

    return array(TRUE);
}


/**
 * ar_syslog($message)
 * 
 * Write a message to the syslog.
 * 
 * @param string $message  text to send
 * 
 * Returns:
 *   nothing
 */
function ar_syslog($message)
{
    define_syslog_variables();
    openlog('forward', LOG_NDELAY, LOG_DAEMON);
    syslog(LOG_ERR, $message);
    closelog();
}


/**
 * ar_gettext($str)
 * 
 * Switch the gettext domain to autorespond's before reading the localized
 * version of astring, then switch the translation domain back to squirrelmail
 * 
 * @param string $str  string to be translated
 * 
 * @returns string containing gettext-translated text
 */
function ar_gettext($str)
{
    $localstr = '';

    bindtextdomain('autorespond', SM_PATH. 'plugins/autorespond/locale');
    textdomain('autorespond');

    $localstr = _($str);

    bindtextdomain('squirrelmail', SM_PATH. 'locale');
    textdomain('squirrelmail');

    return $localstr;
}


/**
 * ar_change_autoresponse()
 * 
 * Prints out a form allowing the user to change various options for
 * processing new mail (typically the things you might want to do in a
 * .forward file).  These include:
 *   - forwarding to another address
 *   - enabling and modifying a vacation message
 *   - emptying your vacation message's reply cache
 *   - keeping mail locally, either filtered or unfiltered
 * 
 * The user's forward_file and vacation_file are loaded before the form is
 * printed out.  Both are processed to set defaults for the form.  An effort
 * is made to make as few assumptions as possible about the forward file so
 * that external editing of it won't confuse this plugin, but any changes
 * outside the scope of the features that this plugin can make will most
 * likely still be lost.
 * 
 * Parameters:
 *   none
 * 
 * Returns:
 *   nothing
 */
function ar_change_autoresponse()
{
    global $AUTORESPOND_OPTS, $color, $trash_folder;
    sqgetGlobalVar('AUTORESPOND_OPTS', $AUTORESPOND_OPTS, SQ_SESSION);
    sqgetGlobalVar('trash_folder', $trash_folder, SQ_FORM);

    $addrs = array();
    $aliases = '';
    $check_fwd = '';
    $check_keep = '';
    $check_trash = '';
    $check_vacation = '';
    $keep_nofilt = '';
    $keep_filt = '';

    // download and process the installed forward file
    list($stat, $forward) = ar_ftp_get($AUTORESPOND_OPTS['forward_file']);
    if ($stat === FALSE) {
        print ar_gettext("There was a problem connecting to your FTP server:").
              '&nbsp;&nbsp;'. ($forward ? "\"$forward\".&nbsp;&nbsp;" : '').
              ar_gettext("Please contact your support department."). "<br>\n";
    }
    else if (! empty($forward)) {
        foreach ($forward as $line) {
            $line = trim($line);
            // skip if the line is empty
            if (!$line) {
                continue;
            }

            // if it looks like a disabled forward that we created
            // earlier, remember it as the default forward address
            $matches = array();
            if (@ereg('^# forward: *([^ ].*)', $line, $matches)) {
                array_push($addrs, $matches[1]);
                continue;
            }
            // skip other comments
            else if (@ereg('^#', $line)) {
                continue;
            }

            // split it by commas since sendmail allows those
            // then run through each component
            // (blindly hoping there's nothing like this:  abc, "this, that")
            $splitline = preg_split('/\s*,\s*/', $line);
            foreach ($splitline as $f) {
                // if the forward entry is a path,
                // either it's being put into the Trash
                // or we will presume that means a local copy is being kept
                $matches = array();
                // does the line look like a path?
                if (@preg_match('/^[\.\/\\\\]\S+/', $f, $matches)) {
                    // now see if the end matches our trash_folder
                    if (strpos(strrev($matches[0]), strrev($trash_folder))
                        === 0) {
                        $check_trash = 'checked';
                    }
                    else {
                        $keep_nofilt = $check_keep = 'checked';
                        $keep_filt = '';
                    }
                }
                // if it matches our vacation regex, turn on
                // the reply section's checked flag for the form,
                // then look for aliases in the vacation command
                else if ((isset($AUTORESPOND_OPTS['vacation_pattern']) &&
                 @preg_match($AUTORESPOND_OPTS['vacation_pattern'], $f)) ||
                 (isset($AUTORESPOND_OPTS['vacation_pcre']) &&
                 @preg_match($AUTORESPOND_OPTS['vacation_pcre'], $f))) {
                    $vac_aliases = array();
                    $check_vacation = 'checked';
                    if (@preg_match_all($AUTORESPOND_OPTS['vacation_aliases_pat'],
                                        $f, $vac_aliases)) {
                        $aliases = implode(', ', $vac_aliases[1]);
                    }
                }
                // if it matches our filter command pattern, turn on the
                // keep checkbox and the filter option
                else if (@preg_match($AUTORESPOND_OPTS['filter_pattern'], $f)) {
                    $keep_filt = $check_keep = 'checked';
                    $keep_nofilt = '';
                }
                // ignore other pipes
                else if (@ereg('^\"?\|', $f)) {
                    continue;
                }
                // finally, if the line matches anything else, we presume
                // it's an email address for forwarding
                else {
                    $check_fwd = 'checked';
                    array_push($addrs, $f);
                }
            }
        }
    }

    if (empty($addrs)) {
        $def_fwd = '';
    }
    else {
        $def_fwd = implode(', ', $addrs);
    }

    // If a forwarding address wasn't defined in the .forward and
    // no "keep a copy" option was defined there either, we may
    // want to enable 'keep a copy' to simplify life for vacation
    // users that don't realize they won't have any new mail without
    // enabling this.  See 'keep_by_default' in options.php to change
    // the default.
    if (!$check_fwd && !$keep_nofilt && !$keep_filt && 
        $AUTORESPOND_OPTS['keep_by_default'] === TRUE) {
        $keep_nofilt = $check_keep = 'checked';
        $keep_filt = '';
    }

    // If no vacation aliases were pulled from the forward_file, we should
    // load a default set from the squirrelmail preferences
    if (!$aliases) {
        global $data_dir;
        sqgetGlobalVar('username', $username, SQ_SESSION);
        $aliases = getPref($data_dir, $username, 'autorespond_aliases');
    }

    list($subject, $message) = ar_read_vacation();
    if (!$subject) {
        $subject = $AUTORESPOND_OPTS['default_subject'];
    }
    if (!$message) {
        $message = $AUTORESPOND_OPTS['default_message'];
    }

    echo <<<EOform_top
<tr bgcolor="{$color[12]}">
  <td colspan=3><form action="options.php" method=GET>
EOform_top;

    print '<b>'. ar_gettext("You can choose one or more of these options:").
          "</br>\n";

    echo <<<EOform_top
  </td>
</tr>

EOform_top;

    // show the text input for the forward address
    $desc = $AUTORESPOND_OPTS['forward_desc'];
    if ($desc) {
        $to_label = ar_gettext("To:");
        $check_label = ar_gettext("Forward?");
        echo <<<EOforward
    <tr align=left valign=top bgcolor="{$color[4]}">
      <td><input type=checkbox name=forward $check_fwd></input></td>
      <td><b>$check_label</b></td>
      <td>$desc<br><br>
        <i>$to_label</i> &nbsp;&nbsp;
        <input type=text name=addr size=40 value="$def_fwd"></input>
      </td>
    </tr>

EOforward;
    }

    // show the vacation subject and message inputs
    $desc = $AUTORESPOND_OPTS['vacation_desc'];
    if ($desc) {
        $subj_label = ar_gettext("Subject:");
        $mesg_label = ar_gettext("Message:");
        $check_label = ar_gettext("Reply?");
        echo <<<EOvacation
    <tr align=left valign=top bgcolor="$color[12]">
      <td><input type=checkbox name=vacation $check_vacation></input></td>
      <td><b>$check_label</b></td>
      <td>$desc<br><br>
        <i>$subj_label</i> &nbsp;&nbsp;
        <input type=text name=subject size=60 value="$subject"></input><br>
        <br><i>$mesg_label</i><br>
        <textarea name=message rows=10 cols=65>$message</textarea>
        <br><br>
EOvacation;

        // show the aliases checkbox and input box
        $desc = $AUTORESPOND_OPTS['aliases_desc'];
        if ($desc) {
            $opt_title = ar_gettext("Aliases:");
            echo <<<EOaliases
        <i>$opt_title</i> &nbsp;&nbsp;
        <input type=text name=aliases size=60 value="$aliases"></input>
        <br><small><i>$desc</i></small>
        <br><br>
EOaliases;
        }

        // show the reset-cache checkbox
        $desc = $AUTORESPOND_OPTS['empty_cache_desc'];
        if ($desc) {
            $opt_title = ar_gettext("Empty reply cache?");
            echo <<<EOcache
        <input type=checkbox name=reset_cache></input>
        <i>$opt_title</i><br>
        <small><i>$desc</i></small>
EOcache;
        }


        echo <<<EOvacation
        <br>
      </td>
    </tr>
EOvacation;
    }

    // show the "keep a local copy" checkboxes:
    // In $AUTORESPOND_OPTS, we check whether keep_desc is set.  This will
    // enable the whole form section that lets a user enable storing their
    // mail locallly, either via the keep_string method (typically "\username"
    // in a .forward file) or the filter_string method (typically a pipe to
    // procmail or maildrop).  If filter_string is defined, we show radio
    // boxes allowing the user to choose whether to filter or not; if it's not
    // defined, we don't show the radio boxes and assume the keep_string is
    // defined to allow the user to enable local retention w/o filtering.
    $desc = $AUTORESPOND_OPTS['keep_desc'];
    if ($desc) {
        $unfilt_label = ar_gettext("unfiltered");
        $filt_label = ar_gettext("filtered");
        $check_label = ar_gettext("Keep a copy here?");
        $keep_text = isset($AUTORESPOND_OPTS['filter_string'])
         ? '<input type=radio name=keeptype value=unfiltered '.$keep_nofilt.'>'.
           "<i>$unfilt_label</i>&nbsp;&nbsp;\n".
           '<input type=radio name=keeptype value=filtered '. $keep_filt. '>'.
           "<i>$filt_label</i>\n"
         : "&nbsp;\n";
        echo <<<EOkeep
    <tr align=left valign=top bgcolor="{$color[4]}">
      <td><input type=checkbox name=keep $check_keep></td>
      <td><b>$check_label</b></td>
      <td>$desc<br><br>
        $keep_text
      </td>
    </tr>

EOkeep;
    }

    // show the "send it to the Trash" option
    $desc = $AUTORESPOND_OPTS['trash_desc'];
    if ($desc) {
        $check_label = ar_gettext("Sent it to your Trash?");
        echo <<<EOkeep
<tr align=left valign=top bgcolor="{$color[12]}">
  <td><input type=checkbox name=trash $check_trash></td>
  <td><b>$check_label</b></td>
  <td>$desc
  </td>
</tr>

EOkeep;
    }

    // close out the form
    $finish_label = ar_gettext("Finish");
    echo <<<EOend
    <tr align=left>
      <td align=left colspan=3> 
        <br>
        <input type=submit name=action value="$finish_label"> &nbsp;
        <input type=reset> &nbsp;
        </form>
      </td>
    </tr>
EOend;
}


/**
 * ar_install_autoresponse()
 * 
 * Reads the data submitted by the autoresponse options form and
 * installs it all in the user's forward and vacation files via FTP,
 * printing status messages along the way:
 *   - possibly adds a forwarding address
 *   - possibly installs a vacation message, with or without aliases
 *   - possibly empties out the existing vacation reply cache
 *   - possibly keeps mail stored locally, piped through a filter
 *     or just sent directly to the inbox
 *   - possibly sends all mail to the trash
 * 
 * Parameters:
 *   none
 * 
 * Returns:
 *   nothing
 */
function ar_install_autoresponse()
{
    global $AUTORESPOND_OPTS, $color, $addr, $aliases, $forward, $keep,
           $reset_cache, $trash, $trash_folder, $vacation;
    sqgetGlobalVar('AUTORESPOND_OPTS', $AUTORESPOND_OPTS, SQ_SESSION);
    sqgetGlobalVar('addr', $addr, SQ_FORM);
    sqgetGlobalVar('aliases', $aliases, SQ_FORM);
    sqgetGlobalVar('forward', $forward, SQ_FORM);
    sqgetGlobalVar('keep', $keep, SQ_FORM);
    sqgetGlobalVar('reset_cache', $reset_cache, SQ_FORM);
    sqgetGlobalVar('trash', $trash, SQ_FORM);
    sqgetGlobalVar('trash_folder', $trash_folder);
    sqgetGlobalVar('vacation', $vacation, SQ_FORM);
    $data = '';

    if ($addr) {
        if ($forward) {
            $data = "$addr\n";
            $mesg = ar_gettext("New mail will be sent to"). ' '.
                    '<a href="mailto:'. htmlspecialchars($addr).
                    '"><i>'. htmlspecialchars($addr). '</i></a>.';
        }
        else
            $data = "# forward: $addr\n";
    }
    if (!$forward) {
        $mesg = ar_gettext("New mail will be kept here in your Inbox.");
    }

    // build up a list of vacation command options for each alias provided
    // by the form, then create the vacation command string from this
    // and the predefined command string
    //
    // note that we want to process the aliases whether or not a vacation
    // is being saved so that we can remember the old aliases when the
    // vacation message is being disabled.  it will save the user from
    // having to put them in every time they reactivate their vacation reply.
    if ($aliases) {
        $aliases_pref = array();
        $alias_opts = '';
        $alias_list = preg_split('/[\s,]+/', $aliases);
        if (sizeof($alias_list) > 0) {
            foreach ($alias_list as $alias_addr) {
                if (preg_match('/^\S+$/', $alias_addr)) {
                    $mesg .= "<br>\n... ". htmlspecialchars($alias_addr). ' '.
                             ar_gettext("is recognized as an alias.");
                    $alias_opts .=
                               sprintf($AUTORESPOND_OPTS['vacation_alias'],
                               $alias_addr);
                    array_push($aliases_pref, $alias_addr);
                }
            }
            if (!empty($aliases_pref)) {
                global $data_dir;
                sqgetGlobalVar('username', $username, SQ_SESSION);
                setPref($data_dir, $username, 'autorespond_aliases',
                        implode(', ', $aliases_pref));
            }
        }
    }

    // if the vacation feature was enabled, recreate the vacation_file
    // and upload it, then append the vacation_string to the forward data
    if ($vacation) {
        sqgetGlobalVar('message', $message, SQ_FORM);
        sqgetGlobalVar('subject', $subject, SQ_FORM);
        $vacdata = '';

        // build up the new vacation message file in $vacdata:
        // from and subject headers, then a newline and the message body.

        // include a From header in the vacation message derived from
        // the address the user has defined in squirrelmail's prefs.
        // this can be disabled by changing the vacation_from config var
        // to FALSE.  this is necessary for platforms where vacation
        // doesn't support a From header (e.g. Solaris 8 & before)
        if ($AUTORESPOND_OPTS['vacation_from'] !== FALSE) {
            $name = getPref($data_dir, $username, 'full_name');
            $email = getPref($data_dir, $username, 'email_address');
            if (!empty($name) && !empty($email)) {
                $vacdata .= "From: $name <$email>\n";
            }
        }

        $vacdata .= "Subject: ". $subject. "\n\n". $message;

        // convert <return><newline> to <newline> in the vacation file
        // because the form may pass it to us with the extra returns
        $vacdata = str_replace("\r\n", "\n", $vacdata);

        // ensure the new vacation file ends with a newline
        if (! @ereg("\n$", $vacdata)) {
            $vacdata .= "\n";
        }

        list($stat, $ftpmesg) = ar_ftp_put($AUTORESPOND_OPTS['vacation_file'],
         $vacdata);
        if ($stat === FALSE) {
            print ar_gettext("There was a problem uploading your vacation ".
                             "file:").
                  '&nbsp;&nbsp;'. ($ftpmesg ? "\"$ftpmesg\".&nbsp;&nbsp;" : '').
                  ar_gettext("Please contact your support department.").
                  "<br>\n";
        }
        $data .= sprintf($AUTORESPOND_OPTS['vacation_string'], $alias_opts).
                 "\n";

        $mesg .= "<br>\n". ar_gettext("Senders will get an automatic reply.");
    }
    else {
        $mesg .= "<br>\n". ar_gettext("No automatic reply will be sent.");
    }

    // empty out the user's vacation reply cache if requsted
    if ($reset_cache) {
        $mesg .= "<br>\n". ar_gettext("Your reply cache has been reset.");
        // use FTP to remove the vacation_cache file from the user's account
        ar_ftp_put($AUTORESPOND_OPTS['vacation_cache'], '');
    }

    // if the user requested to keep local mail, build up a filter_string
    // or keep_string and add it to the forward file data
    if ($keep) {
        sqgetGlobalVar('keeptype', $keeptype, SQ_FORM);
        if ($keeptype === 'filtered') {
            $data = $AUTORESPOND_OPTS['filter_string']. "\n" . $data;
            if ($addr) {
                $mesg .= "<br>\n". ar_gettext("Local copies will also be kept ".
                         "(and will be filtered through"). ' '.
                         $AUTORESPOND_OPTS['filter_descr']. ").";
            }
            else {
                $mesg .= "<br>\n..." . ar_gettext("filtered through"). ' '.
                         $AUTORESPOND_OPTS['filter_descr']. '.';
            }
        }
        else {
            $data = $AUTORESPOND_OPTS['keep_string']. "\n" . $data;
            if ($addr) {
                $mesg .= "<br>\n".
                         ar_gettext("Local copies will also be kept.");
            }
        }
    }

    // add a directive to send all mail to the trash folder if requested
    if ($trash) {
        $trash = @ereg('^[\.\/\\]', $trash_folder) ? $trash_folder
                 : "./$trash_folder";
        $mesg .= "<br>\n".
                 ar_gettext("Mail will be kept in your Trash mailbox.");
        $data .= $trash. "\n";
    }

    // upload the new forward file with all the commands that we just
    // put together
    list ($stat, $ftpmesg) = ar_ftp_put($AUTORESPOND_OPTS['forward_file'],
                                        $data);
    if ($stat === FALSE) {
        print ar_gettext("There was a problem uploading your forward file:").
              '&nbsp;&nbsp;'. ($ftpmesg ? "\"$ftpmesg\".&nbsp;&nbsp;" : '').
              ar_gettext("Please contact your support department."). "<br>\n";
    }

    // tell the user what features were just activated
    echo <<<EOmesg

    <tr align=left valign=top bgcolor="$color[12]">
      <td>&nbsp;</td>
      <td><b>
EOmesg;
    print ar_gettext("New settings saved:"). "</b></td>\n";
    echo <<<EOmesg
      <td><font color=green>$mesg</font></td>
    </tr>
    <tr align=left>
      <td colspan=3>&nbsp;</td>
    </tr>
EOmesg;
}


/**
 * ar_read_vacation()
 * 
 * Load the user's vacation message via FTP.
 * 
 * @returns array:  (string subject, string message body (individual lines joined together by newlines))
 */
function ar_read_vacation()
{
    global $AUTORESPOND_OPTS;
    sqgetGlobalVar('AUTORESPOND_OPTS', $AUTORESPOND_OPTS, SQ_SESSION);
    $subject = '';
    $message = '';

    // download and process the installed vacation message
    list($stat, $vacation) = ar_ftp_get($AUTORESPOND_OPTS['vacation_file']);
    if ($stat === FALSE) {
        print ar_gettext("There was a problem connecting to your FTP server:").
              '&nbsp;&nbsp;'. ($vacation ? "\"$vacation\".&nbsp;&nbsp;" : '').
              ar_gettext("Please contact your support department."). "<br>\n";
    }
    else if ($stat === TRUE && !empty($vacation)) {
        // Look for headers at the beginning of the file and strip them out.
        // If we find the subject, store it to return to the calling function.
        // The rest of the file is the message body, returned as a string
        // of joined lines.
        $hdr = array();

        while (preg_match('/^(\S+):\s+(.*)/', $vacation[0], $hdr)) {
            if (!strcasecmp($hdr[1], 'Subject')) {
                $subject = $hdr[2];
            }
            array_shift($vacation);
            reset($vacation);
        }

        // if we stripped headers, the vacation may still have a leading
        // newline or return/newline that needs to be stripped
        if (@ereg("^\r?\n?$", $vacation[0])) {
            array_shift($vacation);

        }
        $message = implode('', $vacation);
    }

    return array($subject, $message);
}


/**
 * ar_edit_vacation()
 * 
 * Function unused at present, but designed to allow a popup or otherwise
 * standalone window to edit a vacation file.  Provides a form with inputs
 * for changing the subject and email message body of the vacation message.
 * 
 * Parameters:
 *   none
 * 
 * Returns:
 *   nothing
 */
function ar_edit_vacation()
{
    global $AUTORESPOND_OPTS, $color, $message, $subject;
    sqgetGlobalVar('message', $message, SQ_FORM);
    sqgetGlobalVar('subject', $subject, SQ_FORM);

    // if a message and subject weren't defined already (can this ever happen?),
    // we need to read them from the user's vacation file
    //
    // if they were set, then we need to build up a new vacation file message
    // and upload it to the account's home via ftp
    if (!$subject && !$message) {
        list($subject, $message) = ar_read_vacation();
    }
    else {
        if (! @ereg("\n$", $message)) {
            $message .= "\n";
        }
        $data = "Subject: $subject\n\n". $message;
        $data = str_replace("\r\n", "\n", $data);
        list($stat, $ftpmesg) = ar_ftp_put($AUTORESPOND_OPTS['vacation_file'],
         $data);
        if ($stat === FALSE) {
            print ar_gettext("There was a problem uploading your vacation ".
                  "file:").
                  '&nbsp;&nbsp;"'. $ftpmesg. "\".&nbsp;&nbsp;";
                  ar_gettext("Please contact your support department.").
                  "<br>\n";
        }
    }

    // set vacation subject and message defaults based on predefined values
    // if they weren't passed through a form or loaded from the vacation file
    if (!$subject) {
        $subject = $AUTORESPOND_OPTS['default_subject'];
    }
    if (!$message) {
        $message = $AUTORESPOND_OPTS['default_message'];
    }

    $subj_label = ar_gettext("Subject:");
    $mesg_label = ar_gettext("Message:");
    echo <<<EOform
  <form action="options.php?action=editvacation" method=POST
   onSubmit="window.refresh()">
  <tr valign=top bgcolor="$color[4]">
    <td>$subj_label</td>
    <td><input type=text name=subject size=40 value="$subject"></input></td>
  </tr>
  <tr valign=top bgcolor="$color[4]">
    <td>$mesg_label</td>
    <td><textarea name=message rows=12 cols=75>$message</textarea></td>
  </tr>

  <tr>
    <td colspan=3> 
      <br><input type=submit value="Update"> &nbsp; <input type=reset>
    </td>
  </tr>
  </form>

EOform;
}

?>
